/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.cashu.voucher.app;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.cashu.voucher.app.VoucherService;
import xyz.tcheeric.cashu.voucher.app.dto.StoredVoucher;
import xyz.tcheeric.cashu.voucher.domain.SignedVoucher;

public class VoucherBackupService {
    private static final Logger log = LoggerFactory.getLogger(VoucherBackupService.class);
    private final VoucherService voucherService;

    public VoucherBackupService(@NonNull VoucherService voucherService) {
        if (voucherService == null) {
            throw new NullPointerException("voucherService is marked non-null but is null");
        }
        this.voucherService = voucherService;
        log.info("VoucherBackupService initialized");
    }

    public int backupIfNeeded(@NonNull List<StoredVoucher> vouchers, @NonNull String userPrivateKey) {
        if (vouchers == null) {
            throw new NullPointerException("vouchers is marked non-null but is null");
        }
        if (userPrivateKey == null) {
            throw new NullPointerException("userPrivateKey is marked non-null but is null");
        }
        if (userPrivateKey.isBlank()) {
            throw new IllegalArgumentException("User private key cannot be blank");
        }
        log.debug("Checking {} voucher(s) for backup", (Object)vouchers.size());
        List<StoredVoucher> needsBackup = vouchers.stream().filter(StoredVoucher::needsBackup).collect(Collectors.toList());
        if (needsBackup.isEmpty()) {
            log.debug("No vouchers need backup");
            return 0;
        }
        log.info("Backing up {} of {} voucher(s)", (Object)needsBackup.size(), (Object)vouchers.size());
        List<SignedVoucher> toBackup = needsBackup.stream().map(StoredVoucher::getVoucher).collect(Collectors.toList());
        this.voucherService.backup(toBackup, userPrivateKey);
        long backupTime = Instant.now().getEpochSecond();
        needsBackup.forEach(v -> v.setLastBackupAt(backupTime));
        log.info("Successfully backed up {} voucher(s)", (Object)needsBackup.size());
        return needsBackup.size();
    }

    public void backupAll(@NonNull List<StoredVoucher> vouchers, @NonNull String userPrivateKey) {
        if (vouchers == null) {
            throw new NullPointerException("vouchers is marked non-null but is null");
        }
        if (userPrivateKey == null) {
            throw new NullPointerException("userPrivateKey is marked non-null but is null");
        }
        if (userPrivateKey.isBlank()) {
            throw new IllegalArgumentException("User private key cannot be blank");
        }
        log.info("Backing up all {} voucher(s)", (Object)vouchers.size());
        List<SignedVoucher> toBackup = vouchers.stream().map(StoredVoucher::getVoucher).collect(Collectors.toList());
        this.voucherService.backup(toBackup, userPrivateKey);
        long backupTime = Instant.now().getEpochSecond();
        vouchers.forEach(v -> v.setLastBackupAt(backupTime));
        log.info("Successfully backed up all {} voucher(s)", (Object)vouchers.size());
    }

    public List<StoredVoucher> restoreAndMerge(@NonNull List<StoredVoucher> localVouchers, @NonNull String userPrivateKey) {
        if (localVouchers == null) {
            throw new NullPointerException("localVouchers is marked non-null but is null");
        }
        if (userPrivateKey == null) {
            throw new NullPointerException("userPrivateKey is marked non-null but is null");
        }
        if (userPrivateKey.isBlank()) {
            throw new IllegalArgumentException("User private key cannot be blank");
        }
        log.info("Restoring vouchers and merging with {} local voucher(s)", (Object)localVouchers.size());
        List<SignedVoucher> restored = this.voucherService.restore(userPrivateKey);
        log.info("Restored {} voucher(s) from backup", (Object)restored.size());
        List<StoredVoucher> restoredStored = restored.stream().map(v -> {
            StoredVoucher stored = StoredVoucher.from(v);
            stored.markBackedUp();
            return stored;
        }).collect(Collectors.toList());
        List<StoredVoucher> merged = this.mergeVouchers(localVouchers, restoredStored);
        log.info("Merge complete: {} total voucher(s)", (Object)merged.size());
        return merged;
    }

    public List<StoredVoucher> restore(@NonNull String userPrivateKey) {
        if (userPrivateKey == null) {
            throw new NullPointerException("userPrivateKey is marked non-null but is null");
        }
        if (userPrivateKey.isBlank()) {
            throw new IllegalArgumentException("User private key cannot be blank");
        }
        log.info("Restoring vouchers from backup");
        List<SignedVoucher> restored = this.voucherService.restore(userPrivateKey);
        log.info("Restored {} voucher(s)", (Object)restored.size());
        return restored.stream().map(v -> {
            StoredVoucher stored = StoredVoucher.from(v);
            stored.markBackedUp();
            return stored;
        }).collect(Collectors.toList());
    }

    private List<StoredVoucher> mergeVouchers(List<StoredVoucher> local, List<StoredVoucher> restored) {
        log.debug("Merging {} local and {} restored vouchers", (Object)local.size(), (Object)restored.size());
        HashMap<String, StoredVoucher> localMap = new HashMap<String, StoredVoucher>();
        for (StoredVoucher v : local) {
            localMap.put(v.getVoucherId(), v);
        }
        int addedCount = 0;
        for (StoredVoucher restoredVoucher : restored) {
            String voucherId = restoredVoucher.getVoucherId();
            if (!localMap.containsKey(voucherId)) {
                localMap.put(voucherId, restoredVoucher);
                ++addedCount;
                log.debug("Added restored voucher: voucherId={}", (Object)voucherId);
                continue;
            }
            log.debug("Voucher already exists locally, keeping local version: voucherId={}", (Object)voucherId);
        }
        log.debug("Merge added {} new voucher(s) from backup", (Object)addedCount);
        return new ArrayList<StoredVoucher>(localMap.values());
    }

    public boolean verifyBackup(@NonNull List<String> expectedVoucherIds, @NonNull String userPrivateKey) {
        if (expectedVoucherIds == null) {
            throw new NullPointerException("expectedVoucherIds is marked non-null but is null");
        }
        if (userPrivateKey == null) {
            throw new NullPointerException("userPrivateKey is marked non-null but is null");
        }
        if (userPrivateKey.isBlank()) {
            throw new IllegalArgumentException("User private key cannot be blank");
        }
        log.info("Verifying backup integrity: {} expected voucher(s)", (Object)expectedVoucherIds.size());
        try {
            List<SignedVoucher> restored = this.voucherService.restore(userPrivateKey);
            Map<String, SignedVoucher> restoredMap = restored.stream().collect(Collectors.toMap(v -> v.getSecret().getVoucherId(), v -> v));
            boolean allPresent = true;
            for (String expectedId : expectedVoucherIds) {
                if (restoredMap.containsKey(expectedId)) continue;
                log.warn("Backup verification failed: missing voucher {}", (Object)expectedId);
                allPresent = false;
            }
            if (allPresent) {
                log.info("Backup verification successful: all {} voucher(s) present", (Object)expectedVoucherIds.size());
            } else {
                log.warn("Backup verification failed: some vouchers missing");
            }
            return allPresent;
        }
        catch (Exception e) {
            log.error("Backup verification failed with exception", (Throwable)e);
            return false;
        }
    }
}

