/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.gateway.app.persistence;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import javax.crypto.Cipher;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.crypto.password.PasswordEncoder;
import xyz.tcheeric.gateway.rest.apikey.ApiKey;
import xyz.tcheeric.gateway.rest.apikey.ApiKeyRepository;
import xyz.tcheeric.gateway.security.auth.ApiKeyAuthentication;
import xyz.tcheeric.gateway.security.auth.ApiKeyStore;

public class PostgresApiKeyStore
implements ApiKeyStore {
    private static final Logger LOGGER = LoggerFactory.getLogger(PostgresApiKeyStore.class);
    private static final String ENCRYPTION_ALGO = "AES/GCM/NoPadding";
    private static final int IV_LENGTH = 12;
    private static final int TAG_LENGTH = 128;
    private static final int KEY_BYTES = 32;
    private final ApiKeyRepository repository;
    private final PasswordEncoder passwordEncoder;
    private final SecretKeySpec encryptionKey;
    private final SecureRandom secureRandom = new SecureRandom();

    public PostgresApiKeyStore(ApiKeyRepository repository, PasswordEncoder passwordEncoder, String masterKey) {
        this.repository = repository;
        this.passwordEncoder = passwordEncoder;
        this.encryptionKey = new SecretKeySpec(this.deriveKey(masterKey), "AES");
        LOGGER.info("api_key_store initialized storage=postgresql");
    }

    public Optional<ApiKeyAuthentication> validate(String apiKey) {
        LOGGER.debug("api_key_store validate_attempt");
        List activeKeys = this.repository.findAllActive();
        for (ApiKey key : activeKeys) {
            if (!this.passwordEncoder.matches((CharSequence)apiKey, key.getSecretHash())) continue;
            if (key.isExpired()) {
                LOGGER.debug("api_key_store validate_expired api_key_id={}", (Object)key.getApiKeyId());
                continue;
            }
            this.recordUsage(key.getApiKeyId());
            LOGGER.debug("api_key_store validate_success api_key_id={} client_id={}", (Object)key.getApiKeyId(), (Object)key.getClientId());
            return Optional.of(key.toAuthentication());
        }
        LOGGER.debug("api_key_store validate_failed no_matching_key");
        return Optional.empty();
    }

    public ApiKeyStore.ApiKeyCreationResult create(String clientId, Set<String> permissions, ApiKeyAuthentication.RateLimitConfig rateLimit) {
        return this.create(ApiKeyStore.ApiKeyCreateCommand.basic((String)clientId, permissions, (ApiKeyAuthentication.RateLimitConfig)rateLimit));
    }

    public ApiKeyStore.ApiKeyCreationResult create(ApiKeyStore.ApiKeyCreateCommand command) {
        LOGGER.debug("api_key_store create_attempt client_id={}", (Object)command.clientId());
        String apiKeyId = this.generateApiKeyId();
        String rawKey = this.generateKey();
        String hashed = this.passwordEncoder.encode((CharSequence)rawKey);
        byte[] encrypted = this.encrypt(rawKey);
        ApiKey apiKey = new ApiKey(apiKeyId, command.clientId(), hashed, encrypted, command.permissions(), command.rateLimit());
        apiKey.setName(command.name());
        apiKey.setDescription(command.description());
        apiKey.setExpiresAt(command.expiresAt());
        this.repository.save(apiKey);
        this.repository.recordAuditEvent(apiKeyId, "CREATED", null, command.clientId(), null);
        LOGGER.info("api_key_store create_success api_key_id={} client_id={}", (Object)apiKeyId, (Object)command.clientId());
        return new ApiKeyStore.ApiKeyCreationResult(apiKeyId, rawKey, command.clientId());
    }

    public boolean revoke(String apiKeyId) {
        LOGGER.debug("api_key_store revoke_attempt api_key_id={}", (Object)apiKeyId);
        boolean revoked = this.repository.markRevoked(apiKeyId);
        if (revoked) {
            this.repository.recordAuditEvent(apiKeyId, "REVOKED", null, null, null);
            LOGGER.info("api_key_store revoke_success api_key_id={}", (Object)apiKeyId);
        } else {
            LOGGER.warn("api_key_store revoke_not_found api_key_id={}", (Object)apiKeyId);
        }
        return revoked;
    }

    public List<ApiKeyStore.ApiKeyInfo> listByClientId(String clientId) {
        LOGGER.debug("api_key_store list_by_client_attempt client_id={}", (Object)clientId);
        List keys = this.repository.findByClientId(clientId);
        List<ApiKeyStore.ApiKeyInfo> result = keys.stream().map(arg_0 -> this.toApiKeyInfo(arg_0)).toList();
        LOGGER.debug("api_key_store list_by_client_success client_id={} count={}", (Object)clientId, (Object)result.size());
        return result;
    }

    public Optional<ApiKeyStore.ApiKeyInfo> findById(String apiKeyId, String clientId) {
        LOGGER.debug("api_key_store find_by_id_attempt api_key_id={} client_id={}", (Object)apiKeyId, (Object)clientId);
        return this.repository.findByApiKeyIdAndClientId(apiKeyId, clientId).map(arg_0 -> this.toApiKeyInfo(arg_0));
    }

    public Optional<ApiKeyStore.ApiKeyCreationResult> rotate(String oldKeyId, String clientId, Duration gracePeriod) {
        LOGGER.debug("api_key_store rotate_attempt old_key_id={} client_id={} grace_period={}", new Object[]{oldKeyId, clientId, gracePeriod});
        Optional oldKeyOpt = this.repository.findByApiKeyIdAndClientId(oldKeyId, clientId);
        if (oldKeyOpt.isEmpty()) {
            LOGGER.warn("api_key_store rotate_not_found old_key_id={} client_id={}", (Object)oldKeyId, (Object)clientId);
            return Optional.empty();
        }
        ApiKey oldKey = (ApiKey)oldKeyOpt.get();
        if (oldKey.isRevoked()) {
            LOGGER.warn("api_key_store rotate_already_revoked old_key_id={}", (Object)oldKeyId);
            return Optional.empty();
        }
        String newApiKeyId = this.generateApiKeyId();
        String rawKey = this.generateKey();
        String hashed = this.passwordEncoder.encode((CharSequence)rawKey);
        byte[] encrypted = this.encrypt(rawKey);
        ApiKey newKey = new ApiKey(newApiKeyId, clientId, hashed, encrypted, oldKey.getPermissions(), oldKey.getRateLimit());
        newKey.setName(oldKey.getName());
        newKey.setDescription(oldKey.getDescription());
        newKey.setExpiresAt(oldKey.getExpiresAt());
        newKey.setRotatedFromId(oldKeyId);
        Instant graceUntil = Instant.now().plus(gracePeriod);
        oldKey.setRotationGraceUntil(graceUntil);
        this.repository.save(newKey);
        this.repository.recordAuditEvent(oldKeyId, "ROTATED", "rotated_to=" + newApiKeyId + " grace_until=" + String.valueOf(graceUntil), clientId, null);
        this.repository.recordAuditEvent(newApiKeyId, "CREATED", "rotated_from=" + oldKeyId, clientId, null);
        LOGGER.info("api_key_store rotate_success old_key_id={} new_key_id={} grace_until={}", new Object[]{oldKeyId, newApiKeyId, graceUntil});
        return Optional.of(new ApiKeyStore.ApiKeyCreationResult(newApiKeyId, rawKey, clientId));
    }

    public void recordUsage(String apiKeyId) {
        try {
            this.repository.updateLastUsedAt(apiKeyId);
        }
        catch (Exception e) {
            LOGGER.warn("api_key_store record_usage_failed api_key_id={} error={}", (Object)apiKeyId, (Object)e.getMessage());
        }
    }

    public Optional<String> findSecretByClientId(String clientId) {
        List keys = this.repository.findActiveByClientId(clientId);
        if (keys.isEmpty()) {
            return Optional.empty();
        }
        ApiKey key = (ApiKey)keys.get(0);
        if (key.getEncryptedSecret() == null) {
            return Optional.empty();
        }
        return Optional.of(this.decrypt(key.getEncryptedSecret()));
    }

    private ApiKeyStore.ApiKeyInfo toApiKeyInfo(ApiKey key) {
        return new ApiKeyStore.ApiKeyInfo(key.getApiKeyId(), key.getClientId(), key.getName(), key.getDescription(), key.getPermissions(), key.getRateLimit(), key.getCreatedAt(), key.getExpiresAt(), key.getLastUsedAt(), key.isRevoked(), key.getRevokedAt(), key.getRotatedFromId(), key.getRotationGraceUntil());
    }

    private String generateApiKeyId() {
        return "key_" + Math.abs(this.secureRandom.nextInt());
    }

    private String generateKey() {
        byte[] bytes = new byte[32];
        this.secureRandom.nextBytes(bytes);
        return Base64.getUrlEncoder().withoutPadding().encodeToString(bytes);
    }

    private byte[] deriveKey(String masterKey) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            return digest.digest(masterKey.getBytes(StandardCharsets.UTF_8));
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to derive encryption key", e);
        }
    }

    private byte[] encrypt(String secret) {
        try {
            byte[] iv = new byte[12];
            this.secureRandom.nextBytes(iv);
            Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGO);
            cipher.init(1, (Key)this.encryptionKey, new GCMParameterSpec(128, iv));
            byte[] cipherText = cipher.doFinal(secret.getBytes(StandardCharsets.UTF_8));
            ByteBuffer buffer = ByteBuffer.allocate(iv.length + cipherText.length);
            buffer.put(iv);
            buffer.put(cipherText);
            return buffer.array();
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to encrypt API key", e);
        }
    }

    private String decrypt(byte[] encrypted) {
        try {
            ByteBuffer buffer = ByteBuffer.wrap(encrypted);
            byte[] iv = new byte[12];
            buffer.get(iv);
            byte[] cipherText = new byte[buffer.remaining()];
            buffer.get(cipherText);
            Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGO);
            cipher.init(2, (Key)this.encryptionKey, new GCMParameterSpec(128, iv));
            byte[] plain = cipher.doFinal(cipherText);
            return new String(plain, StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to decrypt API key", e);
        }
    }
}

