-- API Keys table with soft delete support
-- Stores API keys for client authentication with PostgreSQL
CREATE TABLE IF NOT EXISTS api_keys (
    id                      BIGINT GENERATED ALWAYS AS IDENTITY PRIMARY KEY,
    api_key_id              VARCHAR(64) NOT NULL UNIQUE,
    client_id               VARCHAR(128) NOT NULL,

    -- Security: hashed secret for validation, encrypted secret for HMAC
    secret_hash             VARCHAR(256) NOT NULL,
    encrypted_secret        BYTEA,

    -- Metadata
    name                    VARCHAR(128),
    description             VARCHAR(512),

    -- Permissions stored as JSON array
    permissions             TEXT NOT NULL DEFAULT '["*"]',

    -- Rate limiting configuration
    rate_limit_per_minute   INT NOT NULL DEFAULT 60,
    rate_limit_per_hour     INT NOT NULL DEFAULT 1000,

    -- Timestamps
    created_at              TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT CURRENT_TIMESTAMP,
    expires_at              TIMESTAMP WITH TIME ZONE,
    last_used_at            TIMESTAMP WITH TIME ZONE,

    -- Soft delete / revocation
    revoked                 BOOLEAN NOT NULL DEFAULT FALSE,
    revoked_at              TIMESTAMP WITH TIME ZONE,

    -- Rotation support
    rotated_from_id         VARCHAR(64),
    rotation_grace_until    TIMESTAMP WITH TIME ZONE,

    -- Constraints
    CONSTRAINT chk_expires_after_created CHECK (expires_at IS NULL OR expires_at > created_at)
);

-- Index for client lookups (list keys by client)
CREATE INDEX idx_api_keys_client_id ON api_keys(client_id);

-- Index for validation lookups (find by hash)
CREATE INDEX idx_api_keys_secret_hash ON api_keys(secret_hash);

-- Index for expiration checks (only on non-revoked keys)
CREATE INDEX idx_api_keys_expires_at ON api_keys(expires_at) WHERE revoked = FALSE;

-- Index for rotation grace period lookups
CREATE INDEX idx_api_keys_rotation_grace ON api_keys(rotation_grace_until)
    WHERE rotation_grace_until IS NOT NULL;

-- Audit table for key lifecycle events
CREATE TABLE IF NOT EXISTS api_key_audit (
    id              BIGINT GENERATED ALWAYS AS IDENTITY PRIMARY KEY,
    api_key_id      VARCHAR(64) NOT NULL,
    event_type      VARCHAR(32) NOT NULL,
    event_details   TEXT,
    performed_by    VARCHAR(128),
    ip_address      VARCHAR(45),
    created_at      TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT CURRENT_TIMESTAMP
);

-- Index for audit lookups by key
CREATE INDEX idx_api_key_audit_key_id ON api_key_audit(api_key_id);

-- Index for audit time-based queries
CREATE INDEX idx_api_key_audit_created ON api_key_audit(created_at);
