/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.application.usecases.bunker;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.identity.api.exceptions.IdentityException;
import xyz.tcheeric.identity.api.exceptions.IdentityNotFoundException;
import xyz.tcheeric.identity.api.ports.BunkerIdentityRepository;
import xyz.tcheeric.identity.api.ports.SigningPort;
import xyz.tcheeric.identity.domain.BunkerIdentity;

public class DecryptMessageUseCase {
    private static final Logger LOGGER = LoggerFactory.getLogger(DecryptMessageUseCase.class);
    private static final Pattern HEX_PUBKEY_PATTERN = Pattern.compile("^[0-9a-fA-F]{64}$");
    private static final Pattern NPUB_PATTERN = Pattern.compile("^npub1[qpzry9x8gf2tvdw0s3jn54khce6mua7l]{58}$");
    private final SigningPort signingPort;
    private final BunkerIdentityRepository identityRepository;

    public DecryptMessageUseCase(SigningPort signingPort, BunkerIdentityRepository identityRepository) {
        this.signingPort = Objects.requireNonNull(signingPort, "Signing port cannot be null");
        this.identityRepository = Objects.requireNonNull(identityRepository, "Identity repository cannot be null");
    }

    public CompletableFuture<String> decryptNip44(String keyName, String senderPubkey, String ciphertext) {
        return this.decrypt(keyName, senderPubkey, ciphertext, DecryptionType.NIP44);
    }

    public CompletableFuture<String> decryptNip04(String keyName, String senderPubkey, String ciphertext) {
        return this.decrypt(keyName, senderPubkey, ciphertext, DecryptionType.NIP04);
    }

    public CompletableFuture<String> decryptAuto(String keyName, String senderPubkey, String ciphertext) {
        return ((CompletableFuture)this.decryptNip44(keyName, senderPubkey, ciphertext).exceptionally(ex -> {
            LOGGER.debug("nip44_decrypt_failed key_name={} falling_back_to_nip04", (Object)keyName);
            return null;
        })).thenCompose(result -> {
            if (result != null) {
                return CompletableFuture.completedFuture(result);
            }
            return this.decryptNip04(keyName, senderPubkey, ciphertext);
        });
    }

    private CompletableFuture<String> decrypt(String keyName, String senderPubkey, String ciphertext, DecryptionType type) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        Objects.requireNonNull(senderPubkey, "Sender public key cannot be null");
        Objects.requireNonNull(ciphertext, "Ciphertext cannot be null");
        String normalizedPubkey = this.normalizePublicKey(senderPubkey);
        if (normalizedPubkey == null) {
            return CompletableFuture.failedFuture(new IdentityException("Invalid sender public key format. Expected 64-character hex or npub1... format. Suggestion: Verify the public key and try again."));
        }
        LOGGER.debug("decrypt_message_started key_name={} type={} sender={}", new Object[]{keyName, type, normalizedPubkey.substring(0, 16)});
        return ((CompletableFuture)this.identityRepository.findByKeyName(keyName).thenCompose(optIdentity -> {
            BunkerIdentity identity = (BunkerIdentity)optIdentity.orElseThrow(() -> new IdentityNotFoundException("Bunker identity with key name '" + keyName + "' not found. Suggestion: Check the key name or create/import the identity first."));
            if (identity.isLocked()) {
                LOGGER.warn("decrypt_message_rejected key_name={} reason=identity_locked", (Object)keyName);
                return CompletableFuture.failedFuture(new IdentityException("Identity '" + keyName + "' is locked. Suggestion: Unlock the identity with the passphrase before decrypting."));
            }
            return this.performDecryption(identity, normalizedPubkey, ciphertext, type);
        })).whenComplete((plaintext, ex) -> {
            if (ex != null) {
                LOGGER.error("decrypt_message_failed key_name={} type={} error={}", new Object[]{keyName, type, ex.getMessage()});
            } else {
                LOGGER.info("decrypt_message_complete key_name={} type={} plaintext_length={}", new Object[]{keyName, type, plaintext.length()});
            }
        });
    }

    private CompletableFuture<String> performDecryption(BunkerIdentity identity, String senderPubkey, String ciphertext, DecryptionType type) {
        LOGGER.debug("bunker_decrypt identity={} type={}", (Object)identity.getKeyName(), (Object)type);
        CompletableFuture<String> decryptFuture = switch (type.ordinal()) {
            default -> throw new MatchException(null, null);
            case 0 -> this.signingPort.nip04Decrypt(identity, senderPubkey, ciphertext);
            case 1 -> this.signingPort.nip44Decrypt(identity, senderPubkey, ciphertext);
        };
        return decryptFuture.exceptionally(ex -> {
            throw new IdentityException("Failed to decrypt message with identity '" + identity.getKeyName() + "': " + ex.getMessage() + ". Suggestion: Verify the ciphertext format and that the correct identity is being used.", (Throwable)ex);
        });
    }

    private String normalizePublicKey(String pubkey) {
        if (HEX_PUBKEY_PATTERN.matcher(pubkey).matches()) {
            return pubkey.toLowerCase();
        }
        if (NPUB_PATTERN.matcher(pubkey).matches()) {
            return null;
        }
        return null;
    }

    private static enum DecryptionType {
        NIP04,
        NIP44;

    }

    public record DecryptionResult(boolean success, String plaintext, String type, String error) {
        public static DecryptionResult success(String plaintext, String type) {
            return new DecryptionResult(true, plaintext, type, null);
        }

        public static DecryptionResult failure(String error) {
            return new DecryptionResult(false, null, null, error);
        }
    }
}

