/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.application.usecases.bunker;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.identity.api.exceptions.IdentityException;
import xyz.tcheeric.identity.api.ports.BunkerConnectionPort;
import xyz.tcheeric.identity.api.ports.BunkerIdentityRepository;
import xyz.tcheeric.identity.domain.BunkerIdentity;

public class ImportIdentityToBunkerUseCase {
    private static final Logger LOGGER = LoggerFactory.getLogger(ImportIdentityToBunkerUseCase.class);
    private static final Pattern NSEC_PATTERN = Pattern.compile("^nsec1[qpzry9x8gf2tvdw0s3jn54khce6mua7l]{58}$");
    private static final Pattern HEX_PATTERN = Pattern.compile("^[0-9a-fA-F]{64}$");
    private final BunkerConnectionPort connectionPort;
    private final BunkerIdentityRepository identityRepository;

    public ImportIdentityToBunkerUseCase(BunkerConnectionPort connectionPort, BunkerIdentityRepository identityRepository) {
        this.connectionPort = Objects.requireNonNull(connectionPort, "Connection port cannot be null");
        this.identityRepository = Objects.requireNonNull(identityRepository, "Identity repository cannot be null");
    }

    public CompletableFuture<BunkerIdentity> execute(String keyName, String nsec) {
        return this.execute(keyName, nsec, null);
    }

    public CompletableFuture<BunkerIdentity> execute(String keyName, String nsec, String passphrase) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        Objects.requireNonNull(nsec, "Private key (nsec) cannot be null");
        LOGGER.info("import_identity_to_bunker_started key_name={} has_passphrase={}", (Object)keyName, (Object)(passphrase != null ? 1 : 0));
        if (!this.isValidKeyFormat(nsec)) {
            LOGGER.warn("import_identity_rejected key_name={} reason=invalid_key_format", (Object)keyName);
            return CompletableFuture.failedFuture(new IdentityException("Invalid private key format. Expected nsec1... (bech32) or 64-character hex string. Suggestion: Verify the key format and try again."));
        }
        return ((CompletableFuture)((CompletableFuture)this.identityRepository.existsByKeyName(keyName).thenCompose(exists -> {
            if (exists.booleanValue()) {
                LOGGER.warn("import_identity_rejected key_name={} reason=already_exists", (Object)keyName);
                return CompletableFuture.failedFuture(new IdentityException("An identity with key name '" + keyName + "' already exists. Suggestion: Use a different key name or delete the existing identity first."));
            }
            return this.importKeyToBunker(keyName, nsec, passphrase);
        })).thenCompose(identity -> this.persistIdentity((BunkerIdentity)identity))).whenComplete((identity, ex) -> {
            if (ex != null) {
                LOGGER.error("import_identity_to_bunker_failed key_name={} error={}", (Object)keyName, (Object)ex.getMessage());
            } else {
                LOGGER.info("import_identity_to_bunker_complete key_name={} id={} pubkey={}", identity.getKeyName(), identity.getId(), identity.getPublicKey().toHex().substring(0, 16));
            }
        });
    }

    private boolean isValidKeyFormat(String key) {
        return NSEC_PATTERN.matcher(key).matches() || HEX_PATTERN.matcher(key).matches();
    }

    private CompletableFuture<BunkerIdentity> importKeyToBunker(String keyName, String nsec, String passphrase) {
        LOGGER.debug("bunker_import_key key_name={}", (Object)keyName);
        return this.connectionPort.importKey(keyName, nsec, passphrase).exceptionally(ex -> {
            throw new IdentityException("Failed to import key '" + keyName + "' to bunker: " + ex.getMessage() + ". Suggestion: Check bunker connectivity and verify the key format.", (Throwable)ex);
        });
    }

    private CompletableFuture<BunkerIdentity> persistIdentity(BunkerIdentity identity) {
        LOGGER.debug("persist_identity id={} key_name={}", (Object)identity.getId(), (Object)identity.getKeyName());
        return ((CompletableFuture)this.identityRepository.save(identity).thenApply(v -> identity)).exceptionally(ex -> {
            LOGGER.warn("persist_identity_failed id={} error={} action=rollback", (Object)identity.getId(), (Object)ex.getMessage());
            this.connectionPort.deleteKey(identity.getKeyName()).exceptionally(deleteEx -> {
                LOGGER.error("rollback_delete_key_failed key_name={} error={}", (Object)identity.getKeyName(), (Object)deleteEx.getMessage());
                return false;
            });
            throw new IdentityException("Failed to persist identity locally: " + ex.getMessage() + ". Suggestion: Check local storage and retry.", (Throwable)ex);
        });
    }

    public record ImportIdentityResult(boolean success, BunkerIdentity identity, String errorMessage) {
        public static ImportIdentityResult success(BunkerIdentity identity) {
            Objects.requireNonNull(identity, "Identity cannot be null for success result");
            return new ImportIdentityResult(true, identity, null);
        }

        public static ImportIdentityResult failure(String errorMessage) {
            Objects.requireNonNull(errorMessage, "Error message cannot be null for failure result");
            return new ImportIdentityResult(false, null, errorMessage);
        }
    }
}

