/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.infrastructure.bunker;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import nostr.crypto.bech32.Bech32;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.identity.api.ports.BunkerConnectionPort;
import xyz.tcheeric.identity.domain.BunkerIdentity;
import xyz.tcheeric.identity.domain.PublicKey;
import xyz.tcheeric.nsecbunker.admin.AdminException;
import xyz.tcheeric.nsecbunker.admin.NsecBunkerAdminClient;
import xyz.tcheeric.nsecbunker.core.exception.BunkerConnectionException;
import xyz.tcheeric.nsecbunker.core.exception.BunkerException;
import xyz.tcheeric.nsecbunker.core.model.BunkerKey;

public class BunkerConnectionManager
implements BunkerConnectionPort,
AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(BunkerConnectionManager.class);
    private final String bunkerPubkey;
    private final String adminPrivateKey;
    private final List<String> relays;
    private final AtomicReference<BunkerConnectionPort.ConnectionState> state = new AtomicReference<BunkerConnectionPort.ConnectionState>(BunkerConnectionPort.ConnectionState.DISCONNECTED);
    private final AtomicReference<NsecBunkerAdminClient> clientRef = new AtomicReference();
    private Duration connectTimeout = Duration.ofSeconds(30L);
    private Duration requestTimeout = Duration.ofSeconds(60L);
    private volatile long lastPingTime = 0L;

    public BunkerConnectionManager(String bunkerPubkey, String adminPrivateKey, List<String> relays) {
        this.bunkerPubkey = Objects.requireNonNull(bunkerPubkey, "Bunker pubkey cannot be null");
        this.adminPrivateKey = Objects.requireNonNull(adminPrivateKey, "Admin private key cannot be null");
        this.relays = List.copyOf((Collection)Objects.requireNonNull(relays, "Relays cannot be null"));
        if (relays.isEmpty()) {
            throw new IllegalArgumentException("At least one relay URL is required");
        }
    }

    @Override
    public CompletableFuture<Void> connect() {
        return CompletableFuture.runAsync(() -> {
            if (this.state.get() == BunkerConnectionPort.ConnectionState.CONNECTED) {
                LOGGER.debug("bunker_connection already_connected bunker_pubkey={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey));
                return;
            }
            this.state.set(BunkerConnectionPort.ConnectionState.CONNECTING);
            LOGGER.info("bunker_connection connecting bunker_pubkey={} relays={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey), (Object)this.relays.size());
            try {
                NsecBunkerAdminClient client = NsecBunkerAdminClient.builder().bunkerPubkey(this.bunkerPubkey).adminPrivateKey(this.adminPrivateKey).relays(this.relays).connectTimeout(this.connectTimeout).requestTimeout(this.requestTimeout).autoReconnect(true).useEphemeralKey(false).build();
                client.connect();
                this.clientRef.set(client);
                this.state.set(BunkerConnectionPort.ConnectionState.CONNECTED);
                this.lastPingTime = System.currentTimeMillis();
                LOGGER.info("bunker_connection connected bunker_pubkey={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey));
            }
            catch (BunkerConnectionException e) {
                this.state.set(BunkerConnectionPort.ConnectionState.FAILED);
                LOGGER.error("bunker_connection failed bunker_pubkey={} error={}", BunkerConnectionManager.truncatePubkey(this.bunkerPubkey), e.getMessage(), e);
                throw new RuntimeException("Failed to connect to bunker: " + e.getMessage(), e);
            }
        });
    }

    @Override
    public CompletableFuture<Void> disconnect() {
        return CompletableFuture.runAsync(() -> {
            NsecBunkerAdminClient client = this.clientRef.get();
            if (client == null) {
                this.state.set(BunkerConnectionPort.ConnectionState.DISCONNECTED);
                return;
            }
            this.state.set(BunkerConnectionPort.ConnectionState.DISCONNECTING);
            LOGGER.info("bunker_connection disconnecting bunker_pubkey={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey));
            try {
                client.disconnect();
                client.close();
            }
            catch (Exception e) {
                LOGGER.warn("bunker_connection disconnect_error bunker_pubkey={} error={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey), (Object)e.getMessage());
            }
            finally {
                this.clientRef.set(null);
                this.state.set(BunkerConnectionPort.ConnectionState.DISCONNECTED);
                LOGGER.info("bunker_connection disconnected bunker_pubkey={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey));
            }
        });
    }

    @Override
    public boolean isConnected() {
        return this.state.get() == BunkerConnectionPort.ConnectionState.CONNECTED && this.clientRef.get() != null;
    }

    @Override
    public BunkerConnectionPort.ConnectionState getConnectionState() {
        return this.state.get();
    }

    @Override
    public CompletableFuture<String> ping() {
        return ((CompletableFuture)this.ensureConnected().thenCompose(client -> client.ping())).whenComplete((pong, error) -> {
            if (error == null) {
                this.lastPingTime = System.currentTimeMillis();
                LOGGER.debug("bunker_ping success bunker_pubkey={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey));
            } else {
                LOGGER.warn("bunker_ping failed bunker_pubkey={} error={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey), (Object)error.getMessage());
            }
        });
    }

    @Override
    public CompletableFuture<BunkerConnectionPort.HealthStatus> healthCheck() {
        if (!this.isConnected()) {
            return CompletableFuture.completedFuture(BunkerConnectionPort.HealthStatus.disconnected());
        }
        long startTime = System.currentTimeMillis();
        return ((CompletableFuture)((CompletableFuture)this.ping().thenCompose(pong -> this.listKeys())).thenApply(keys2 -> {
            long latency = System.currentTimeMillis() - startTime;
            return BunkerConnectionPort.HealthStatus.healthy(latency, keys2.size());
        })).exceptionally(error -> {
            LOGGER.warn("bunker_health_check failed bunker_pubkey={} error={}", (Object)BunkerConnectionManager.truncatePubkey(this.bunkerPubkey), (Object)error.getMessage());
            return BunkerConnectionPort.HealthStatus.unhealthy(error.getMessage());
        });
    }

    @Override
    public CompletableFuture<BunkerIdentity> createKey(String keyName, String passphrase) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        LOGGER.info("bunker_key creating key_name={}", (Object)keyName);
        return ((CompletableFuture)((CompletableFuture)this.ensureConnected().thenCompose(client -> client.keyManager().createKey(keyName, passphrase))).thenApply(this::toBunkerIdentity)).whenComplete((identity, error) -> {
            if (error == null) {
                LOGGER.info("bunker_key created key_name={} npub={}", (Object)keyName, (Object)BunkerConnectionManager.truncatePubkey(identity.getPublicKey().toHex()));
            } else {
                LOGGER.error("bunker_key create_failed key_name={} error={}", keyName, error.getMessage(), error);
            }
        });
    }

    @Override
    public CompletableFuture<BunkerIdentity> importKey(String keyName, String nsec, String passphrase) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        Objects.requireNonNull(nsec, "Nsec cannot be null");
        LOGGER.info("bunker_key importing key_name={}", (Object)keyName);
        return ((CompletableFuture)((CompletableFuture)this.ensureConnected().thenCompose(client -> client.keyManager().createKey(keyName, nsec, passphrase))).thenApply(this::toBunkerIdentity)).whenComplete((identity, error) -> {
            if (error == null) {
                LOGGER.info("bunker_key imported key_name={} npub={}", (Object)keyName, (Object)BunkerConnectionManager.truncatePubkey(identity.getPublicKey().toHex()));
            } else {
                LOGGER.error("bunker_key import_failed key_name={} error={}", keyName, error.getMessage(), error);
            }
        });
    }

    @Override
    public CompletableFuture<List<BunkerIdentity>> listKeys() {
        return ((CompletableFuture)((CompletableFuture)this.ensureConnected().thenCompose(client -> client.keyManager().listKeys())).thenApply(keys2 -> keys2.stream().map(this::toBunkerIdentity).collect(Collectors.toList()))).whenComplete((keys2, error) -> {
            if (error == null) {
                LOGGER.debug("bunker_keys listed count={}", (Object)keys2.size());
            } else {
                LOGGER.error("bunker_keys list_failed error={}", (Object)error.getMessage(), error);
            }
        });
    }

    @Override
    public CompletableFuture<Optional<BunkerIdentity>> getKey(String keyName) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        return ((CompletableFuture)((CompletableFuture)this.ensureConnected().thenCompose(client -> client.keyManager().getKeyDetails(keyName))).thenApply(key -> Optional.of(this.toBunkerIdentity((BunkerKey)key)))).exceptionally(error -> {
            if (this.isKeyNotFoundError((Throwable)error)) {
                LOGGER.debug("bunker_key not_found key_name={}", (Object)keyName);
                return Optional.empty();
            }
            throw new RuntimeException((Throwable)error);
        });
    }

    private boolean isKeyNotFoundError(Throwable error) {
        for (Throwable current = error; current != null; current = current.getCause()) {
            if (current instanceof BunkerException || current instanceof AdminException) {
                return true;
            }
            String message = current.getMessage();
            if (message == null || !message.toLowerCase().contains("not found")) continue;
            return true;
        }
        return false;
    }

    @Override
    public CompletableFuture<Optional<BunkerIdentity>> findKeyByPublicKey(PublicKey publicKey) {
        Objects.requireNonNull(publicKey, "Public key cannot be null");
        String pubkeyHex = publicKey.toHex();
        return this.listKeys().thenApply(keys2 -> keys2.stream().filter(key -> key.getPublicKey().toHex().equalsIgnoreCase(pubkeyHex)).findFirst());
    }

    @Override
    public CompletableFuture<Boolean> unlockKey(String keyName, String passphrase) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        Objects.requireNonNull(passphrase, "Passphrase cannot be null");
        LOGGER.info("bunker_key unlocking key_name={}", (Object)keyName);
        return ((CompletableFuture)this.ensureConnected().thenCompose(client -> client.keyManager().unlockKey(keyName, passphrase))).whenComplete((success, error) -> {
            if (error == null && success.booleanValue()) {
                LOGGER.info("bunker_key unlocked key_name={}", (Object)keyName);
            } else if (error != null) {
                LOGGER.error("bunker_key unlock_failed key_name={} error={}", keyName, error.getMessage(), error);
            }
        });
    }

    @Override
    public CompletableFuture<Boolean> deleteKey(String keyName) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        LOGGER.info("bunker_key deleting key_name={}", (Object)keyName);
        return ((CompletableFuture)this.ensureConnected().thenCompose(client -> client.keyManager().deleteKey(keyName))).whenComplete((success, error) -> {
            if (error == null && success.booleanValue()) {
                LOGGER.info("bunker_key deleted key_name={}", (Object)keyName);
            } else if (error != null) {
                LOGGER.error("bunker_key delete_failed key_name={} error={}", keyName, error.getMessage(), error);
            }
        });
    }

    @Override
    public String getBunkerPubkey() {
        return this.bunkerPubkey;
    }

    @Override
    public List<String> getRelays() {
        return this.relays;
    }

    @Override
    public void setConnectionTimeout(Duration timeout2) {
        this.connectTimeout = Objects.requireNonNull(timeout2, "Timeout cannot be null");
    }

    @Override
    public void setRequestTimeout(Duration timeout2) {
        this.requestTimeout = Objects.requireNonNull(timeout2, "Timeout cannot be null");
    }

    @Override
    public void close() {
        this.disconnect().join();
    }

    public Optional<NsecBunkerAdminClient> getAdminClient() {
        if (!this.isConnected()) {
            return Optional.empty();
        }
        return Optional.ofNullable(this.clientRef.get());
    }

    private CompletableFuture<NsecBunkerAdminClient> ensureConnected() {
        NsecBunkerAdminClient client = this.clientRef.get();
        if (client != null && this.isConnected()) {
            return CompletableFuture.completedFuture(client);
        }
        return this.connect().thenApply(v -> {
            NsecBunkerAdminClient newClient = this.clientRef.get();
            if (newClient == null) {
                throw new IllegalStateException("Client not available after connect");
            }
            return newClient;
        });
    }

    private BunkerIdentity toBunkerIdentity(BunkerKey key) {
        PublicKey publicKey = this.resolvePublicKey(key);
        return new BunkerIdentity(key.getName(), publicKey, this.relays, key.isLocked(), key.getDescription());
    }

    private PublicKey resolvePublicKey(BunkerKey key) {
        if (key.getPubkeyHex() != null && !key.getPubkeyHex().isEmpty()) {
            return PublicKey.fromHex(key.getPubkeyHex());
        }
        String npub = key.getNpub();
        if (npub != null && npub.startsWith("npub")) {
            try {
                String hex = Bech32.fromBech32(npub);
                return PublicKey.fromHex(hex);
            }
            catch (Exception e) {
                throw new IllegalStateException("Failed to decode npub: " + npub, e);
            }
        }
        throw new IllegalStateException("BunkerKey has no valid public key: name=" + key.getName());
    }

    private static String truncatePubkey(String pubkey) {
        if (pubkey == null || pubkey.length() <= 16) {
            return pubkey;
        }
        return pubkey.substring(0, 8) + "..." + pubkey.substring(pubkey.length() - 8);
    }
}

