/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.infrastructure.bunker;

import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.identity.api.ports.BunkerConnectionPort;
import xyz.tcheeric.identity.api.ports.BunkerIdentityRepository;
import xyz.tcheeric.identity.domain.BunkerIdentity;
import xyz.tcheeric.identity.domain.PublicKey;
import xyz.tcheeric.identity.infrastructure.cache.IdentityCache;

public class BunkerIdentityRepositoryAdapter
implements BunkerIdentityRepository,
AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(BunkerIdentityRepositoryAdapter.class);
    private static final int DEFAULT_CACHE_SIZE = 100;
    private static final Duration DEFAULT_CACHE_TTL = Duration.ofMinutes(5L);
    private final BunkerConnectionPort connectionPort;
    private final IdentityCache cache;

    public BunkerIdentityRepositoryAdapter(BunkerConnectionPort connectionPort) {
        this(connectionPort, 100, DEFAULT_CACHE_TTL);
    }

    public BunkerIdentityRepositoryAdapter(BunkerConnectionPort connectionPort, int cacheSize, Duration cacheTtl) {
        this.connectionPort = Objects.requireNonNull(connectionPort, "Connection port cannot be null");
        this.cache = new IdentityCache.Builder().maxSize(cacheSize).ttl(cacheTtl).build();
    }

    public BunkerIdentityRepositoryAdapter(BunkerConnectionPort connectionPort, IdentityCache cache) {
        this.connectionPort = Objects.requireNonNull(connectionPort, "Connection port cannot be null");
        this.cache = Objects.requireNonNull(cache, "Cache cannot be null");
    }

    @Override
    public CompletableFuture<Optional<BunkerIdentity>> findById(String id) {
        Objects.requireNonNull(id, "Identity ID cannot be null");
        Optional<BunkerIdentity> cached = this.cache.getById(id);
        if (cached.isPresent()) {
            LOGGER.debug("bunker_identity_found source=cache id={}", (Object)id);
            return CompletableFuture.completedFuture(cached);
        }
        LOGGER.debug("bunker_identity_not_in_cache id={} action=sync_and_retry", (Object)id);
        return this.syncWithBunker().thenApply(count -> this.cache.getById(id));
    }

    @Override
    public CompletableFuture<Optional<BunkerIdentity>> findByKeyName(String keyName) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        Optional<BunkerIdentity> cached = this.cache.getByKeyName(keyName);
        if (cached.isPresent()) {
            LOGGER.debug("bunker_identity_found source=cache key_name={}", (Object)keyName);
            return CompletableFuture.completedFuture(cached);
        }
        LOGGER.debug("bunker_identity_lookup key_name={} source=bunker", (Object)keyName);
        return this.connectionPort.getKey(keyName).thenApply(optIdentity -> {
            optIdentity.ifPresent(this.cache::put);
            return optIdentity;
        });
    }

    @Override
    public CompletableFuture<Optional<BunkerIdentity>> findByPublicKey(PublicKey publicKey) {
        Objects.requireNonNull(publicKey, "Public key cannot be null");
        Optional<BunkerIdentity> cached = this.cache.getByPublicKey(publicKey);
        if (cached.isPresent()) {
            LOGGER.debug("bunker_identity_found source=cache public_key={}", (Object)publicKey.toHex().substring(0, 16));
            return CompletableFuture.completedFuture(cached);
        }
        LOGGER.debug("bunker_identity_lookup public_key={} source=bunker", (Object)publicKey.toHex().substring(0, 16));
        return this.connectionPort.findKeyByPublicKey(publicKey).thenApply(optIdentity -> {
            optIdentity.ifPresent(this.cache::put);
            return optIdentity;
        });
    }

    @Override
    public CompletableFuture<List<BunkerIdentity>> findAll() {
        LOGGER.debug("bunker_identity_list_all source=bunker");
        return this.connectionPort.listKeys().thenApply(identities -> {
            identities.forEach(this.cache::put);
            LOGGER.info("bunker_identity_list_complete count={}", (Object)identities.size());
            return identities;
        });
    }

    @Override
    public CompletableFuture<Void> save(BunkerIdentity identity) {
        Objects.requireNonNull(identity, "Identity cannot be null");
        this.cache.put(identity);
        LOGGER.debug("bunker_identity_saved id={} key_name={}", (Object)identity.getId(), (Object)identity.getKeyName());
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> delete(String id) {
        Objects.requireNonNull(id, "Identity ID cannot be null");
        boolean removed = this.cache.remove(id);
        if (removed) {
            LOGGER.info("bunker_identity_deleted id={}", (Object)id);
        } else {
            LOGGER.debug("bunker_identity_delete_skipped id={} reason=not_found", (Object)id);
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> deleteByKeyName(String keyName) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        boolean removed = this.cache.removeByKeyName(keyName);
        if (removed) {
            LOGGER.info("bunker_identity_deleted key_name={}", (Object)keyName);
        } else {
            LOGGER.debug("bunker_identity_delete_skipped key_name={} reason=not_found", (Object)keyName);
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Boolean> exists(String id) {
        Objects.requireNonNull(id, "Identity ID cannot be null");
        if (this.cache.containsId(id)) {
            return CompletableFuture.completedFuture(true);
        }
        return this.syncWithBunker().thenApply(count -> this.cache.containsId(id));
    }

    @Override
    public CompletableFuture<Boolean> existsByKeyName(String keyName) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        if (this.cache.containsKeyName(keyName)) {
            return CompletableFuture.completedFuture(true);
        }
        return this.connectionPort.getKey(keyName).thenApply(optIdentity -> {
            optIdentity.ifPresent(this.cache::put);
            return optIdentity.isPresent();
        });
    }

    @Override
    public CompletableFuture<Long> count() {
        return this.findAll().thenApply(list -> list.size());
    }

    @Override
    public CompletableFuture<Integer> syncWithBunker() {
        LOGGER.debug("bunker_identity_sync_started");
        return ((CompletableFuture)this.connectionPort.listKeys().thenApply(identities -> {
            this.cache.clear();
            identities.forEach(this.cache::put);
            LOGGER.info("bunker_identity_sync_complete count={}", (Object)identities.size());
            return identities.size();
        })).exceptionally(ex -> {
            LOGGER.error("bunker_identity_sync_failed error={}", (Object)ex.getMessage(), ex);
            return 0;
        });
    }

    public void clearCache() {
        this.cache.clear();
        LOGGER.debug("bunker_identity_cache_cleared");
    }

    public int getCacheSize() {
        return this.cache.size();
    }

    public boolean isCached(String id) {
        return this.cache.containsId(id);
    }

    public IdentityCache.CacheStats getCacheStats() {
        return this.cache.getStats();
    }

    public IdentityCache getCache() {
        return this.cache;
    }

    @Override
    public void close() {
        this.cache.close();
        LOGGER.info("bunker_identity_repository_adapter_closed");
    }
}

