/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.infrastructure.bunker;

import java.time.Duration;
import java.time.Instant;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.identity.domain.BunkerIdentity;
import xyz.tcheeric.nsecbunker.client.signer.NsecBunkerSigner;
import xyz.tcheeric.nsecbunker.client.signer.SignerConfig;
import xyz.tcheeric.nsecbunker.client.transport.RelayNip46Transport;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Decoder;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Encoder;

public class SignerPool
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SignerPool.class);
    private final String clientPrivateKey;
    private final int maxSize;
    private final Duration idleTimeout;
    private final Duration connectTimeout;
    private final Duration requestTimeout;
    private final Map<String, PooledSigner> signers;
    private final Map<String, RelayNip46Transport> transports;
    private final ScheduledExecutorService evictionExecutor;
    private final AtomicLong acquireCount = new AtomicLong(0L);
    private final AtomicLong createCount = new AtomicLong(0L);
    private final AtomicLong evictionCount = new AtomicLong(0L);
    private final AtomicLong reconnectCount = new AtomicLong(0L);
    private volatile boolean closed = false;

    private SignerPool(Builder builder) {
        this.clientPrivateKey = Objects.requireNonNull(builder.clientPrivateKey, "Client private key cannot be null");
        this.maxSize = builder.maxSize;
        this.idleTimeout = builder.idleTimeout;
        this.connectTimeout = builder.connectTimeout;
        this.requestTimeout = builder.requestTimeout;
        this.signers = new ConcurrentHashMap<String, PooledSigner>();
        this.transports = new ConcurrentHashMap<String, RelayNip46Transport>();
        this.evictionExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "signer-pool-eviction");
            t.setDaemon(true);
            return t;
        });
        long evictionPeriodMs = Math.max(this.idleTimeout.toMillis() / 2L, 10000L);
        this.evictionExecutor.scheduleAtFixedRate(this::evictIdleSigners, evictionPeriodMs, evictionPeriodMs, TimeUnit.MILLISECONDS);
        LOGGER.info("signer_pool_initialized max_size={} idle_timeout_ms={}", (Object)this.maxSize, (Object)this.idleTimeout.toMillis());
    }

    public CompletableFuture<NsecBunkerSigner> acquire(BunkerIdentity identity) {
        Objects.requireNonNull(identity, "Identity cannot be null");
        if (this.closed) {
            return CompletableFuture.failedFuture(new IllegalStateException("Signer pool is closed"));
        }
        this.acquireCount.incrementAndGet();
        String key = identity.getId();
        PooledSigner pooledSigner = this.signers.get(key);
        if (pooledSigner != null) {
            if (pooledSigner.isConnected()) {
                pooledSigner.markUsed();
                LOGGER.debug("signer_acquired source=cache identity={}", (Object)identity.getKeyName());
                return CompletableFuture.completedFuture(pooledSigner.getSigner());
            }
            LOGGER.debug("signer_reconnecting identity={}", (Object)identity.getKeyName());
            return this.reconnectSigner(pooledSigner, identity);
        }
        if (this.signers.size() >= this.maxSize) {
            this.evictOldest();
        }
        return this.createSigner(identity).thenApply(signer -> {
            PooledSigner newPooled = new PooledSigner((NsecBunkerSigner)signer, identity);
            this.signers.put(key, newPooled);
            this.createCount.incrementAndGet();
            LOGGER.info("signer_created identity={} pool_size={}", (Object)identity.getKeyName(), (Object)this.signers.size());
            return signer;
        });
    }

    public void release(BunkerIdentity identity) {
        Objects.requireNonNull(identity, "Identity cannot be null");
        PooledSigner pooledSigner = this.signers.get(identity.getId());
        if (pooledSigner != null) {
            pooledSigner.markUsed();
            LOGGER.debug("signer_released identity={}", (Object)identity.getKeyName());
        }
    }

    public void remove(BunkerIdentity identity) {
        Objects.requireNonNull(identity, "Identity cannot be null");
        PooledSigner pooledSigner = this.signers.remove(identity.getId());
        if (pooledSigner != null) {
            this.closeSigner(pooledSigner);
            LOGGER.info("signer_removed identity={} pool_size={}", (Object)identity.getKeyName(), (Object)this.signers.size());
        }
    }

    public boolean contains(BunkerIdentity identity) {
        return this.signers.containsKey(identity.getId());
    }

    public int size() {
        return this.signers.size();
    }

    public PoolStats getStats() {
        int connectedCount = 0;
        for (PooledSigner pooled : this.signers.values()) {
            if (!pooled.isConnected()) continue;
            ++connectedCount;
        }
        return new PoolStats(this.acquireCount.get(), this.createCount.get(), this.evictionCount.get(), this.reconnectCount.get(), this.signers.size(), connectedCount);
    }

    private CompletableFuture<NsecBunkerSigner> createSigner(BunkerIdentity identity) {
        return CompletableFuture.supplyAsync(() -> {
            RelayNip46Transport transport = RelayNip46Transport.builder().relays(identity.getRelays()).clientPrivateKey(this.clientPrivateKey).bunkerPubkey(identity.getPublicKey().toHex()).timeout(this.requestTimeout).connectTimeout(this.connectTimeout).build();
            try {
                transport.connect();
                LOGGER.debug("signer_transport_connected identity={} relays={}", (Object)identity.getKeyName(), (Object)transport.getConnectedRelayCount());
                SignerConfig config = SignerConfig.builder().bunkerPubkey(identity.getPublicKey().toHex()).clientPrivateKey(this.clientPrivateKey).relays(identity.getRelays()).connectTimeout(this.connectTimeout).requestTimeout(this.requestTimeout).build();
                NsecBunkerSigner signer = new NsecBunkerSigner(config, transport.createRequestHandler(), new Nip46Encoder(), new Nip46Decoder());
                this.transports.put(identity.getId(), transport);
                signer.connect().get(this.connectTimeout.toMillis(), TimeUnit.MILLISECONDS);
                LOGGER.debug("signer_connected identity={}", (Object)identity.getKeyName());
                return signer;
            }
            catch (Exception ex) {
                LOGGER.error("signer_connection_failed identity={} error={}", (Object)identity.getKeyName(), (Object)ex.getMessage());
                try {
                    transport.close();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                throw new RuntimeException("Failed to create signer: " + ex.getMessage(), ex);
            }
        });
    }

    private CompletableFuture<NsecBunkerSigner> reconnectSigner(PooledSigner pooledSigner, BunkerIdentity identity) {
        this.closeSigner(pooledSigner);
        this.signers.remove(identity.getId());
        this.reconnectCount.incrementAndGet();
        return this.createSigner(identity).thenApply(signer -> {
            PooledSigner newPooled = new PooledSigner((NsecBunkerSigner)signer, identity);
            this.signers.put(identity.getId(), newPooled);
            LOGGER.info("signer_reconnected identity={}", (Object)identity.getKeyName());
            return signer;
        });
    }

    private void evictIdleSigners() {
        if (this.closed) {
            return;
        }
        Instant cutoff = Instant.now().minus(this.idleTimeout);
        int evictedCount = 0;
        Iterator<Map.Entry<String, PooledSigner>> it = this.signers.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, PooledSigner> entry = it.next();
            PooledSigner pooled = entry.getValue();
            if (!pooled.getLastUsed().isBefore(cutoff)) continue;
            this.closeSigner(pooled);
            it.remove();
            ++evictedCount;
            this.evictionCount.incrementAndGet();
        }
        if (evictedCount > 0) {
            LOGGER.info("signer_pool_eviction evicted={} remaining={}", (Object)evictedCount, (Object)this.signers.size());
        }
    }

    private void evictOldest() {
        PooledSigner pooled;
        String oldestKey = null;
        Instant oldestTime = Instant.MAX;
        for (Map.Entry<String, PooledSigner> entry : this.signers.entrySet()) {
            if (!entry.getValue().getLastUsed().isBefore(oldestTime)) continue;
            oldestTime = entry.getValue().getLastUsed();
            oldestKey = entry.getKey();
        }
        if (oldestKey != null && (pooled = this.signers.remove(oldestKey)) != null) {
            this.closeSigner(pooled);
            this.evictionCount.incrementAndGet();
            LOGGER.debug("signer_evicted_oldest identity={}", (Object)pooled.getIdentity().getKeyName());
        }
    }

    private void closeSigner(PooledSigner pooled) {
        String identityId = pooled.getIdentity().getId();
        try {
            pooled.getSigner().close();
        }
        catch (Exception e) {
            LOGGER.warn("signer_close_failed identity={} error={}", (Object)pooled.getIdentity().getKeyName(), (Object)e.getMessage());
        }
        RelayNip46Transport transport = this.transports.remove(identityId);
        if (transport != null) {
            try {
                transport.close();
            }
            catch (Exception e) {
                LOGGER.warn("transport_close_failed identity={} error={}", (Object)pooled.getIdentity().getKeyName(), (Object)e.getMessage());
            }
        }
    }

    @Override
    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        LOGGER.info("signer_pool_closing size={}", (Object)this.signers.size());
        this.evictionExecutor.shutdown();
        try {
            if (!this.evictionExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.evictionExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.evictionExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
        for (PooledSigner pooled : this.signers.values()) {
            this.closeSigner(pooled);
        }
        this.signers.clear();
        this.transports.clear();
        LOGGER.info("signer_pool_closed");
    }

    public static class Builder {
        private String clientPrivateKey;
        private int maxSize = 10;
        private Duration idleTimeout = Duration.ofMinutes(5L);
        private Duration connectTimeout = Duration.ofSeconds(30L);
        private Duration requestTimeout = Duration.ofSeconds(60L);

        public Builder clientPrivateKey(String clientPrivateKey) {
            this.clientPrivateKey = clientPrivateKey;
            return this;
        }

        public Builder maxSize(int maxSize) {
            if (maxSize <= 0) {
                throw new IllegalArgumentException("Max size must be positive: " + maxSize);
            }
            this.maxSize = maxSize;
            return this;
        }

        public Builder idleTimeout(Duration idleTimeout) {
            this.idleTimeout = Objects.requireNonNull(idleTimeout, "Idle timeout cannot be null");
            return this;
        }

        public Builder connectTimeout(Duration connectTimeout) {
            this.connectTimeout = Objects.requireNonNull(connectTimeout, "Connect timeout cannot be null");
            return this;
        }

        public Builder requestTimeout(Duration requestTimeout) {
            this.requestTimeout = Objects.requireNonNull(requestTimeout, "Request timeout cannot be null");
            return this;
        }

        public SignerPool build() {
            return new SignerPool(this);
        }
    }

    private static class PooledSigner {
        private final NsecBunkerSigner signer;
        private final BunkerIdentity identity;
        private final Instant createdAt;
        private volatile Instant lastUsed;

        PooledSigner(NsecBunkerSigner signer, BunkerIdentity identity) {
            this.signer = signer;
            this.identity = identity;
            this.lastUsed = this.createdAt = Instant.now();
        }

        NsecBunkerSigner getSigner() {
            return this.signer;
        }

        BunkerIdentity getIdentity() {
            return this.identity;
        }

        Instant getLastUsed() {
            return this.lastUsed;
        }

        void markUsed() {
            this.lastUsed = Instant.now();
        }

        boolean isConnected() {
            return this.signer.isConnected();
        }
    }

    public record PoolStats(long acquireCount, long createCount, long evictionCount, long reconnectCount, int poolSize, int connectedCount) {
        @Override
        public String toString() {
            return String.format("PoolStats{acquires=%d, creates=%d, evictions=%d, reconnects=%d, size=%d, connected=%d}", this.acquireCount, this.createCount, this.evictionCount, this.reconnectCount, this.poolSize, this.connectedCount);
        }
    }
}

