/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.infrastructure.cache;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.identity.domain.BunkerIdentity;
import xyz.tcheeric.identity.domain.PublicKey;

public class IdentityCache
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(IdentityCache.class);
    private final Map<String, CacheEntry> entriesById;
    private final Map<String, String> keyNameToId;
    private final Map<String, String> publicKeyToId;
    private final int maxSize;
    private final Duration ttl;
    private final ScheduledExecutorService cleanupExecutor;
    private final AtomicLong hits = new AtomicLong(0L);
    private final AtomicLong misses = new AtomicLong(0L);
    private final AtomicLong evictions = new AtomicLong(0L);
    private final AtomicLong expirations = new AtomicLong(0L);

    private IdentityCache(int maxSize, Duration ttl) {
        if (maxSize <= 0) {
            throw new IllegalArgumentException("Max size must be positive: " + maxSize);
        }
        Objects.requireNonNull(ttl, "TTL cannot be null");
        if (ttl.isNegative() || ttl.isZero()) {
            throw new IllegalArgumentException("TTL must be positive: " + String.valueOf(ttl));
        }
        this.maxSize = maxSize;
        this.ttl = ttl;
        this.entriesById = new ConcurrentHashMap<String, CacheEntry>();
        this.keyNameToId = new ConcurrentHashMap<String, String>();
        this.publicKeyToId = new ConcurrentHashMap<String, String>();
        this.cleanupExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "identity-cache-cleanup");
            t.setDaemon(true);
            return t;
        });
        long cleanupPeriodMs = Math.max(ttl.toMillis() / 2L, 1000L);
        this.cleanupExecutor.scheduleAtFixedRate(this::cleanupExpired, cleanupPeriodMs, cleanupPeriodMs, TimeUnit.MILLISECONDS);
        LOGGER.info("identity_cache_initialized max_size={} ttl_ms={}", (Object)maxSize, (Object)ttl.toMillis());
    }

    public void put(BunkerIdentity identity) {
        Objects.requireNonNull(identity, "Identity cannot be null");
        if (this.entriesById.size() >= this.maxSize) {
            this.evictOldest();
        }
        CacheEntry entry = new CacheEntry(identity);
        this.entriesById.put(identity.getId(), entry);
        this.keyNameToId.put(identity.getKeyName(), identity.getId());
        this.publicKeyToId.put(identity.getPublicKey().toHex(), identity.getId());
        LOGGER.debug("identity_cache_put id={} key_name={}", (Object)identity.getId(), (Object)identity.getKeyName());
    }

    public Optional<BunkerIdentity> getById(String id) {
        Objects.requireNonNull(id, "ID cannot be null");
        CacheEntry entry = this.entriesById.get(id);
        if (entry == null) {
            this.misses.incrementAndGet();
            return Optional.empty();
        }
        if (entry.isExpired(this.ttl)) {
            this.remove(id);
            this.expirations.incrementAndGet();
            this.misses.incrementAndGet();
            return Optional.empty();
        }
        entry.recordAccess();
        this.hits.incrementAndGet();
        return Optional.of(entry.getIdentity());
    }

    public Optional<BunkerIdentity> getByKeyName(String keyName) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        String id = this.keyNameToId.get(keyName);
        if (id == null) {
            this.misses.incrementAndGet();
            return Optional.empty();
        }
        return this.getById(id);
    }

    public Optional<BunkerIdentity> getByPublicKey(PublicKey publicKey) {
        Objects.requireNonNull(publicKey, "Public key cannot be null");
        String id = this.publicKeyToId.get(publicKey.toHex());
        if (id == null) {
            this.misses.incrementAndGet();
            return Optional.empty();
        }
        return this.getById(id);
    }

    public List<BunkerIdentity> getAll() {
        ArrayList<BunkerIdentity> result = new ArrayList<BunkerIdentity>();
        for (CacheEntry entry : this.entriesById.values()) {
            if (entry.isExpired(this.ttl)) continue;
            result.add(entry.getIdentity());
        }
        return result;
    }

    public boolean remove(String id) {
        Objects.requireNonNull(id, "ID cannot be null");
        CacheEntry removed = this.entriesById.remove(id);
        if (removed != null) {
            this.keyNameToId.remove(removed.getIdentity().getKeyName());
            this.publicKeyToId.remove(removed.getIdentity().getPublicKey().toHex());
            LOGGER.debug("identity_cache_remove id={}", (Object)id);
            return true;
        }
        return false;
    }

    public boolean removeByKeyName(String keyName) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        String id = this.keyNameToId.get(keyName);
        if (id != null) {
            return this.remove(id);
        }
        return false;
    }

    public boolean containsId(String id) {
        return this.getById(id).isPresent();
    }

    public boolean containsKeyName(String keyName) {
        return this.getByKeyName(keyName).isPresent();
    }

    public void clear() {
        this.entriesById.clear();
        this.keyNameToId.clear();
        this.publicKeyToId.clear();
        LOGGER.info("identity_cache_cleared");
    }

    public int size() {
        return this.entriesById.size();
    }

    public int getMaxSize() {
        return this.maxSize;
    }

    public Duration getTtl() {
        return this.ttl;
    }

    public CacheStats getStats() {
        return new CacheStats(this.hits.get(), this.misses.get(), this.evictions.get(), this.expirations.get(), this.entriesById.size());
    }

    private void cleanupExpired() {
        int expiredCount = 0;
        for (Map.Entry<String, CacheEntry> entry : this.entriesById.entrySet()) {
            if (!entry.getValue().isExpired(this.ttl)) continue;
            this.remove(entry.getKey());
            this.expirations.incrementAndGet();
            ++expiredCount;
        }
        if (expiredCount > 0) {
            LOGGER.debug("identity_cache_cleanup expired_count={}", (Object)expiredCount);
        }
    }

    private void evictOldest() {
        String oldestId = null;
        Instant oldestAccess = Instant.MAX;
        for (Map.Entry<String, CacheEntry> entry : this.entriesById.entrySet()) {
            if (!entry.getValue().getLastAccess().isBefore(oldestAccess)) continue;
            oldestAccess = entry.getValue().getLastAccess();
            oldestId = entry.getKey();
        }
        if (oldestId != null) {
            this.remove(oldestId);
            this.evictions.incrementAndGet();
            LOGGER.debug("identity_cache_eviction id={}", (Object)oldestId);
        }
    }

    @Override
    public void close() {
        this.cleanupExecutor.shutdown();
        try {
            if (!this.cleanupExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.cleanupExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.cleanupExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
        this.clear();
        LOGGER.info("identity_cache_closed");
    }

    private static class CacheEntry {
        private final BunkerIdentity identity;
        private final Instant createdAt;
        private volatile Instant lastAccess;

        CacheEntry(BunkerIdentity identity) {
            this.identity = identity;
            this.lastAccess = this.createdAt = Instant.now();
        }

        BunkerIdentity getIdentity() {
            return this.identity;
        }

        Instant getLastAccess() {
            return this.lastAccess;
        }

        void recordAccess() {
            this.lastAccess = Instant.now();
        }

        boolean isExpired(Duration ttl) {
            return Instant.now().isAfter(this.createdAt.plus(ttl));
        }
    }

    public record CacheStats(long hits, long misses, long evictions, long expirations, int currentSize) {
        public double hitRatio() {
            long total = this.hits + this.misses;
            return total > 0L ? (double)this.hits / (double)total : 0.0;
        }

        @Override
        public String toString() {
            return String.format("CacheStats{hits=%d, misses=%d, hitRatio=%.2f, evictions=%d, expirations=%d, size=%d}", this.hits, this.misses, this.hitRatio(), this.evictions, this.expirations, this.currentSize);
        }
    }

    public static class Builder {
        private int maxSize = 100;
        private Duration ttl = Duration.ofMinutes(5L);

        public Builder maxSize(int maxSize) {
            this.maxSize = maxSize;
            return this;
        }

        public Builder ttl(Duration ttl) {
            this.ttl = ttl;
            return this;
        }

        public IdentityCache build() {
            return new IdentityCache(this.maxSize, this.ttl);
        }
    }
}

