/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.nsecbunker.admin;

import java.io.Closeable;
import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import nostr.base.PublicKey;
import nostr.crypto.bech32.Bech32;
import nostr.event.BaseTag;
import nostr.event.impl.GenericEvent;
import nostr.event.tag.PubKeyTag;
import nostr.id.Identity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.nsecbunker.admin.AdminConfig;
import xyz.tcheeric.nsecbunker.admin.AdminEventListener;
import xyz.tcheeric.nsecbunker.admin.key.DefaultKeyManager;
import xyz.tcheeric.nsecbunker.admin.key.KeyManager;
import xyz.tcheeric.nsecbunker.admin.permission.DefaultPermissionManager;
import xyz.tcheeric.nsecbunker.admin.permission.PermissionManager;
import xyz.tcheeric.nsecbunker.admin.policy.DefaultPolicyManager;
import xyz.tcheeric.nsecbunker.admin.policy.PolicyManager;
import xyz.tcheeric.nsecbunker.admin.token.DefaultTokenManager;
import xyz.tcheeric.nsecbunker.admin.token.TokenManager;
import xyz.tcheeric.nsecbunker.connection.ConnectionListener;
import xyz.tcheeric.nsecbunker.connection.ConnectionState;
import xyz.tcheeric.nsecbunker.connection.ExponentialBackoffStrategy;
import xyz.tcheeric.nsecbunker.connection.RelayConnection;
import xyz.tcheeric.nsecbunker.connection.RelayListener;
import xyz.tcheeric.nsecbunker.connection.RelayPool;
import xyz.tcheeric.nsecbunker.core.exception.BunkerConnectionException;
import xyz.tcheeric.nsecbunker.protocol.crypto.Nip04Crypto;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Decoder;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Encoder;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Request;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Response;
import xyz.tcheeric.nsecbunker.protocol.nip46.PendingRequestManager;

public class NsecBunkerAdminClient
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(NsecBunkerAdminClient.class);
    public static final int KIND_ADMIN_REQUEST = 24133;
    public static final int KIND_RESPONSE = 24133;
    public static final int KIND_RESPONSE_LEGACY = 24134;
    private final AdminConfig config;
    private final Identity adminIdentity;
    private final Identity ephemeralIdentity;
    private final Identity communicationIdentity;
    private final String bunkerPubkeyHex;
    private final Nip04Crypto crypto;
    private final Nip46Encoder encoder;
    private final Nip46Decoder decoder;
    private RelayPool relayPool;
    private final PendingRequestManager pendingRequests;
    private final List<AdminEventListener> eventListeners;
    private final AtomicReference<ConnectionState> connectionState;
    private final AtomicBoolean closed;
    private volatile String subscriptionId;

    private NsecBunkerAdminClient(AdminConfig config) {
        this.config = Objects.requireNonNull(config, "Config must not be null");
        config.validate();
        this.adminIdentity = this.createIdentity(config.getAdminPrivateKey());
        if (config.isUseEphemeralKey()) {
            this.communicationIdentity = this.ephemeralIdentity = Identity.generateRandomIdentity();
            log.debug("Using ephemeral key for communication: {}", (Object)this.ephemeralIdentity.getPublicKey().toString());
        } else {
            this.ephemeralIdentity = null;
            this.communicationIdentity = this.adminIdentity;
            log.debug("Using admin key directly for communication");
        }
        this.bunkerPubkeyHex = this.resolvePubkeyToHex(config.getBunkerPubkey());
        String privateKeyHex = this.communicationIdentity.getPrivateKey().toHexString();
        this.crypto = Nip04Crypto.create(privateKeyHex, this.bunkerPubkeyHex);
        this.encoder = new Nip46Encoder();
        this.decoder = new Nip46Decoder();
        this.pendingRequests = new PendingRequestManager();
        this.eventListeners = new CopyOnWriteArrayList<AdminEventListener>();
        this.connectionState = new AtomicReference<ConnectionState>(ConnectionState.DISCONNECTED);
        this.closed = new AtomicBoolean(false);
    }

    public static Builder builder() {
        return new Builder();
    }

    public void connect() throws BunkerConnectionException {
        if (this.closed.get()) {
            throw new BunkerConnectionException("Client has been closed");
        }
        if (this.connectionState.get() == ConnectionState.CONNECTED) {
            log.debug("Already connected");
            return;
        }
        this.connectionState.set(ConnectionState.CONNECTING);
        log.info("Connecting to bunker at {} via {} relays", (Object)this.bunkerPubkeyHex, (Object)this.config.getRelays().size());
        try {
            RelayPool.Builder poolBuilder = RelayPool.builder().connectTimeout(this.config.getConnectTimeout()).minConnectedRelays(1).deduplicateEvents(true);
            for (String relayUrl : this.config.getRelays()) {
                poolBuilder.relay(relayUrl);
            }
            this.relayPool = poolBuilder.build();
            AdminRelayListener relayListener = new AdminRelayListener();
            AdminConnectionListener connectionListener = new AdminConnectionListener();
            for (RelayConnection relay : this.relayPool.getRelays()) {
                relay.addListener(relayListener);
                relay.addConnectionListener(connectionListener);
                if (!this.config.isAutoReconnect()) continue;
                relay.setReconnectionStrategy(ExponentialBackoffStrategy.builder().maxAttempts(this.config.getMaxReconnectAttempts()).initialDelay(this.config.getReconnectDelay()).maxDelay(this.config.getMaxReconnectDelay()).build());
            }
            this.relayPool.connectAll(this.config.getConnectTimeout());
            this.subscribeToResponses();
            this.connectionState.set(ConnectionState.CONNECTED);
            log.info("Connected to bunker successfully");
        }
        catch (Exception e) {
            this.connectionState.set(ConnectionState.FAILED);
            throw new BunkerConnectionException("Failed to connect to bunker: " + e.getMessage(), this.config.getRelays().isEmpty() ? null : this.config.getRelays().get(0), e);
        }
    }

    public CompletableFuture<Void> connectAsync() {
        return CompletableFuture.runAsync(() -> {
            try {
                this.connect();
            }
            catch (BunkerConnectionException e) {
                throw new RuntimeException(e);
            }
        });
    }

    public CompletableFuture<Nip46Response> sendRequest(Nip46Request request) {
        if (this.closed.get()) {
            return CompletableFuture.failedFuture(new IllegalStateException("Client has been closed"));
        }
        if (this.connectionState.get() != ConnectionState.CONNECTED) {
            return CompletableFuture.failedFuture(new IllegalStateException("Not connected to bunker"));
        }
        Duration timeout2 = this.config.getRequestTimeout();
        CompletableFuture<Nip46Response> future = this.pendingRequests.register(request, timeout2);
        try {
            String requestJson = this.encoder.encodeRequest(request);
            String encryptedContent = this.crypto.encrypt(requestJson);
            String eventJson = this.createAdminEventJson(encryptedContent);
            int sent = this.relayPool.broadcastEvent(eventJson);
            if (sent == 0) {
                this.pendingRequests.cancel(request.getId());
                return CompletableFuture.failedFuture(new RuntimeException("Failed to send request to any relay"));
            }
            log.debug("Sent admin request: method={}, id={}", (Object)request.getMethod(), (Object)request.getId());
        }
        catch (Exception e) {
            this.pendingRequests.cancel(request.getId());
            return CompletableFuture.failedFuture(e);
        }
        return future;
    }

    public CompletableFuture<String> ping() {
        Nip46Request request = Nip46Request.ping();
        return this.sendRequest(request).thenApply(response -> {
            if (response.isError()) {
                throw new RuntimeException("Ping failed: " + String.valueOf(response.getError()));
            }
            return response.getResult();
        });
    }

    public ConnectionState getConnectionState() {
        return this.connectionState.get();
    }

    public boolean isConnected() {
        return this.connectionState.get() == ConnectionState.CONNECTED;
    }

    public KeyManager keyManager() {
        return new DefaultKeyManager(this);
    }

    public PolicyManager policyManager() {
        return new DefaultPolicyManager(this);
    }

    public PermissionManager permissionManager() {
        return new DefaultPermissionManager(this);
    }

    public TokenManager tokenManager() {
        return new DefaultTokenManager(this);
    }

    public void addEventListener(AdminEventListener listener) {
        if (listener != null) {
            this.eventListeners.add(listener);
        }
    }

    public void removeEventListener(AdminEventListener listener) {
        this.eventListeners.remove(listener);
    }

    public void disconnect() {
        if (this.connectionState.compareAndSet(ConnectionState.CONNECTED, ConnectionState.DISCONNECTING)) {
            log.info("Disconnecting from bunker");
            this.pendingRequests.cancelAll();
            this.relayPool.close();
            this.connectionState.set(ConnectionState.DISCONNECTED);
            log.info("Disconnected from bunker");
        }
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            this.disconnect();
            this.pendingRequests.shutdown();
        }
    }

    private Identity createIdentity(String privateKey) {
        String hexKey = privateKey;
        if (privateKey.startsWith("nsec1")) {
            try {
                hexKey = Bech32.fromBech32(privateKey);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid nsec: " + privateKey, e);
            }
        }
        return Identity.create(hexKey);
    }

    private String resolvePubkeyToHex(String pubkey) {
        if (pubkey.startsWith("npub1")) {
            try {
                return Bech32.fromBech32(pubkey);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid npub: " + pubkey, e);
            }
        }
        return pubkey;
    }

    private void subscribeToResponses() {
        String commPubkeyHex = this.communicationIdentity.getPublicKey().toString();
        String kinds = String.format("[%d,%d]", 24133, 24134);
        String filter2 = String.format("{\"kinds\":%s,\"#p\":[\"%s\"],\"authors\":[\"%s\"]}", kinds, commPubkeyHex, this.bunkerPubkeyHex);
        this.subscriptionId = "admin-" + System.currentTimeMillis();
        this.relayPool.broadcastReq(this.subscriptionId, filter2);
        log.debug("Subscribed for responses with filter: {}", (Object)filter2);
    }

    private String createAdminEventJson(String encryptedContent) {
        String commPubkeyHex = this.communicationIdentity.getPublicKey().toString();
        long createdAt = System.currentTimeMillis() / 1000L;
        GenericEvent event = new GenericEvent(this.communicationIdentity.getPublicKey(), 24133);
        event.setContent(encryptedContent);
        event.setCreatedAt(createdAt);
        PubKeyTag pTag = new PubKeyTag(new PublicKey(this.bunkerPubkeyHex));
        event.addTag(pTag);
        event.update();
        this.communicationIdentity.sign(event);
        return this.serializeEvent(event);
    }

    private String serializeEvent(GenericEvent event) {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        sb.append("\"id\":\"").append(event.getId()).append("\",");
        sb.append("\"pubkey\":\"").append(event.getPubKey().toString()).append("\",");
        sb.append("\"created_at\":").append(event.getCreatedAt()).append(",");
        sb.append("\"kind\":").append(event.getKind()).append(",");
        sb.append("\"tags\":[");
        List<BaseTag> tags = event.getTags();
        for (int i2 = 0; i2 < tags.size(); ++i2) {
            if (i2 > 0) {
                sb.append(",");
            }
            BaseTag tag = tags.get(i2);
            sb.append("[\"").append(tag.getCode()).append("\"");
            if (tag instanceof PubKeyTag) {
                PubKeyTag pt = (PubKeyTag)tag;
                sb.append(",\"").append(pt.getPublicKey().toString()).append("\"");
            }
            sb.append("]");
        }
        sb.append("],");
        sb.append("\"content\":\"").append(this.escapeJson(event.getContent())).append("\",");
        sb.append("\"sig\":\"").append(event.getSignature().toString()).append("\"");
        sb.append("}");
        return sb.toString();
    }

    private String escapeJson(String value) {
        if (value == null) {
            return "";
        }
        return value.replace("\\", "\\\\").replace("\"", "\\\"").replace("\n", "\\n").replace("\r", "\\r").replace("\t", "\\t");
    }

    private void handleEvent(GenericEvent event) {
        int kind = event.getKind();
        if (kind != 24133 && kind != 24134) {
            return;
        }
        if (!event.getPubKey().toString().equals(this.bunkerPubkeyHex)) {
            log.debug("Ignoring event from unknown pubkey: {}", (Object)event.getPubKey());
            return;
        }
        try {
            String decrypted = this.crypto.decrypt(event.getContent());
            log.debug("Decrypted response (kind={}): {}", (Object)kind, (Object)decrypted);
            Nip46Response response = this.decoder.decodeResponse(decrypted);
            if (this.pendingRequests.complete(response)) {
                log.debug("Completed request: id={}", (Object)response.getId());
            } else {
                log.debug("No pending request for response: id={}", (Object)response.getId());
            }
            for (AdminEventListener listener : this.eventListeners) {
                try {
                    listener.onResponse(this, response);
                }
                catch (Exception e) {
                    log.warn("Error in event listener: {}", (Object)e.getMessage());
                }
            }
        }
        catch (Exception e) {
            log.error("Failed to handle response event: {}", (Object)e.getMessage());
        }
    }

    private static String bytesToHex(byte[] bytes) {
        StringBuilder sb = new StringBuilder(bytes.length * 2);
        for (byte b : bytes) {
            sb.append(String.format("%02x", b));
        }
        return sb.toString();
    }

    public AdminConfig getConfig() {
        return this.config;
    }

    public Identity getCommunicationIdentity() {
        return this.communicationIdentity;
    }

    public String getBunkerPubkeyHex() {
        return this.bunkerPubkeyHex;
    }

    public static class Builder {
        private final AdminConfig.AdminConfigBuilder configBuilder = AdminConfig.builder();

        public Builder bunkerPubkey(String bunkerPubkey) {
            this.configBuilder.bunkerPubkey(bunkerPubkey);
            return this;
        }

        public Builder adminPrivateKey(String adminPrivateKey) {
            this.configBuilder.adminPrivateKey(adminPrivateKey);
            return this;
        }

        public Builder relay(String relay) {
            return this.relays(List.of(relay));
        }

        public Builder relays(List<String> relays) {
            this.configBuilder.relays(relays);
            return this;
        }

        public Builder secret(String secret) {
            this.configBuilder.secret(secret);
            return this;
        }

        public Builder connectTimeout(Duration connectTimeout) {
            this.configBuilder.connectTimeout(connectTimeout);
            return this;
        }

        public Builder requestTimeout(Duration requestTimeout) {
            this.configBuilder.requestTimeout(requestTimeout);
            return this;
        }

        public Builder autoReconnect(boolean autoReconnect) {
            this.configBuilder.autoReconnect(autoReconnect);
            return this;
        }

        public Builder maxReconnectAttempts(int maxReconnectAttempts) {
            this.configBuilder.maxReconnectAttempts(maxReconnectAttempts);
            return this;
        }

        public Builder reconnectDelay(Duration reconnectDelay) {
            this.configBuilder.reconnectDelay(reconnectDelay);
            return this;
        }

        public Builder maxReconnectDelay(Duration maxReconnectDelay) {
            this.configBuilder.maxReconnectDelay(maxReconnectDelay);
            return this;
        }

        public Builder useEphemeralKey(boolean useEphemeralKey) {
            this.configBuilder.useEphemeralKey(useEphemeralKey);
            return this;
        }

        public Builder config(AdminConfig config) {
            return this.bunkerPubkey(config.getBunkerPubkey()).adminPrivateKey(config.getAdminPrivateKey()).relays(config.getRelays()).secret(config.getSecret()).connectTimeout(config.getConnectTimeout()).requestTimeout(config.getRequestTimeout()).autoReconnect(config.isAutoReconnect()).maxReconnectAttempts(config.getMaxReconnectAttempts()).reconnectDelay(config.getReconnectDelay()).maxReconnectDelay(config.getMaxReconnectDelay()).useEphemeralKey(config.isUseEphemeralKey());
        }

        public Builder fromConnectionString(String connectionString, String adminPrivateKey) {
            AdminConfig config = AdminConfig.fromConnectionString(connectionString, adminPrivateKey);
            return this.config(config);
        }

        public NsecBunkerAdminClient build() {
            return new NsecBunkerAdminClient(this.configBuilder.build());
        }
    }

    private class AdminRelayListener
    implements RelayListener {
        private AdminRelayListener() {
        }

        @Override
        public void onEvent(RelayConnection relay, String subId, GenericEvent event) {
            NsecBunkerAdminClient.this.handleEvent(event);
        }

        @Override
        public void onOk(RelayConnection relay, String eventId, boolean success, String message) {
            log.debug("OK from {}: eventId={}, success={}, message={}", relay.getUrl(), eventId, success, message);
        }

        @Override
        public void onNotice(RelayConnection relay, String message) {
            log.info("Notice from {}: {}", (Object)relay.getUrl(), (Object)message);
        }
    }

    private class AdminConnectionListener
    implements ConnectionListener {
        private AdminConnectionListener() {
        }

        @Override
        public void onConnected(String relayUrl) {
            log.debug("Connected to relay: {}", (Object)relayUrl);
        }

        @Override
        public void onDisconnected(String relayUrl, int code, String reason) {
            log.debug("Disconnected from relay: {} (code={}, reason={})", relayUrl, code, reason);
        }

        @Override
        public void onError(String relayUrl, Throwable error) {
            log.error("Error from relay {}: {}", (Object)relayUrl, (Object)error.getMessage());
        }

        @Override
        public void onReconnecting(String relayUrl, int attempt) {
            log.info("Reconnecting to relay {} (attempt {})", (Object)relayUrl, (Object)attempt);
        }
    }
}

