/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.nsecbunker.connection;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.nsecbunker.connection.ConnectionHealth;
import xyz.tcheeric.nsecbunker.connection.ConnectionListener;
import xyz.tcheeric.nsecbunker.connection.ConnectionState;
import xyz.tcheeric.nsecbunker.connection.HealthMonitor;
import xyz.tcheeric.nsecbunker.connection.RelayConnection;

public class RelayHealthMonitor
implements HealthMonitor {
    private static final Logger log = LoggerFactory.getLogger(RelayHealthMonitor.class);
    private static final Duration DEFAULT_CHECK_INTERVAL = Duration.ofSeconds(30L);
    private static final Duration DEFAULT_PING_TIMEOUT = Duration.ofSeconds(10L);
    private static final int DEFAULT_UNHEALTHY_THRESHOLD = 3;
    private static final int LATENCY_SAMPLE_SIZE = 10;
    private final RelayConnection connection;
    private final List<Consumer<ConnectionHealth>> listeners;
    private final AtomicBoolean running;
    private final AtomicReference<ConnectionHealth> currentHealth;
    private final AtomicLong messagesSent;
    private final AtomicLong messagesReceived;
    private final AtomicLong successfulPings;
    private final AtomicLong failedPings;
    private final AtomicReference<Instant> connectedSince;
    private final AtomicReference<Instant> lastPingTime;
    private final AtomicReference<Instant> lastPongTime;
    private final long[] latencySamples;
    private int latencyIndex;
    private int latencyCount;
    private final Object latencyLock = new Object();
    private volatile Duration checkInterval;
    private volatile Duration pingTimeout;
    private volatile int unhealthyThreshold;
    private volatile ScheduledExecutorService executor;
    private volatile ScheduledFuture<?> checkFuture;
    private final ConnectionListener connectionListener;

    public RelayHealthMonitor(RelayConnection connection) {
        this.connection = Objects.requireNonNull(connection, "Connection must not be null");
        this.listeners = new CopyOnWriteArrayList<Consumer<ConnectionHealth>>();
        this.running = new AtomicBoolean(false);
        this.messagesSent = new AtomicLong(0L);
        this.messagesReceived = new AtomicLong(0L);
        this.successfulPings = new AtomicLong(0L);
        this.failedPings = new AtomicLong(0L);
        this.connectedSince = new AtomicReference();
        this.lastPingTime = new AtomicReference();
        this.lastPongTime = new AtomicReference();
        this.latencySamples = new long[10];
        this.latencyIndex = 0;
        this.latencyCount = 0;
        this.checkInterval = DEFAULT_CHECK_INTERVAL;
        this.pingTimeout = DEFAULT_PING_TIMEOUT;
        this.unhealthyThreshold = 3;
        this.currentHealth = new AtomicReference<ConnectionHealth>(ConnectionHealth.disconnected(connection.getUrl()));
        this.connectionListener = new ConnectionListener(){

            @Override
            public void onConnected(String url) {
                RelayHealthMonitor.this.connectedSince.set(Instant.now());
                RelayHealthMonitor.this.resetMetrics();
                RelayHealthMonitor.this.updateHealth();
            }

            @Override
            public void onDisconnected(String url, int code, String reason) {
                RelayHealthMonitor.this.connectedSince.set(null);
                RelayHealthMonitor.this.updateHealth();
            }

            @Override
            public void onError(String url, Throwable error) {
                RelayHealthMonitor.this.updateHealth();
            }

            @Override
            public void onStateChanged(String url, ConnectionState oldState, ConnectionState newState) {
                RelayHealthMonitor.this.updateHealth();
            }
        };
    }

    @Override
    public void start() {
        if (!this.running.compareAndSet(false, true)) {
            return;
        }
        log.debug("Starting health monitor for {}", (Object)this.connection.getUrl());
        this.connection.addConnectionListener(this.connectionListener);
        if (this.connection.isConnected()) {
            this.connectedSince.set(Instant.now());
        }
        this.updateHealth();
        this.executor = new ScheduledThreadPoolExecutor(1);
        this.scheduleNextCheck();
    }

    @Override
    public void stop() {
        if (!this.running.compareAndSet(true, false)) {
            return;
        }
        log.debug("Stopping health monitor for {}", (Object)this.connection.getUrl());
        this.connection.removeConnectionListener(this.connectionListener);
        if (this.checkFuture != null) {
            this.checkFuture.cancel(false);
            this.checkFuture = null;
        }
        if (this.executor != null) {
            this.executor.shutdownNow();
            this.executor = null;
        }
    }

    @Override
    public boolean isRunning() {
        return this.running.get();
    }

    @Override
    public ConnectionHealth getHealth() {
        return this.currentHealth.get();
    }

    @Override
    public void checkNow() {
        if (this.running.get()) {
            this.performHealthCheck();
        }
    }

    @Override
    public void addHealthListener(Consumer<ConnectionHealth> listener) {
        if (listener != null) {
            this.listeners.add(listener);
        }
    }

    @Override
    public void removeHealthListener(Consumer<ConnectionHealth> listener) {
        this.listeners.remove(listener);
    }

    @Override
    public void setCheckInterval(Duration interval) {
        Objects.requireNonNull(interval, "Interval must not be null");
        if (interval.isNegative() || interval.isZero()) {
            throw new IllegalArgumentException("Interval must be positive");
        }
        this.checkInterval = interval;
    }

    @Override
    public Duration getCheckInterval() {
        return this.checkInterval;
    }

    @Override
    public void setPingTimeout(Duration timeout2) {
        Objects.requireNonNull(timeout2, "Timeout must not be null");
        if (timeout2.isNegative() || timeout2.isZero()) {
            throw new IllegalArgumentException("Timeout must be positive");
        }
        this.pingTimeout = timeout2;
    }

    @Override
    public Duration getPingTimeout() {
        return this.pingTimeout;
    }

    @Override
    public void setUnhealthyThreshold(int threshold) {
        if (threshold < 1) {
            throw new IllegalArgumentException("Threshold must be at least 1");
        }
        this.unhealthyThreshold = threshold;
    }

    @Override
    public int getUnhealthyThreshold() {
        return this.unhealthyThreshold;
    }

    public void recordMessageSent() {
        this.messagesSent.incrementAndGet();
    }

    public void recordMessageReceived() {
        this.messagesReceived.incrementAndGet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void recordLatency(Duration latency) {
        if (latency == null || latency.isNegative()) {
            return;
        }
        Object object = this.latencyLock;
        synchronized (object) {
            this.latencySamples[this.latencyIndex] = latency.toMillis();
            this.latencyIndex = (this.latencyIndex + 1) % 10;
            if (this.latencyCount < 10) {
                ++this.latencyCount;
            }
        }
        this.lastPongTime.set(Instant.now());
        this.successfulPings.incrementAndGet();
        this.updateHealth();
    }

    public void recordPingFailure() {
        this.failedPings.incrementAndGet();
        this.updateHealth();
    }

    public void recordPingSent() {
        this.lastPingTime.set(Instant.now());
    }

    public RelayConnection getConnection() {
        return this.connection;
    }

    private void scheduleNextCheck() {
        if (!this.running.get() || this.executor == null) {
            return;
        }
        this.checkFuture = this.executor.schedule(() -> {
            try {
                this.performHealthCheck();
            }
            finally {
                this.scheduleNextCheck();
            }
        }, this.checkInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void performHealthCheck() {
        Duration sinceLastPong;
        if (!this.running.get()) {
            return;
        }
        log.trace("Performing health check for {}", (Object)this.connection.getUrl());
        Instant lastPong = this.lastPongTime.get();
        if (lastPong != null && this.connection.isConnected() && (sinceLastPong = Duration.between(lastPong, Instant.now())).compareTo(this.checkInterval.multipliedBy(2L)) > 0) {
            this.recordPingFailure();
        }
        this.updateHealth();
    }

    private void updateHealth() {
        ConnectionHealth oldHealth = this.currentHealth.get();
        ConnectionHealth newHealth = this.buildHealthSnapshot();
        this.currentHealth.set(newHealth);
        if (this.healthChanged(oldHealth, newHealth)) {
            this.notifyListeners(newHealth);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ConnectionHealth buildHealthSnapshot() {
        ConnectionState state = this.connection.getState();
        boolean connected = state == ConnectionState.CONNECTED;
        Duration latency = null;
        Duration avgLatency = null;
        Duration minLatency = null;
        Duration maxLatency = null;
        Object object = this.latencyLock;
        synchronized (object) {
            if (this.latencyCount > 0) {
                long sum = 0L;
                long min = Long.MAX_VALUE;
                long max = Long.MIN_VALUE;
                long last = this.latencySamples[(this.latencyIndex - 1 + 10) % 10];
                for (int i2 = 0; i2 < this.latencyCount; ++i2) {
                    long sample = this.latencySamples[i2];
                    sum += sample;
                    min = Math.min(min, sample);
                    max = Math.max(max, sample);
                }
                latency = Duration.ofMillis(last);
                avgLatency = Duration.ofMillis(sum / (long)this.latencyCount);
                minLatency = Duration.ofMillis(min);
                maxLatency = Duration.ofMillis(max);
            }
        }
        long totalPings = this.successfulPings.get() + this.failedPings.get();
        int consecutiveFailures = this.calculateConsecutiveFailures();
        boolean healthy = connected && consecutiveFailures < this.unhealthyThreshold;
        return ConnectionHealth.builder().url(this.connection.getUrl()).state(state).healthy(healthy).latency(latency).averageLatency(avgLatency).minLatency(minLatency).maxLatency(maxLatency).timestamp(Instant.now()).lastPingTime(this.lastPingTime.get()).lastPongTime(this.lastPongTime.get()).connectedSince(this.connectedSince.get()).successfulPings(this.successfulPings.get()).failedPings(this.failedPings.get()).consecutiveFailures(consecutiveFailures).messagesSent(this.messagesSent.get()).messagesReceived(this.messagesReceived.get()).build();
    }

    private int calculateConsecutiveFailures() {
        Instant lastPong = this.lastPongTime.get();
        if (lastPong == null) {
            return 0;
        }
        Duration sinceLastPong = Duration.between(lastPong, Instant.now());
        long missedChecks = sinceLastPong.toMillis() / this.checkInterval.toMillis();
        return (int)Math.max(0L, missedChecks - 1L);
    }

    private boolean healthChanged(ConnectionHealth oldHealth, ConnectionHealth newHealth) {
        if (oldHealth == null) {
            return true;
        }
        return oldHealth.isHealthy() != newHealth.isHealthy() || oldHealth.getState() != newHealth.getState();
    }

    private void notifyListeners(ConnectionHealth health) {
        for (Consumer<ConnectionHealth> listener : this.listeners) {
            try {
                listener.accept(health);
            }
            catch (Exception e) {
                log.warn("Error in health listener: {}", (Object)e.getMessage());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void resetMetrics() {
        Object object = this.latencyLock;
        synchronized (object) {
            this.latencyIndex = 0;
            this.latencyCount = 0;
        }
    }
}

