/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.nsecbunker.connection;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import nostr.event.impl.GenericEvent;
import okhttp3.OkHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.nsecbunker.connection.ConnectionHealth;
import xyz.tcheeric.nsecbunker.connection.ConnectionState;
import xyz.tcheeric.nsecbunker.connection.RelayConnection;
import xyz.tcheeric.nsecbunker.connection.RelayHealthMonitor;
import xyz.tcheeric.nsecbunker.connection.RelayListener;
import xyz.tcheeric.nsecbunker.connection.RelayPoolListener;
import xyz.tcheeric.nsecbunker.core.exception.BunkerConnectionException;

public class RelayPool
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(RelayPool.class);
    private final Map<String, RelayConnection> relays = new ConcurrentHashMap<String, RelayConnection>();
    private final OkHttpClient client;
    private final Duration connectTimeout;
    private final int minConnectedRelays;
    private final List<RelayPoolListener> listeners;
    private final Set<String> seenEventIds;
    private final boolean deduplicateEvents;
    private final RelayListener internalListener;
    private volatile boolean closed;

    private RelayPool(Builder builder) {
        this.client = builder.client != null ? builder.client : this.createDefaultClient(builder.connectTimeout);
        this.connectTimeout = builder.connectTimeout;
        this.minConnectedRelays = builder.minConnectedRelays;
        this.listeners = new CopyOnWriteArrayList<RelayPoolListener>();
        this.seenEventIds = new CopyOnWriteArraySet<String>();
        this.deduplicateEvents = builder.deduplicateEvents;
        this.internalListener = new InternalRelayListener();
        this.closed = false;
        for (String url : builder.relayUrls) {
            this.addRelay(url);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public RelayConnection addRelay(String url) {
        Objects.requireNonNull(url, "URL must not be null");
        if (this.closed) {
            throw new IllegalStateException("Pool is closed");
        }
        return this.relays.computeIfAbsent(url, u -> {
            RelayConnection relay = new RelayConnection((String)u, this.client, this.connectTimeout);
            relay.addListener(this.internalListener);
            log.debug("Added relay to pool: {}", u);
            return relay;
        });
    }

    public boolean removeRelay(String url) {
        RelayConnection relay = this.relays.remove(url);
        if (relay != null) {
            relay.removeListener(this.internalListener);
            relay.close();
            log.debug("Removed relay from pool: {}", (Object)url);
            return true;
        }
        return false;
    }

    public RelayConnection getRelay(String url) {
        return this.relays.get(url);
    }

    public Collection<RelayConnection> getRelays() {
        return Collections.unmodifiableCollection(this.relays.values());
    }

    public Set<String> getRelayUrls() {
        return Collections.unmodifiableSet(this.relays.keySet());
    }

    public int size() {
        return this.relays.size();
    }

    public boolean isEmpty() {
        return this.relays.isEmpty();
    }

    public List<RelayConnection> getConnectedRelays() {
        return this.relays.values().stream().filter(RelayConnection::isConnected).collect(Collectors.toList());
    }

    public int getConnectedCount() {
        return (int)this.relays.values().stream().filter(RelayConnection::isConnected).count();
    }

    public Map<String, ConnectionHealth> getHealthMap() {
        return this.relays.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ((RelayConnection)e.getValue()).getHealth()));
    }

    public List<RelayConnection> getHealthyRelays() {
        return this.relays.values().stream().filter(r -> r.getHealth().isHealthy()).collect(Collectors.toList());
    }

    public int getHealthyCount() {
        return (int)this.relays.values().stream().filter(r -> r.getHealth().isHealthy()).count();
    }

    public void startHealthMonitoring() {
        for (RelayConnection relay : this.relays.values()) {
            RelayHealthMonitor monitor = relay.getHealthMonitor();
            if (monitor.isRunning()) continue;
            monitor.start();
        }
    }

    public void stopHealthMonitoring() {
        for (RelayConnection relay : this.relays.values()) {
            RelayHealthMonitor monitor = relay.getHealthMonitor();
            if (!monitor.isRunning()) continue;
            monitor.stop();
        }
    }

    public boolean hasMinimumConnections() {
        return this.getConnectedCount() >= this.minConnectedRelays;
    }

    public void addListener(RelayPoolListener listener) {
        if (listener != null) {
            this.listeners.add(listener);
        }
    }

    public void removeListener(RelayPoolListener listener) {
        this.listeners.remove(listener);
    }

    public void connectAll() throws BunkerConnectionException {
        this.connectAll(this.connectTimeout);
    }

    public void connectAll(Duration timeout2) throws BunkerConnectionException {
        if (this.closed) {
            throw new IllegalStateException("Pool is closed");
        }
        if (this.relays.isEmpty()) {
            throw new BunkerConnectionException("No relays in pool");
        }
        CountDownLatch latch = new CountDownLatch(this.relays.size());
        AtomicInteger successCount = new AtomicInteger(0);
        CopyOnWriteArrayList errors = new CopyOnWriteArrayList();
        for (RelayConnection relay : this.relays.values()) {
            new Thread(() -> {
                try {
                    relay.connect(timeout2);
                    successCount.incrementAndGet();
                }
                catch (Exception e) {
                    log.warn("Failed to connect to {}: {}", (Object)relay.getUrl(), (Object)e.getMessage());
                    errors.add(e);
                }
                finally {
                    latch.countDown();
                }
            }, "relay-connect-" + relay.getUrl()).start();
        }
        try {
            boolean completed = latch.await(timeout2.toMillis() + 1000L, TimeUnit.MILLISECONDS);
            if (!completed) {
                log.warn("Connection timeout - some relays did not respond");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BunkerConnectionException("Connection interrupted", e);
        }
        if (successCount.get() < this.minConnectedRelays) {
            throw new BunkerConnectionException(String.format("Only %d of %d required relays connected", successCount.get(), this.minConnectedRelays));
        }
        log.info("Connected to {}/{} relays", (Object)successCount.get(), (Object)this.relays.size());
    }

    public void connectAllAsync() {
        if (this.closed) {
            return;
        }
        for (RelayConnection relay : this.relays.values()) {
            relay.connectAsync();
        }
    }

    public int broadcast(String message) {
        return this.sendTo(message, (RelayConnection relay) -> relay.isConnected());
    }

    public int sendTo(String message, Predicate<RelayConnection> predicate) {
        int count = 0;
        for (RelayConnection relay : this.relays.values()) {
            if (!predicate.test(relay)) continue;
            try {
                if (!relay.send(message)) continue;
                ++count;
            }
            catch (Exception e) {
                log.warn("Failed to send to {}: {}", (Object)relay.getUrl(), (Object)e.getMessage());
            }
        }
        return count;
    }

    public boolean sendTo(String url, String message) {
        RelayConnection relay = this.relays.get(url);
        if (relay != null && relay.isConnected()) {
            try {
                return relay.send(message);
            }
            catch (Exception e) {
                log.warn("Failed to send to {}: {}", (Object)url, (Object)e.getMessage());
            }
        }
        return false;
    }

    public int broadcastReq(String subscriptionId, String ... filters) {
        StringBuilder sb = new StringBuilder("[\"REQ\",\"").append(subscriptionId).append("\"");
        for (String filter2 : filters) {
            sb.append(",").append(filter2);
        }
        sb.append("]");
        return this.broadcast(sb.toString());
    }

    public int broadcastClose(String subscriptionId) {
        return this.broadcast("[\"CLOSE\",\"" + subscriptionId + "\"]");
    }

    public int broadcastEvent(String eventJson) {
        return this.broadcast("[\"EVENT\"," + eventJson + "]");
    }

    public void disconnectAll() {
        for (RelayConnection relay : this.relays.values()) {
            try {
                relay.close();
            }
            catch (Exception e) {
                log.warn("Error closing {}: {}", (Object)relay.getUrl(), (Object)e.getMessage());
            }
        }
    }

    public void clearSeenEvents() {
        this.seenEventIds.clear();
    }

    public int getSeenEventCount() {
        return this.seenEventIds.size();
    }

    @Override
    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.disconnectAll();
        this.relays.clear();
        this.listeners.clear();
        this.seenEventIds.clear();
        log.debug("RelayPool closed");
    }

    public boolean isClosed() {
        return this.closed;
    }

    private OkHttpClient createDefaultClient(Duration timeout2) {
        return new OkHttpClient.Builder().connectTimeout(timeout2).readTimeout(Duration.ofMinutes(5L)).writeTimeout(Duration.ofSeconds(30L)).pingInterval(Duration.ofSeconds(30L)).build();
    }

    public String toString() {
        return String.format("RelayPool{relays=%d, connected=%d, closed=%s}", this.relays.size(), this.getConnectedCount(), this.closed);
    }

    public Duration getConnectTimeout() {
        return this.connectTimeout;
    }

    public int getMinConnectedRelays() {
        return this.minConnectedRelays;
    }

    public boolean isDeduplicateEvents() {
        return this.deduplicateEvents;
    }

    public static class Builder {
        private final List<String> relayUrls = new ArrayList<String>();
        private OkHttpClient client;
        private Duration connectTimeout = Duration.ofSeconds(30L);
        private int minConnectedRelays = 1;
        private boolean deduplicateEvents = true;

        public Builder relay(String url) {
            if (url != null && !url.isEmpty()) {
                this.relayUrls.add(url);
            }
            return this;
        }

        public Builder relays(Collection<String> urls2) {
            if (urls2 != null) {
                urls2.stream().filter(u -> u != null && !u.isEmpty()).forEach(this.relayUrls::add);
            }
            return this;
        }

        public Builder client(OkHttpClient client) {
            this.client = client;
            return this;
        }

        public Builder connectTimeout(Duration timeout2) {
            this.connectTimeout = timeout2 != null ? timeout2 : Duration.ofSeconds(30L);
            return this;
        }

        public Builder minConnectedRelays(int min) {
            this.minConnectedRelays = Math.max(1, min);
            return this;
        }

        public Builder deduplicateEvents(boolean deduplicate) {
            this.deduplicateEvents = deduplicate;
            return this;
        }

        public RelayPool build() {
            return new RelayPool(this);
        }
    }

    private class InternalRelayListener
    implements RelayListener {
        private InternalRelayListener() {
        }

        @Override
        public void onStateChange(RelayConnection relay, ConnectionState oldState, ConnectionState newState) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onRelayStateChange(relay, oldState, newState);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onRelayStateChange: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onConnect(RelayConnection relay) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onRelayConnect(relay);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onRelayConnect: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onDisconnect(RelayConnection relay, int code, String reason) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onRelayDisconnect(relay, code, reason);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onRelayDisconnect: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onEvent(RelayConnection relay, String subscriptionId, GenericEvent event) {
            if (RelayPool.this.deduplicateEvents && event.getId() != null && !RelayPool.this.seenEventIds.add(event.getId())) {
                log.debug("Duplicate event {} from {}", (Object)event.getId(), (Object)relay.getUrl());
                return;
            }
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onEvent(relay, subscriptionId, event);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onEvent: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onOk(RelayConnection relay, String eventId, boolean success, String message) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onOk(relay, eventId, success, message);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onOk: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onEndOfStoredEvents(RelayConnection relay, String subscriptionId) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onEndOfStoredEvents(relay, subscriptionId);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onEndOfStoredEvents: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onNotice(RelayConnection relay, String message) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onNotice(relay, message);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onNotice: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onClosed(RelayConnection relay, String subscriptionId, String message) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onClosed(relay, subscriptionId, message);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onClosed: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onAuth(RelayConnection relay, String challenge) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onAuth(relay, challenge);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onAuth: {}", (Object)e.getMessage());
                }
            }
        }

        @Override
        public void onError(RelayConnection relay, Throwable throwable) {
            for (RelayPoolListener listener : RelayPool.this.listeners) {
                try {
                    listener.onRelayError(relay, throwable);
                }
                catch (Exception e) {
                    log.warn("Error in pool listener onRelayError: {}", (Object)e.getMessage());
                }
            }
        }
    }
}

