/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.storage.file;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Set;
import xyz.tcheeric.identity.api.exceptions.IdentityStorageException;
import xyz.tcheeric.identity.api.exceptions.KeyNotFoundException;
import xyz.tcheeric.identity.api.ports.IdentityStorage;
import xyz.tcheeric.identity.domain.PrivateKey;
import xyz.tcheeric.identity.storage.file.encryption.PasswordBasedEncryption;

public class EncryptedFileStorage
implements IdentityStorage {
    private static final String DEFAULT_BASE_DIR = System.getProperty("user.home") + "/.cashu/identities/keys";
    private static final String FILE_EXTENSION = ".enc";
    private static final Set<PosixFilePermission> SECURE_FILE_PERMISSIONS = PosixFilePermissions.fromString("rw-------");
    private static final Set<PosixFilePermission> SECURE_DIR_PERMISSIONS = PosixFilePermissions.fromString("rwx------");
    private final Path storageDirectory;
    private final PasswordBasedEncryption encryption;
    private final char[] password;

    public EncryptedFileStorage(char[] password) throws IdentityStorageException {
        this(Paths.get(DEFAULT_BASE_DIR, new String[0]), password);
    }

    public EncryptedFileStorage(Path storageDirectory, char[] password) throws IdentityStorageException {
        this.storageDirectory = storageDirectory;
        this.password = password;
        this.encryption = new PasswordBasedEncryption();
        this.ensureDirectoryExists();
    }

    public void save(String identityId, PrivateKey privateKey) throws IdentityStorageException {
        this.validateIdentityId(identityId);
        try {
            byte[] keyBytes = privateKey.getRawBytes();
            byte[] encryptedData = this.encryption.encrypt(keyBytes, this.password);
            Path keyFile = this.getKeyFilePath(identityId);
            Files.write(keyFile, encryptedData, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            this.setSecureFilePermissions(keyFile);
        }
        catch (IOException e) {
            throw new IdentityStorageException("Failed to save key for identity: " + identityId, (Throwable)e);
        }
    }

    public PrivateKey load(String identityId) throws KeyNotFoundException, IdentityStorageException {
        this.validateIdentityId(identityId);
        Path keyFile = this.getKeyFilePath(identityId);
        if (!Files.exists(keyFile, new LinkOption[0])) {
            throw new KeyNotFoundException("Key not found for identity: " + identityId);
        }
        try {
            byte[] encryptedData = Files.readAllBytes(keyFile);
            byte[] keyBytes = this.encryption.decrypt(encryptedData, this.password);
            return new PrivateKey(keyBytes);
        }
        catch (IOException e) {
            throw new IdentityStorageException("Failed to load key for identity: " + identityId, (Throwable)e);
        }
    }

    public void delete(String identityId) throws IdentityStorageException {
        this.validateIdentityId(identityId);
        Path keyFile = this.getKeyFilePath(identityId);
        try {
            if (Files.exists(keyFile, new LinkOption[0])) {
                Files.delete(keyFile);
            }
        }
        catch (IOException e) {
            throw new IdentityStorageException("Failed to delete key for identity: " + identityId, (Throwable)e);
        }
    }

    public boolean exists(String identityId) {
        this.validateIdentityId(identityId);
        Path keyFile = this.getKeyFilePath(identityId);
        return Files.exists(keyFile, new LinkOption[0]);
    }

    private Path getKeyFilePath(String identityId) {
        return this.storageDirectory.resolve(identityId + FILE_EXTENSION);
    }

    private void ensureDirectoryExists() throws IdentityStorageException {
        try {
            if (!Files.exists(this.storageDirectory, new LinkOption[0])) {
                Files.createDirectories(this.storageDirectory, new FileAttribute[0]);
                try {
                    this.setSecureDirectoryPermissions(this.storageDirectory);
                }
                catch (UnsupportedOperationException unsupportedOperationException) {}
            }
        }
        catch (IOException e) {
            throw new IdentityStorageException("Failed to create storage directory: " + String.valueOf(this.storageDirectory), (Throwable)e);
        }
    }

    private void setSecureDirectoryPermissions(Path path) throws UnsupportedOperationException {
        try {
            if (path.getFileSystem().supportedFileAttributeViews().contains("posix")) {
                Files.setPosixFilePermissions(path, SECURE_DIR_PERMISSIONS);
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private void setSecureFilePermissions(Path path) throws UnsupportedOperationException {
        try {
            if (path.getFileSystem().supportedFileAttributeViews().contains("posix")) {
                Files.setPosixFilePermissions(path, SECURE_FILE_PERMISSIONS);
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private void validateIdentityId(String identityId) {
        if (identityId == null || identityId.trim().isEmpty()) {
            throw new IllegalArgumentException("Identity ID cannot be null or empty");
        }
    }

    public Path getStorageDirectory() {
        return this.storageDirectory;
    }
}

