/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.cashu.mint.admin.application.service;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import xyz.tcheeric.cashu.mint.admin.application.port.in.ManageMintLifecycleUseCase;
import xyz.tcheeric.cashu.mint.admin.application.port.out.ConfigurationSetRepository;
import xyz.tcheeric.cashu.mint.admin.application.port.out.MintLifecycleEvent;
import xyz.tcheeric.cashu.mint.admin.application.port.out.MintLifecycleEventPublisher;
import xyz.tcheeric.cashu.mint.admin.application.port.out.MintRepository;
import xyz.tcheeric.cashu.mint.admin.application.port.out.TransactionManager;
import xyz.tcheeric.cashu.mint.admin.application.service.AbstractUseCaseInteractor;
import xyz.tcheeric.cashu.mint.admin.domain.AuditMetadata;
import xyz.tcheeric.cashu.mint.admin.domain.ConfigurationRevisionId;
import xyz.tcheeric.cashu.mint.admin.domain.ConfigurationSet;
import xyz.tcheeric.cashu.mint.admin.domain.MintAggregate;
import xyz.tcheeric.cashu.mint.admin.domain.MintId;
import xyz.tcheeric.cashu.mint.admin.domain.NotificationPolicy;
import xyz.tcheeric.cashu.mint.admin.domain.OperatorAccount;

public class ManageMintLifecycleInteractor
extends AbstractUseCaseInteractor
implements ManageMintLifecycleUseCase {
    private static final ConfigurationRevisionId INITIAL_REVISION = ConfigurationRevisionId.of(1L);
    private static final String CONFIG_VERSION_PARAMETER = "version.tag";
    private final MintRepository mintRepository;
    private final ConfigurationSetRepository configurationSetRepository;
    private final TransactionManager transactionManager;
    private final MintLifecycleEventPublisher eventPublisher;
    private final Clock clock;

    public ManageMintLifecycleInteractor(MintRepository mintRepository, ConfigurationSetRepository configurationSetRepository, TransactionManager transactionManager, MintLifecycleEventPublisher eventPublisher, Clock clock) {
        this.mintRepository = Objects.requireNonNull(mintRepository, "mint repository must not be null");
        this.configurationSetRepository = Objects.requireNonNull(configurationSetRepository, "configuration repository must not be null");
        this.transactionManager = Objects.requireNonNull(transactionManager, "transaction manager must not be null");
        this.eventPublisher = Objects.requireNonNull(eventPublisher, "event publisher must not be null");
        this.clock = clock == null ? Clock.systemUTC() : clock;
    }

    @Override
    public ManageMintLifecycleUseCase.ManageMintLifecycleResponse handle(ManageMintLifecycleUseCase.ManageMintLifecycleRequest request) {
        ManageMintLifecycleUseCase.ManageMintLifecycleRequest validated = this.requireRequest(request, "manage mint lifecycle request");
        MintId mintId = this.validateMintId(validated.mintId());
        UUID operatorId = this.validateUuid(validated.operatorId(), "operator id");
        if (validated.command() == null) {
            throw new IllegalArgumentException("lifecycle command must not be null");
        }
        ManageMintLifecycleUseCase.LifecycleCommand command = validated.command();
        String versionTag = this.validateVersionTag(validated.versionTag());
        UUID requestId = this.resolveRequestId(validated.requestId());
        String correlationId = this.resolveCorrelationId(validated.correlationId(), requestId);
        return switch (command) {
            default -> throw new MatchException(null, null);
            case ManageMintLifecycleUseCase.LifecycleCommand.CREATE -> this.createMint(mintId, operatorId, versionTag, requestId, correlationId);
            case ManageMintLifecycleUseCase.LifecycleCommand.UPDATE_CONFIGURATION -> this.updateConfiguration(mintId, operatorId, versionTag, requestId, correlationId);
            case ManageMintLifecycleUseCase.LifecycleCommand.PAUSE -> this.pauseMint(mintId, operatorId, versionTag, requestId, correlationId);
            case ManageMintLifecycleUseCase.LifecycleCommand.RESUME -> this.resumeMint(mintId, operatorId, versionTag, requestId, correlationId);
            case ManageMintLifecycleUseCase.LifecycleCommand.RETIRE -> this.retireMint(mintId, operatorId, versionTag, requestId, correlationId);
        };
    }

    private ManageMintLifecycleUseCase.ManageMintLifecycleResponse createMint(MintId mintId, UUID operatorId, String versionTag, UUID requestId, String correlationId) {
        return this.transactionManager.execute(() -> {
            if (this.mintRepository.findById(mintId).isPresent()) {
                throw new IllegalStateException("mint already exists: " + mintId.asString());
            }
            AuditMetadata auditMetadata = this.createAuditMetadata(operatorId, ManageMintLifecycleUseCase.LifecycleCommand.CREATE, requestId, correlationId);
            ConfigurationSet configuration = new ConfigurationSet(INITIAL_REVISION, Map.of(CONFIG_VERSION_PARAMETER, versionTag), auditMetadata);
            OperatorAccount operatorAccount = new OperatorAccount(operatorId, operatorId.toString(), Set.of("MINT_ADMIN"), auditMetadata);
            NotificationPolicy notificationPolicy = new NotificationPolicy(true, false, Duration.ofMinutes(5L), auditMetadata);
            MintAggregate aggregate = MintAggregate.create(mintId, configuration, operatorAccount, notificationPolicy, auditMetadata);
            this.mintRepository.save(aggregate);
            this.configurationSetRepository.save(mintId, configuration);
            this.eventPublisher.publish(MintLifecycleEvent.created(mintId, aggregate.lifecycleState().value(), configuration.revisionId(), versionTag, aggregate.auditMetadata()));
            return this.buildResponse(aggregate, versionTag);
        });
    }

    private ManageMintLifecycleUseCase.ManageMintLifecycleResponse updateConfiguration(MintId mintId, UUID operatorId, String versionTag, UUID requestId, String correlationId) {
        return this.transactionManager.execute(() -> {
            MintAggregate current = this.loadExistingAggregate(mintId);
            AuditMetadata auditMetadata = this.createAuditMetadata(operatorId, ManageMintLifecycleUseCase.LifecycleCommand.UPDATE_CONFIGURATION, requestId, correlationId);
            ConfigurationRevisionId nextRevision = current.configurationSet().revisionId().next();
            ConfigurationSet updatedConfiguration = current.configurationSet().updateParameter(CONFIG_VERSION_PARAMETER, versionTag, nextRevision, auditMetadata);
            MintAggregate updated = current.updateConfiguration(updatedConfiguration, auditMetadata);
            this.mintRepository.save(updated);
            this.configurationSetRepository.save(mintId, updatedConfiguration);
            this.eventPublisher.publish(MintLifecycleEvent.configurationUpdated(mintId, updated.lifecycleState().value(), updatedConfiguration.revisionId(), versionTag, updated.auditMetadata()));
            return this.buildResponse(updated, versionTag);
        });
    }

    private ManageMintLifecycleUseCase.ManageMintLifecycleResponse pauseMint(MintId mintId, UUID operatorId, String versionTag, UUID requestId, String correlationId) {
        return this.transactionManager.execute(() -> {
            MintAggregate current = this.loadExistingAggregate(mintId);
            AuditMetadata auditMetadata = this.createAuditMetadata(operatorId, ManageMintLifecycleUseCase.LifecycleCommand.PAUSE, requestId, correlationId);
            MintAggregate suspended = current.suspend(auditMetadata);
            this.mintRepository.save(suspended);
            this.eventPublisher.publish(MintLifecycleEvent.paused(mintId, current.lifecycleState().value(), suspended.lifecycleState().value(), suspended.configurationSet().revisionId(), versionTag, suspended.auditMetadata()));
            return this.buildResponse(suspended, versionTag);
        });
    }

    private ManageMintLifecycleUseCase.ManageMintLifecycleResponse resumeMint(MintId mintId, UUID operatorId, String versionTag, UUID requestId, String correlationId) {
        return this.transactionManager.execute(() -> {
            MintAggregate current = this.loadExistingAggregate(mintId);
            AuditMetadata auditMetadata = this.createAuditMetadata(operatorId, ManageMintLifecycleUseCase.LifecycleCommand.RESUME, requestId, correlationId);
            MintAggregate activated = current.activate(auditMetadata);
            this.mintRepository.save(activated);
            this.eventPublisher.publish(MintLifecycleEvent.resumed(mintId, current.lifecycleState().value(), activated.lifecycleState().value(), activated.configurationSet().revisionId(), versionTag, activated.auditMetadata()));
            return this.buildResponse(activated, versionTag);
        });
    }

    private ManageMintLifecycleUseCase.ManageMintLifecycleResponse retireMint(MintId mintId, UUID operatorId, String versionTag, UUID requestId, String correlationId) {
        return this.transactionManager.execute(() -> {
            MintAggregate current = this.loadExistingAggregate(mintId);
            AuditMetadata auditMetadata = this.createAuditMetadata(operatorId, ManageMintLifecycleUseCase.LifecycleCommand.RETIRE, requestId, correlationId);
            MintAggregate retired = current.decommission(auditMetadata);
            this.mintRepository.save(retired);
            this.eventPublisher.publish(MintLifecycleEvent.retired(mintId, current.lifecycleState().value(), retired.lifecycleState().value(), retired.configurationSet().revisionId(), versionTag, retired.auditMetadata()));
            return this.buildResponse(retired, versionTag);
        });
    }

    private MintAggregate loadExistingAggregate(MintId mintId) {
        return this.mintRepository.findById(mintId).orElseThrow(() -> new IllegalStateException("mint not found: " + mintId.asString()));
    }

    private AuditMetadata createAuditMetadata(UUID operatorId, ManageMintLifecycleUseCase.LifecycleCommand command, UUID requestId, String correlationId) {
        Instant timestamp = this.clock.instant();
        String actor = operatorId.toString();
        String action = switch (command) {
            default -> throw new MatchException(null, null);
            case ManageMintLifecycleUseCase.LifecycleCommand.CREATE -> "Mint created";
            case ManageMintLifecycleUseCase.LifecycleCommand.UPDATE_CONFIGURATION -> "Mint configuration updated";
            case ManageMintLifecycleUseCase.LifecycleCommand.PAUSE -> "Mint paused";
            case ManageMintLifecycleUseCase.LifecycleCommand.RESUME -> "Mint resumed";
            case ManageMintLifecycleUseCase.LifecycleCommand.RETIRE -> "Mint retired";
        };
        return new AuditMetadata(actor, action, timestamp, requestId, correlationId);
    }

    private UUID resolveRequestId(String requestId) {
        if (requestId == null || requestId.isBlank()) {
            return UUID.randomUUID();
        }
        return this.validateUuid(requestId, "request id");
    }

    private String resolveCorrelationId(String correlationId, UUID requestId) {
        if (correlationId == null || correlationId.isBlank()) {
            return requestId.toString();
        }
        String sanitized = correlationId.trim();
        if (sanitized.isEmpty()) {
            throw new IllegalArgumentException("correlation id must not be blank");
        }
        return sanitized;
    }

    private ManageMintLifecycleUseCase.ManageMintLifecycleResponse buildResponse(MintAggregate aggregate, String versionTag) {
        return new ManageMintLifecycleUseCase.ManageMintLifecycleResponse(aggregate.mintId().asString(), aggregate.lifecycleState().value(), versionTag);
    }
}

