/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.annotations.VisibleForTesting;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.stream.IntStream;
import javax.annotation.Nullable;
import org.bitcoinj.base.Address;
import org.bitcoinj.base.Coin;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.VarInt;
import org.bitcoinj.base.internal.Buffers;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.InternalUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.base.internal.Stopwatch;
import org.bitcoinj.base.internal.StreamUtils;
import org.bitcoinj.base.internal.TimeUtils;
import org.bitcoinj.core.BaseMessage;
import org.bitcoinj.core.Context;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Transaction;
import org.bitcoinj.core.TransactionInput;
import org.bitcoinj.core.TransactionOutPoint;
import org.bitcoinj.core.TransactionOutput;
import org.bitcoinj.core.TransactionWitness;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.script.Script;
import org.bitcoinj.script.ScriptBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Block
extends BaseMessage {
    private static final Logger log = LoggerFactory.getLogger(Block.class);
    public static final int HEADER_SIZE = 80;
    static final Duration ALLOWED_TIME_DRIFT = Duration.ofHours(2L);
    public static final int MAX_BLOCK_SIZE = 1000000;
    public static final int MAX_BLOCK_SIGOPS = 20000;
    public static final long STANDARD_MAX_DIFFICULTY_TARGET = 486604799L;
    public static final long EASIEST_DIFFICULTY_TARGET = 545259519L;
    public static final int BLOCK_HEIGHT_UNKNOWN = -1;
    public static final int BLOCK_HEIGHT_GENESIS = 0;
    public static final long BLOCK_VERSION_GENESIS = 1L;
    public static final long BLOCK_VERSION_BIP34 = 2L;
    public static final long BLOCK_VERSION_BIP66 = 3L;
    public static final long BLOCK_VERSION_BIP65 = 4L;
    private final long version;
    private Sha256Hash prevBlockHash;
    private Sha256Hash merkleRoot;
    private Sha256Hash witnessRoot;
    private Instant time;
    private long difficultyTarget;
    private long nonce;
    @Nullable
    List<Transaction> transactions;
    private Sha256Hash hash;
    private static final byte[] genesisTxInputScriptBytes = ByteUtils.parseHex("04ffff001d0104455468652054696d65732030332f4a616e2f32303039204368616e63656c6c6f72206f6e206272696e6b206f66207365636f6e64206261696c6f757420666f722062616e6b73");
    private static final byte[] genesisTxScriptPubKeyBytes = new ScriptBuilder().data(ByteUtils.parseHex("04678afdb0fe5548271967f1a67130b7105cd6a828e03909a67962e0ea1f61deb649f6bc3f4cef38c4f35504e51ec112de5c384df7ba0b8d578a4c702b6bf11d5f")).op(172).build().program();
    private static BigInteger LARGEST_HASH = BigInteger.ONE.shiftLeft(256);
    private static int txCounter;
    private static final byte[] EMPTY_BYTES;
    private static final byte[] pubkeyForTesting;

    public static Block read(ByteBuffer payload) throws BufferUnderflowException, ProtocolException {
        payload.mark();
        long version = ByteUtils.readUint32(payload);
        Sha256Hash prevBlockHash = Sha256Hash.read(payload);
        Sha256Hash merkleRoot = Sha256Hash.read(payload);
        Instant time = Instant.ofEpochSecond(ByteUtils.readUint32(payload));
        long difficultyTarget = ByteUtils.readUint32(payload);
        long nonce = ByteUtils.readUint32(payload);
        payload.reset();
        Sha256Hash hash = Sha256Hash.wrapReversed(Sha256Hash.hashTwice(Buffers.readBytes(payload, 80)));
        List<Transaction> transactions = payload.hasRemaining() ? Block.readTransactions(payload) : null;
        Block block = new Block(version, prevBlockHash, merkleRoot, time, difficultyTarget, nonce, transactions);
        block.hash = hash;
        return block;
    }

    private static List<Transaction> readTransactions(ByteBuffer payload) throws BufferUnderflowException, ProtocolException {
        VarInt numTransactions = VarInt.read(payload);
        Preconditions.check(numTransactions.fitsInt(), BufferUnderflowException::new);
        return IntStream.range(0, numTransactions.intValue()).mapToObj(i2 -> Transaction.read(payload)).collect(StreamUtils.toUnmodifiableList());
    }

    Block(long setVersion) {
        this(setVersion, TimeUtils.currentTime().truncatedTo(ChronoUnit.SECONDS), 487063544L, 0L, Collections.emptyList());
    }

    Block(long setVersion, Instant time, long difficultyTarget, List<Transaction> transactions) {
        this(setVersion, time, difficultyTarget, 0L, transactions);
    }

    Block(long setVersion, Instant time, long difficultyTarget, long nonce, List<Transaction> transactions) {
        this.version = setVersion;
        this.time = time;
        this.difficultyTarget = difficultyTarget;
        this.nonce = nonce;
        this.prevBlockHash = Sha256Hash.ZERO_HASH;
        this.transactions = new ArrayList<Transaction>((Collection)Objects.requireNonNull(transactions));
    }

    public Block(long version, Sha256Hash prevBlockHash, Sha256Hash merkleRoot, Instant time, long difficultyTarget, long nonce, @Nullable List<Transaction> transactions) {
        this.version = version;
        this.prevBlockHash = prevBlockHash;
        this.merkleRoot = merkleRoot;
        this.time = time;
        this.difficultyTarget = difficultyTarget;
        this.nonce = nonce;
        this.transactions = transactions != null ? new ArrayList<Transaction>(transactions) : null;
    }

    @Deprecated
    public Block(long version, Sha256Hash prevBlockHash, Sha256Hash merkleRoot, long time, long difficultyTarget, long nonce, @Nullable List<Transaction> transactions) {
        this(version, prevBlockHash, merkleRoot, Instant.ofEpochSecond(time), difficultyTarget, nonce, transactions);
    }

    public static Block createGenesis(Instant time, long difficultyTarget) {
        return new Block(1L, time, difficultyTarget, Block.genesisTransactions());
    }

    public static Block createGenesis(Instant time, long difficultyTarget, long nonce) {
        return new Block(1L, time, difficultyTarget, nonce, Block.genesisTransactions());
    }

    private static List<Transaction> genesisTransactions() {
        Transaction tx = Transaction.coinbase(genesisTxInputScriptBytes);
        tx.addOutput(new TransactionOutput(tx, Coin.FIFTY_COINS, genesisTxScriptPubKeyBytes));
        return Collections.singletonList(tx);
    }

    @Override
    public int messageSize() {
        int size = 80;
        List<Transaction> transactions = this.getTransactions();
        if (transactions != null) {
            size += VarInt.sizeOf(transactions.size());
            for (Transaction tx : transactions) {
                size += tx.messageSize();
            }
        }
        return size;
    }

    void writeHeader(OutputStream stream) throws IOException {
        ByteUtils.writeInt32LE(this.version, stream);
        stream.write(this.prevBlockHash.serialize());
        stream.write(this.getMerkleRoot().serialize());
        ByteUtils.writeInt32LE(this.time.getEpochSecond(), stream);
        ByteUtils.writeInt32LE(this.difficultyTarget, stream);
        ByteUtils.writeInt32LE(this.nonce, stream);
    }

    private void writeTransactions(OutputStream stream) throws IOException {
        if (this.transactions == null) {
            return;
        }
        stream.write(VarInt.of(this.transactions.size()).serialize());
        for (Transaction tx : this.transactions) {
            tx.bitcoinSerializeToStream(stream);
        }
    }

    @Override
    protected void bitcoinSerializeToStream(OutputStream stream) throws IOException {
        this.writeHeader(stream);
        this.writeTransactions(stream);
    }

    protected void unCache() {
        this.unCacheTransactions();
    }

    private void unCacheHeader() {
        this.hash = null;
    }

    private void unCacheTransactions() {
        this.unCacheHeader();
        this.merkleRoot = null;
    }

    private Sha256Hash calculateHash() {
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream(80);
            this.writeHeader(bos);
            return Sha256Hash.wrapReversed(Sha256Hash.hashTwice(bos.toByteArray()));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public String getHashAsString() {
        return this.getHash().toString();
    }

    public Sha256Hash getHash() {
        if (this.hash == null) {
            this.hash = this.calculateHash();
        }
        return this.hash;
    }

    public BigInteger getWork() throws VerificationException {
        BigInteger target = this.getDifficultyTargetAsInteger();
        return LARGEST_HASH.divide(target.add(BigInteger.ONE));
    }

    public Block cloneAsHeader() {
        Block block = new Block(this.version, this.prevBlockHash, this.getMerkleRoot(), this.time, this.difficultyTarget, this.nonce, null);
        block.hash = this.getHash();
        return block;
    }

    public String toString() {
        StringBuilder s = new StringBuilder();
        s.append(" block: \n");
        s.append("   hash: ").append(this.getHashAsString()).append('\n');
        s.append("   version: ").append(this.version);
        String bips = InternalUtils.commaJoin(this.isBIP34() ? "BIP34" : null, this.isBIP66() ? "BIP66" : null, this.isBIP65() ? "BIP65" : null);
        if (!bips.isEmpty()) {
            s.append(" (").append(bips).append(')');
        }
        s.append('\n');
        s.append("   previous block: ").append(this.getPrevBlockHash()).append("\n");
        s.append("   time: ").append(this.time).append(" (").append(TimeUtils.dateTimeFormat(this.time)).append(")\n");
        s.append("   difficulty target (nBits): ").append(this.difficultyTarget).append("\n");
        s.append("   nonce: ").append(this.nonce).append("\n");
        if (this.transactions != null && this.transactions.size() > 0) {
            s.append("   merkle root: ").append(this.getMerkleRoot()).append("\n");
            s.append("   witness root: ").append(this.getWitnessRoot()).append("\n");
            s.append("   with ").append(this.transactions.size()).append(" transaction(s):\n");
            for (Transaction tx : this.transactions) {
                s.append(tx).append('\n');
            }
        }
        return s.toString();
    }

    @VisibleForTesting
    public void solve() {
        Duration warningThreshold = Duration.ofSeconds(5L);
        Stopwatch watch = Stopwatch.start();
        try {
            while (true) {
                if (this.checkProofOfWork(false)) {
                    return;
                }
                this.setNonce(this.getNonce() + 1L);
                if (!watch.isRunning() || watch.elapsed().compareTo(warningThreshold) <= 0) continue;
                watch.stop();
                log.warn("trying to solve block for longer than {} seconds", (Object)warningThreshold.getSeconds());
            }
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
    }

    public BigInteger getDifficultyTargetAsInteger() {
        return ByteUtils.decodeCompactBits(this.difficultyTarget);
    }

    protected boolean checkProofOfWork(boolean throwException) throws VerificationException {
        if (Context.get().isRelaxProofOfWork()) {
            return true;
        }
        BigInteger target = this.getDifficultyTargetAsInteger();
        BigInteger h = this.getHash().toBigInteger();
        if (h.compareTo(target) > 0) {
            if (throwException) {
                throw new VerificationException("Hash is higher than target: " + this.getHashAsString() + " vs " + target.toString(16));
            }
            return false;
        }
        return true;
    }

    private void checkTimestamp() throws VerificationException {
        Instant allowedTime = TimeUtils.currentTime().truncatedTo(ChronoUnit.SECONDS).plus(ALLOWED_TIME_DRIFT);
        if (this.time.isAfter(allowedTime)) {
            throw new VerificationException(String.format(Locale.US, "Block too far in future: %s (%d) vs allowed %s (%d)", TimeUtils.dateTimeFormat(this.time), this.time.toEpochMilli(), TimeUtils.dateTimeFormat(allowedTime), allowedTime.toEpochMilli()));
        }
    }

    private void checkSigOps() throws VerificationException {
        int sigOps = 0;
        for (Transaction tx : this.transactions) {
            sigOps += tx.getSigOpCount();
        }
        if (sigOps > 20000) {
            throw new VerificationException("Block had too many Signature Operations");
        }
    }

    private void checkMerkleRoot() throws VerificationException {
        Sha256Hash calculatedRoot = this.calculateMerkleRoot();
        if (!calculatedRoot.equals(this.merkleRoot)) {
            log.error("Merkle tree did not verify");
            throw new VerificationException("Merkle hashes do not match: " + calculatedRoot + " vs " + this.merkleRoot);
        }
    }

    void checkWitnessRoot() throws VerificationException {
        Transaction coinbase = this.transactions.get(0);
        Preconditions.checkState(coinbase.isCoinBase());
        Sha256Hash witnessCommitment = coinbase.findWitnessCommitment();
        if (witnessCommitment != null) {
            byte[] witnessReserved = null;
            TransactionWitness witness = coinbase.getInput(0L).getWitness();
            if (witness.getPushCount() != 1) {
                throw new VerificationException("Coinbase witness reserved invalid: push count");
            }
            witnessReserved = witness.getPush(0);
            if (witnessReserved.length != 32) {
                throw new VerificationException("Coinbase witness reserved invalid: length");
            }
            Sha256Hash witnessRootHash = Sha256Hash.twiceOf(this.getWitnessRoot().serialize(), witnessReserved);
            if (!witnessRootHash.equals(witnessCommitment)) {
                throw new VerificationException("Witness merkle root invalid. Expected " + witnessCommitment.toString() + " but got " + witnessRootHash.toString());
            }
        } else {
            for (Transaction tx : this.transactions) {
                if (!tx.hasWitnesses()) continue;
                throw new VerificationException("Transaction witness found but no witness commitment present");
            }
        }
    }

    private Sha256Hash calculateMerkleRoot() {
        List<Sha256Hash> tree = this.buildMerkleTree(false);
        return tree.get(tree.size() - 1);
    }

    private Sha256Hash calculateWitnessRoot() {
        List<Sha256Hash> tree = this.buildMerkleTree(true);
        return tree.get(tree.size() - 1);
    }

    private List<Sha256Hash> buildMerkleTree(boolean useWTxId) {
        ArrayList<Sha256Hash> tree = new ArrayList<Sha256Hash>(this.transactions.size());
        for (Transaction tx : this.transactions) {
            Sha256Hash hash = useWTxId && tx.isCoinBase() ? Sha256Hash.ZERO_HASH : (useWTxId ? tx.getWTxId() : tx.getTxId());
            tree.add(hash);
        }
        int levelOffset = 0;
        int levelSize = this.transactions.size();
        while (levelSize > 1) {
            for (int left = 0; left < levelSize; left += 2) {
                int right = Math.min(left + 1, levelSize - 1);
                Sha256Hash leftHash = tree.get(levelOffset + left);
                Sha256Hash rightHash = tree.get(levelOffset + right);
                tree.add(Sha256Hash.wrapReversed(Sha256Hash.hashTwice(leftHash.serialize(), rightHash.serialize())));
            }
            levelOffset += levelSize;
            levelSize = (levelSize + 1) / 2;
        }
        return tree;
    }

    private void checkTransactions(int height, EnumSet<VerifyFlag> flags) throws VerificationException {
        if (!this.transactions.get(0).isCoinBase()) {
            throw new VerificationException("First tx is not coinbase");
        }
        if (flags.contains((Object)VerifyFlag.HEIGHT_IN_COINBASE) && height >= 0) {
            this.transactions.get(0).checkCoinBaseHeight(height);
        }
        for (int i2 = 1; i2 < this.transactions.size(); ++i2) {
            if (!this.transactions.get(i2).isCoinBase()) continue;
            throw new VerificationException("TX " + i2 + " is coinbase when it should not be.");
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        return this.getHash().equals(((Block)o).getHash());
    }

    public int hashCode() {
        return this.getHash().hashCode();
    }

    public Sha256Hash getMerkleRoot() {
        if (this.merkleRoot == null) {
            this.unCacheHeader();
            this.merkleRoot = this.calculateMerkleRoot();
        }
        return this.merkleRoot;
    }

    void setMerkleRoot(Sha256Hash value) {
        this.unCacheHeader();
        this.merkleRoot = value;
        this.hash = null;
    }

    public Sha256Hash getWitnessRoot() {
        if (this.witnessRoot == null) {
            this.witnessRoot = this.calculateWitnessRoot();
        }
        return this.witnessRoot;
    }

    public void addTransaction(Transaction t) {
        this.addTransaction(t, true);
    }

    void addTransaction(Transaction t, boolean runSanityChecks) {
        this.unCacheTransactions();
        if (this.transactions == null) {
            this.transactions = new ArrayList<Transaction>();
        }
        if (runSanityChecks && this.transactions.size() == 0 && !t.isCoinBase()) {
            throw new RuntimeException("Attempted to add a non-coinbase transaction as the first transaction: " + t);
        }
        if (runSanityChecks && this.transactions.size() > 0 && t.isCoinBase()) {
            throw new RuntimeException("Attempted to add a coinbase transaction when there already is one: " + t);
        }
        this.transactions.add(t);
        this.merkleRoot = null;
        this.hash = null;
    }

    public long getVersion() {
        return this.version;
    }

    public Sha256Hash getPrevBlockHash() {
        return this.prevBlockHash;
    }

    void setPrevBlockHash(Sha256Hash prevBlockHash) {
        this.unCacheHeader();
        this.prevBlockHash = prevBlockHash;
        this.hash = null;
    }

    public Instant time() {
        return this.time;
    }

    @Deprecated
    public long getTimeSeconds() {
        return this.time.getEpochSecond();
    }

    @Deprecated
    public Date getTime() {
        return Date.from(this.time());
    }

    void setTime(Instant time) {
        this.unCacheHeader();
        this.time = time.truncatedTo(ChronoUnit.SECONDS);
        this.hash = null;
    }

    public long getDifficultyTarget() {
        return this.difficultyTarget;
    }

    void setDifficultyTarget(long compactForm) {
        this.unCacheHeader();
        this.difficultyTarget = compactForm;
        this.hash = null;
    }

    public long getNonce() {
        return this.nonce;
    }

    void setNonce(long nonce) {
        this.unCacheHeader();
        this.nonce = nonce;
        this.hash = null;
    }

    @Nullable
    public List<Transaction> getTransactions() {
        return this.transactions == null ? null : Collections.unmodifiableList(this.transactions);
    }

    void addCoinbaseTransaction(byte[] pubKeyTo, Coin value, int height) {
        this.unCacheTransactions();
        this.transactions = new ArrayList<Transaction>();
        Transaction coinbase = new Transaction();
        ScriptBuilder inputBuilder = new ScriptBuilder();
        if (height >= 0) {
            inputBuilder.number(height);
        }
        inputBuilder.data(new byte[]{(byte)txCounter, (byte)(txCounter++ >> 8)});
        coinbase.addInput(TransactionInput.coinbaseInput(coinbase, inputBuilder.build().program()));
        coinbase.addOutput(new TransactionOutput(coinbase, value, ScriptBuilder.createP2PKOutputScript(ECKey.fromPublicOnly(pubKeyTo)).program()));
        this.transactions.add(coinbase);
    }

    @VisibleForTesting
    public Block createNextBlock(@Nullable Address to, long version, Instant time, int height) {
        return this.createNextBlock(to, version, null, time, pubkeyForTesting, Coin.FIFTY_COINS, height);
    }

    Block createNextBlock(@Nullable Address to, long version, @Nullable TransactionOutPoint prevOut, Instant time, byte[] pubKey, Coin coinbaseValue, int height) {
        Block b = new Block(version);
        b.setDifficultyTarget(this.difficultyTarget);
        b.addCoinbaseTransaction(pubKey, coinbaseValue, height);
        if (to != null) {
            Transaction t = new Transaction();
            t.addOutput(new TransactionOutput(t, Coin.FIFTY_COINS, to));
            if (prevOut == null) {
                prevOut = new TransactionOutPoint(0L, this.nextTestOutPointHash());
            }
            TransactionInput input = new TransactionInput(t, Script.createInputScript(EMPTY_BYTES, EMPTY_BYTES), prevOut);
            t.addInput(input);
            b.addTransaction(t);
        }
        b.setPrevBlockHash(this.getHash());
        Instant bitcoinTime = time.truncatedTo(ChronoUnit.SECONDS);
        if (this.time().compareTo(bitcoinTime) >= 0) {
            b.setTime(this.time().plusSeconds(1L));
        } else {
            b.setTime(bitcoinTime);
        }
        b.solve();
        try {
            Block.verifyHeader(b);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
        if (b.getVersion() != version) {
            throw new RuntimeException();
        }
        return b;
    }

    private Sha256Hash nextTestOutPointHash() {
        byte[] counter = new byte[32];
        counter[0] = (byte)txCounter;
        counter[1] = (byte)(txCounter++ >> 8);
        return Sha256Hash.wrap(counter);
    }

    @VisibleForTesting
    public Block createNextBlock(@Nullable Address to, TransactionOutPoint prevOut) {
        return this.createNextBlock(to, 1L, prevOut, this.time().plusSeconds(5L), pubkeyForTesting, Coin.FIFTY_COINS, -1);
    }

    @VisibleForTesting
    public Block createNextBlock(@Nullable Address to, Coin coinbaseValue) {
        return this.createNextBlock(to, 1L, null, this.time().plusSeconds(5L), pubkeyForTesting, coinbaseValue, -1);
    }

    @VisibleForTesting
    public Block createNextBlock(@Nullable Address to) {
        return this.createNextBlock(to, Coin.FIFTY_COINS);
    }

    @VisibleForTesting
    public Block createNextBlockWithCoinbase(long version, byte[] pubKey, Coin coinbaseValue, int height) {
        return this.createNextBlock(null, version, null, TimeUtils.currentTime(), pubKey, coinbaseValue, height);
    }

    Block createNextBlockWithCoinbase(long version, byte[] pubKey, int height) {
        return this.createNextBlock(null, version, null, TimeUtils.currentTime(), pubKey, Coin.FIFTY_COINS, height);
    }

    public boolean hasTransactions() {
        return this.transactions != null && !this.transactions.isEmpty();
    }

    public boolean isBIP34() {
        return this.version >= 2L;
    }

    public boolean isBIP66() {
        return this.version >= 3L;
    }

    public boolean isBIP65() {
        return this.version >= 4L;
    }

    public static void verify(NetworkParameters params, Block block, int height, EnumSet<VerifyFlag> flags) throws VerificationException {
        Block.verifyHeader(block);
        Block.verifyTransactions(params, block, height, flags);
    }

    public static void verifyHeader(Block block) throws VerificationException {
        block.checkProofOfWork(true);
        block.checkTimestamp();
    }

    public static void verifyTransactions(NetworkParameters params, Block block, int height, EnumSet<VerifyFlag> flags) throws VerificationException {
        if (block.transactions.isEmpty()) {
            throw new VerificationException("Block had no transactions");
        }
        if (block.messageSize() > 1000000) {
            throw new VerificationException("Block larger than MAX_BLOCK_SIZE");
        }
        block.checkTransactions(height, flags);
        block.checkMerkleRoot();
        block.checkSigOps();
        for (Transaction tx : block.transactions) {
            Transaction.verify(params.network(), tx);
        }
    }

    static {
        EMPTY_BYTES = new byte[32];
        pubkeyForTesting = new ECKey().getPubKey();
    }

    public static enum VerifyFlag {
        HEIGHT_IN_COINBASE;

    }
}

