/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.base.MoreObjects;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.VarInt;
import org.bitcoinj.base.internal.Buffers;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.core.BaseMessage;
import org.bitcoinj.core.Block;
import org.bitcoinj.core.FilteredBlock;
import org.bitcoinj.core.PartialMerkleTree;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Transaction;
import org.bitcoinj.core.TransactionInput;
import org.bitcoinj.core.TransactionOutPoint;
import org.bitcoinj.core.TransactionOutput;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.script.Script;
import org.bitcoinj.script.ScriptChunk;
import org.bitcoinj.script.ScriptPattern;

public class BloomFilter
extends BaseMessage {
    private byte[] data;
    private final long hashFuncs;
    private final int nTweak;
    private final byte nFlags;
    private static final long MAX_FILTER_SIZE = 36000L;
    private static final int MAX_HASH_FUNCS = 50;

    public static BloomFilter read(ByteBuffer payload) throws BufferUnderflowException, ProtocolException {
        byte[] data = Buffers.readLengthPrefixedBytes(payload);
        if ((long)data.length > 36000L) {
            throw new ProtocolException("Bloom filter out of size range.");
        }
        long hashFuncs = ByteUtils.readUint32(payload);
        if (hashFuncs > 50L) {
            throw new ProtocolException("Bloom filter hash function count out of range");
        }
        int nTweak = ByteUtils.readInt32(payload);
        byte nFlags = payload.get();
        return new BloomFilter(data, hashFuncs, nTweak, nFlags);
    }

    public BloomFilter(int elements, double falsePositiveRate, int randomNonce) {
        this(elements, falsePositiveRate, randomNonce, BloomUpdate.UPDATE_P2PUBKEY_ONLY);
    }

    public BloomFilter(int elements, double falsePositiveRate, int randomNonce, BloomUpdate updateFlag) {
        int size = (int)(-1.0 / Math.pow(Math.log(2.0), 2.0) * (double)elements * Math.log(falsePositiveRate));
        size = Math.max(1, Math.min(size, 288000) / 8);
        this.data = new byte[size];
        long numHashFuncs = (int)((double)(this.data.length * 8) / (double)elements * Math.log(2.0));
        this.hashFuncs = Math.max(1L, Math.min(numHashFuncs, 50L));
        this.nTweak = randomNonce;
        this.nFlags = (byte)(0xFF & updateFlag.ordinal());
    }

    private BloomFilter(byte[] data, long hashFuncs, int nTweak, byte nFlags) {
        this.data = data;
        this.hashFuncs = hashFuncs;
        this.nTweak = nTweak;
        this.nFlags = nFlags;
    }

    public double getFalsePositiveRate(int elements) {
        return Math.pow(1.0 - Math.pow(Math.E, -1.0 * (double)(this.hashFuncs * (long)elements) / (double)(this.data.length * 8)), this.hashFuncs);
    }

    public String toString() {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this).omitNullValues();
        helper.add("data length", this.data.length);
        helper.add("hashFuncs", this.hashFuncs);
        helper.add("nFlags", (Object)this.getUpdateFlag());
        return helper.toString();
    }

    @Override
    protected void bitcoinSerializeToStream(OutputStream stream) throws IOException {
        stream.write(VarInt.of(this.data.length).serialize());
        stream.write(this.data);
        ByteUtils.writeInt32LE(this.hashFuncs, stream);
        ByteUtils.writeInt32LE(this.nTweak, stream);
        stream.write(this.nFlags);
    }

    private static int rotateLeft32(int x, int r) {
        return x << r | x >>> 32 - r;
    }

    public static int murmurHash3(byte[] data, long nTweak, int hashNum, byte[] object) {
        int h1 = (int)((long)hashNum * 4221880213L + nTweak);
        int c1 = -862048943;
        int c2 = 461845907;
        int numBlocks = object.length / 4 * 4;
        for (int i2 = 0; i2 < numBlocks; i2 += 4) {
            int k1 = object[i2] & 0xFF | (object[i2 + 1] & 0xFF) << 8 | (object[i2 + 2] & 0xFF) << 16 | (object[i2 + 3] & 0xFF) << 24;
            k1 *= -862048943;
            k1 = BloomFilter.rotateLeft32(k1, 15);
            h1 ^= (k1 *= 461845907);
            h1 = BloomFilter.rotateLeft32(h1, 13);
            h1 = h1 * 5 + -430675100;
        }
        int k1 = 0;
        switch (object.length & 3) {
            case 3: {
                k1 ^= (object[numBlocks + 2] & 0xFF) << 16;
            }
            case 2: {
                k1 ^= (object[numBlocks + 1] & 0xFF) << 8;
            }
            case 1: {
                k1 ^= object[numBlocks] & 0xFF;
                k1 *= -862048943;
                k1 = BloomFilter.rotateLeft32(k1, 15);
                h1 ^= (k1 *= 461845907);
            }
        }
        h1 ^= object.length;
        h1 ^= h1 >>> 16;
        h1 *= -2048144789;
        h1 ^= h1 >>> 13;
        h1 *= -1028477387;
        h1 ^= h1 >>> 16;
        return (int)(((long)h1 & 0xFFFFFFFFL) % (long)(data.length * 8));
    }

    public synchronized boolean contains(byte[] object) {
        int i2 = 0;
        while ((long)i2 < this.hashFuncs) {
            if (!ByteUtils.checkBitLE(this.data, BloomFilter.murmurHash3(this.data, this.nTweak, i2, object))) {
                return false;
            }
            ++i2;
        }
        return true;
    }

    public synchronized void insert(byte[] object) {
        int i2 = 0;
        while ((long)i2 < this.hashFuncs) {
            ByteUtils.setBitLE(this.data, BloomFilter.murmurHash3(this.data, this.nTweak, i2, object));
            ++i2;
        }
    }

    public synchronized void insert(ECKey key) {
        this.insert(key.getPubKey());
        this.insert(key.getPubKeyHash());
    }

    public synchronized void insert(TransactionOutPoint outpoint) {
        this.insert(outpoint.serialize());
    }

    public synchronized void setMatchAll() {
        this.data = new byte[]{-1};
    }

    public synchronized void merge(BloomFilter filter2) {
        if (!this.matchesAll() && !filter2.matchesAll()) {
            Preconditions.checkArgument(filter2.data.length == this.data.length && filter2.hashFuncs == this.hashFuncs && filter2.nTweak == this.nTweak);
            for (int i2 = 0; i2 < this.data.length; ++i2) {
                int n = i2;
                this.data[n] = (byte)(this.data[n] | filter2.data[i2]);
            }
        } else {
            this.data = new byte[]{-1};
        }
    }

    public synchronized boolean matchesAll() {
        for (byte b : this.data) {
            if (b == -1) continue;
            return false;
        }
        return true;
    }

    public synchronized BloomUpdate getUpdateFlag() {
        if (this.nFlags == 0) {
            return BloomUpdate.UPDATE_NONE;
        }
        if (this.nFlags == 1) {
            return BloomUpdate.UPDATE_ALL;
        }
        if (this.nFlags == 2) {
            return BloomUpdate.UPDATE_P2PUBKEY_ONLY;
        }
        throw new IllegalStateException("Unknown flag combination");
    }

    public synchronized FilteredBlock applyAndUpdate(Block block) {
        List<Transaction> txns = block.getTransactions();
        ArrayList<Sha256Hash> txHashes = new ArrayList<Sha256Hash>(txns.size());
        ArrayList<Transaction> matched = new ArrayList<Transaction>();
        byte[] bits = new byte[(int)Math.ceil((double)txns.size() / 8.0)];
        for (int i2 = 0; i2 < txns.size(); ++i2) {
            Transaction tx = txns.get(i2);
            txHashes.add(tx.getTxId());
            if (!this.applyAndUpdate(tx)) continue;
            ByteUtils.setBitLE(bits, i2);
            matched.add(tx);
        }
        PartialMerkleTree pmt = PartialMerkleTree.buildFromLeaves(bits, txHashes);
        FilteredBlock filteredBlock = new FilteredBlock(block.cloneAsHeader(), pmt);
        for (Transaction transaction : matched) {
            filteredBlock.provideTransaction(transaction);
        }
        return filteredBlock;
    }

    public synchronized boolean applyAndUpdate(Transaction tx) {
        if (this.contains(tx.getTxId().getBytes())) {
            return true;
        }
        boolean found = false;
        BloomUpdate flag = this.getUpdateFlag();
        for (TransactionOutput output : tx.getOutputs()) {
            Script script = output.getScriptPubKey();
            for (ScriptChunk chunk : script.chunks()) {
                boolean isSendingToPubKeys;
                if (!chunk.isPushData() || !this.contains(chunk.data)) continue;
                boolean bl = isSendingToPubKeys = ScriptPattern.isP2PK(script) || ScriptPattern.isSentToMultisig(script);
                if (flag == BloomUpdate.UPDATE_ALL || flag == BloomUpdate.UPDATE_P2PUBKEY_ONLY && isSendingToPubKeys) {
                    this.insert(output.getOutPointFor());
                }
                found = true;
            }
        }
        if (found) {
            return true;
        }
        for (TransactionInput input : tx.getInputs()) {
            if (this.contains(input.getOutpoint().serialize())) {
                return true;
            }
            for (ScriptChunk chunk : input.getScriptSig().chunks()) {
                if (!chunk.isPushData() || !this.contains(chunk.data)) continue;
                return true;
            }
        }
        return false;
    }

    public synchronized boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BloomFilter other = (BloomFilter)o;
        return this.hashFuncs == other.hashFuncs && this.nTweak == other.nTweak && Arrays.equals(this.data, other.data);
    }

    public synchronized int hashCode() {
        return Objects.hash(this.hashFuncs, this.nTweak, Arrays.hashCode(this.data));
    }

    public static enum BloomUpdate {
        UPDATE_NONE,
        UPDATE_ALL,
        UPDATE_P2PUBKEY_ONLY;

    }
}

