/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.hash.HashCode;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import javax.annotation.Nullable;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.base.internal.TimeUtils;
import org.bitcoinj.core.Block;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.StoredBlock;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.store.BlockStore;
import org.bitcoinj.store.BlockStoreException;
import org.bitcoinj.store.FullPrunedBlockStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckpointManager {
    private static final Logger log = LoggerFactory.getLogger(CheckpointManager.class);
    private static final String TEXTUAL_MAGIC = "TXT CHECKPOINTS 1";
    private static final int MAX_SIGNATURES = 256;
    protected final TreeMap<Instant, StoredBlock> checkpoints = new TreeMap();
    protected final NetworkParameters params;
    protected final Sha256Hash dataHash;
    public static final BaseEncoding BASE64 = BaseEncoding.base64().omitPadding();

    public CheckpointManager(NetworkParameters params) throws IOException {
        this(params, null);
    }

    public CheckpointManager(NetworkParameters params, @Nullable InputStream inputStream2) throws IOException {
        this.params = Objects.requireNonNull(params);
        if (inputStream2 == null) {
            inputStream2 = CheckpointManager.openStream(params);
        }
        Objects.requireNonNull(inputStream2);
        inputStream2 = new BufferedInputStream(inputStream2);
        inputStream2.mark(1);
        int first = inputStream2.read();
        inputStream2.reset();
        if (first != TEXTUAL_MAGIC.charAt(0)) {
            throw new IOException("Unsupported format.");
        }
        this.dataHash = this.readTextual(inputStream2);
    }

    public static InputStream openStream(NetworkParameters params) {
        return CheckpointManager.class.getResourceAsStream("/" + params.getId() + ".checkpoints.txt");
    }

    private Sha256Hash readTextual(InputStream inputStream2) throws IOException {
        Hasher hasher = Hashing.sha256().newHasher();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream2, StandardCharsets.US_ASCII));){
            String magic = reader.readLine();
            if (!TEXTUAL_MAGIC.equals(magic)) {
                throw new IOException("unexpected magic: " + magic);
            }
            int numSigs = Integer.parseInt(reader.readLine());
            for (int i2 = 0; i2 < numSigs; ++i2) {
                reader.readLine();
            }
            int numCheckpoints = Integer.parseInt(reader.readLine());
            Preconditions.checkState(numCheckpoints > 0);
            hasher.putBytes(ByteBuffer.allocate(4).order(ByteOrder.BIG_ENDIAN).putInt(numCheckpoints).array());
            for (int i3 = 0; i3 < numCheckpoints; ++i3) {
                StoredBlock block;
                byte[] bytes = BASE64.decode((CharSequence)reader.readLine());
                hasher.putBytes(bytes);
                ByteBuffer buffer = ByteBuffer.wrap(bytes);
                if (bytes.length == 96) {
                    block = StoredBlock.deserializeCompact(buffer);
                } else if (bytes.length == 116) {
                    block = StoredBlock.deserializeCompactV2(buffer);
                } else {
                    throw new IllegalStateException("unexpected length of checkpoint: " + bytes.length);
                }
                this.checkpoints.put(block.getHeader().time(), block);
            }
            HashCode hash = hasher.hash();
            log.info("Read {} checkpoints up to time {}, hash is {}", this.checkpoints.size(), TimeUtils.dateTimeFormat(this.checkpoints.lastEntry().getKey()), hash);
            Sha256Hash sha256Hash = Sha256Hash.wrap(hash.asBytes());
            return sha256Hash;
        }
    }

    public StoredBlock getCheckpointBefore(Instant time) {
        try {
            Preconditions.checkArgument(time.isAfter(this.params.getGenesisBlock().time()));
            Map.Entry<Instant, StoredBlock> entry = this.checkpoints.floorEntry(time);
            if (entry != null) {
                return entry.getValue();
            }
            Block genesis = this.params.getGenesisBlock().cloneAsHeader();
            return new StoredBlock(genesis, genesis.getWork(), 0);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
    }

    @Deprecated
    public StoredBlock getCheckpointBefore(long timeSecs) {
        return this.getCheckpointBefore(Instant.ofEpochSecond(timeSecs));
    }

    public int numCheckpoints() {
        return this.checkpoints.size();
    }

    public Sha256Hash getDataHash() {
        return this.dataHash;
    }

    public static void checkpoint(NetworkParameters params, InputStream checkpoints, BlockStore store, Instant time) throws IOException, BlockStoreException {
        Objects.requireNonNull(params);
        Objects.requireNonNull(store);
        Preconditions.checkArgument(!(store instanceof FullPrunedBlockStore), () -> "you cannot use checkpointing with a full store");
        time = time.minus(7L, ChronoUnit.DAYS);
        log.info("Attempting to initialize a new block store with a checkpoint for time {} ({})", (Object)time.getEpochSecond(), (Object)TimeUtils.dateTimeFormat(time));
        BufferedInputStream stream = new BufferedInputStream(checkpoints);
        CheckpointManager manager = new CheckpointManager(params, stream);
        StoredBlock checkpoint = manager.getCheckpointBefore(time);
        store.put(checkpoint);
        store.setChainHead(checkpoint);
    }

    @Deprecated
    public static void checkpoint(NetworkParameters params, InputStream checkpoints, BlockStore store, long timeSecs) throws IOException, BlockStoreException {
        CheckpointManager.checkpoint(params, checkpoints, store, Instant.ofEpochSecond(timeSecs));
    }
}

