/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.bitcoinj.base.VarInt;
import org.bitcoinj.base.internal.Buffers;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.base.internal.TimeUtils;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Services;
import org.bitcoinj.core.internal.TorUtils;

public class PeerAddress {
    @Nullable
    private final InetAddress addr;
    @Nullable
    private final String hostname;
    private final int port;
    private final Services services;
    private final Instant time;
    private static final byte[] ONIONCAT_PREFIX = ByteUtils.parseHex("fd87d87eeb43");

    public static PeerAddress simple(InetAddress addr, int port) {
        return new PeerAddress(Objects.requireNonNull(addr), null, port, Services.none(), TimeUtils.currentTime().truncatedTo(ChronoUnit.SECONDS));
    }

    public static PeerAddress simple(InetSocketAddress addr) {
        return new PeerAddress(addr.getAddress(), null, addr.getPort(), Services.none(), TimeUtils.currentTime().truncatedTo(ChronoUnit.SECONDS));
    }

    public static PeerAddress inet(InetAddress addr, int port, Services services, Instant time) {
        return new PeerAddress(Objects.requireNonNull(addr), null, port, Objects.requireNonNull(services), Objects.requireNonNull(time));
    }

    public static PeerAddress inet(InetSocketAddress addr, Services services, Instant time) {
        return new PeerAddress(addr.getAddress(), null, addr.getPort(), Objects.requireNonNull(services), Objects.requireNonNull(time));
    }

    public static PeerAddress read(ByteBuffer payload, int protocolVariant) throws BufferUnderflowException, ProtocolException {
        Services services;
        if (protocolVariant < 0 || protocolVariant > 2) {
            throw new IllegalStateException("invalid protocolVariant: " + protocolVariant);
        }
        Instant time = Instant.ofEpochSecond(ByteUtils.readUint32(payload));
        InetAddress addr = null;
        String hostname = null;
        if (protocolVariant == 2) {
            services = Services.of(VarInt.read(payload).longValue());
            byte networkId = payload.get();
            byte[] addrBytes = Buffers.readLengthPrefixedBytes(payload);
            int addrLen = addrBytes.length;
            Optional<NetworkId> id = NetworkId.of(networkId);
            if (id.isPresent()) {
                switch (id.get()) {
                    case IPV4: {
                        if (addrLen != 4) {
                            throw new ProtocolException("invalid length of IPv4 address: " + addrLen);
                        }
                        addr = PeerAddress.getByAddress(addrBytes);
                        hostname = null;
                        break;
                    }
                    case IPV6: {
                        if (addrLen != 16) {
                            throw new ProtocolException("invalid length of IPv6 address: " + addrLen);
                        }
                        addr = PeerAddress.getByAddress(addrBytes);
                        hostname = null;
                        break;
                    }
                    case TORV2: {
                        if (addrLen != 10) {
                            throw new ProtocolException("invalid length of TORv2 address: " + addrLen);
                        }
                        hostname = TorUtils.encodeOnionUrlV2(addrBytes);
                        addr = null;
                        break;
                    }
                    case TORV3: {
                        if (addrLen != 32) {
                            throw new ProtocolException("invalid length of TORv3 address: " + addrLen);
                        }
                        hostname = TorUtils.encodeOnionUrlV3(addrBytes);
                        addr = null;
                        break;
                    }
                    case I2P: 
                    case CJDNS: {
                        addr = null;
                        hostname = null;
                    }
                }
            } else {
                addr = null;
                hostname = null;
            }
        } else {
            services = Services.read(payload);
            byte[] addrBytes = Buffers.readBytes(payload, 16);
            if (Arrays.equals(ONIONCAT_PREFIX, Arrays.copyOf(addrBytes, 6))) {
                byte[] onionAddress = Arrays.copyOfRange(addrBytes, 6, 16);
                hostname = TorUtils.encodeOnionUrlV2(onionAddress);
            } else {
                addr = PeerAddress.getByAddress(addrBytes);
                hostname = null;
            }
        }
        int port = ByteUtils.readUint16BE(payload);
        return new PeerAddress(addr, hostname, port, services, time);
    }

    private PeerAddress(InetAddress addr, String hostname, int port, Services services, Instant time) {
        this.addr = addr;
        this.hostname = hostname;
        this.port = port;
        this.services = services;
        this.time = time;
    }

    public static PeerAddress localhost(NetworkParameters params) {
        return PeerAddress.simple(InetAddress.getLoopbackAddress(), params.getPort());
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public ByteBuffer write(ByteBuffer buf, int protocolVariant) throws BufferOverflowException {
        if (protocolVariant < 1 || protocolVariant > 2) {
            throw new IllegalStateException("invalid protocolVariant: " + protocolVariant);
        }
        ByteUtils.writeInt32LE(this.time.getEpochSecond(), buf);
        if (protocolVariant == 2) {
            VarInt.of(this.services.bits()).write(buf);
            if (this.addr != null) {
                if (this.addr instanceof Inet4Address) {
                    buf.put((byte)1);
                    VarInt.of(4L).write(buf);
                    buf.put(this.addr.getAddress());
                } else {
                    if (!(this.addr instanceof Inet6Address)) throw new IllegalStateException();
                    buf.put((byte)2);
                    VarInt.of(16L).write(buf);
                    buf.put(this.addr.getAddress());
                }
            } else {
                if (this.addr != null || this.hostname == null || !this.hostname.toLowerCase(Locale.ROOT).endsWith(".onion")) throw new IllegalStateException();
                byte[] onionAddress = TorUtils.decodeOnionUrl(this.hostname);
                if (onionAddress.length == 10) {
                    buf.put((byte)3);
                    VarInt.of(10L).write(buf);
                    buf.put(onionAddress);
                } else {
                    if (onionAddress.length != 32) throw new IllegalStateException();
                    buf.put((byte)4);
                    VarInt.of(32L).write(buf);
                    buf.put(onionAddress);
                }
            }
        } else {
            this.services.write(buf);
            if (this.addr != null) {
                byte[] ipBytes = this.addr.getAddress();
                buf.put(PeerAddress.mapIntoIPv6(ipBytes));
            } else {
                if (this.hostname == null || !this.hostname.toLowerCase(Locale.ROOT).endsWith(".onion")) throw new IllegalStateException();
                byte[] onionAddress = TorUtils.decodeOnionUrl(this.hostname);
                if (onionAddress.length != 10) throw new IllegalStateException();
                buf.put(ONIONCAT_PREFIX);
                buf.put(onionAddress);
            }
        }
        ByteUtils.writeInt16BE(this.port, buf);
        return buf;
    }

    public byte[] serialize(int protocolVariant) {
        return this.write(ByteBuffer.allocate(this.getMessageSize(protocolVariant)), protocolVariant).array();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public int getMessageSize(int protocolVariant) {
        if (protocolVariant < 1 || protocolVariant > 2) {
            throw new IllegalStateException("invalid protocolVariant: " + protocolVariant);
        }
        int size = 0;
        size += 4;
        if (protocolVariant == 2) {
            size += VarInt.sizeOf(this.services.bits());
            ++size;
            if (this.addr != null) {
                if (this.addr instanceof Inet4Address) {
                    size += VarInt.sizeOf(4L) + 4;
                    return size += 2;
                } else {
                    if (!(this.addr instanceof Inet6Address)) throw new IllegalStateException();
                    size += VarInt.sizeOf(16L) + 16;
                }
                return size += 2;
            } else {
                if (this.addr != null || this.hostname == null || !this.hostname.toLowerCase(Locale.ROOT).endsWith(".onion")) throw new IllegalStateException();
                byte[] onionAddress = TorUtils.decodeOnionUrl(this.hostname);
                if (onionAddress.length != 10 && onionAddress.length != 32) throw new IllegalStateException();
                size += VarInt.sizeOf(onionAddress.length) + onionAddress.length;
            }
            return size += 2;
        } else {
            size += 8;
            size += 16;
        }
        return size += 2;
    }

    public static InetAddress getByAddress(byte[] addrBytes) {
        try {
            return InetAddress.getByAddress(addrBytes);
        }
        catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] mapIntoIPv6(byte[] ip) {
        Preconditions.checkArgument(ip.length == 4 || ip.length == 16, () -> "need IPv4 or IPv6");
        if (ip.length == 16) {
            return ip;
        }
        byte[] ipv6 = new byte[16];
        System.arraycopy(ip, 0, ipv6, 12, 4);
        ipv6[10] = -1;
        ipv6[11] = -1;
        return ipv6;
    }

    public String getHostname() {
        return this.hostname;
    }

    public InetAddress getAddr() {
        return this.addr;
    }

    public InetSocketAddress getSocketAddress() {
        return new InetSocketAddress(this.getAddr(), this.getPort());
    }

    public int getPort() {
        return this.port;
    }

    public Services getServices() {
        return this.services;
    }

    public Instant time() {
        return this.time;
    }

    @Deprecated
    public long getTime() {
        return this.time.getEpochSecond();
    }

    public String toString() {
        if (this.hostname != null) {
            return "[" + this.hostname + "]:" + this.port;
        }
        if (this.addr != null) {
            return "[" + this.addr.getHostAddress() + "]:" + this.port;
        }
        return "[ PeerAddress of unsupported type ]:" + this.port;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PeerAddress other = (PeerAddress)o;
        return Objects.equals(this.addr, other.addr) && Objects.equals(this.hostname, other.hostname) && this.port == other.port && Objects.equals(this.services, other.services);
    }

    public int hashCode() {
        return Objects.hash(this.addr, this.hostname, this.port, this.services);
    }

    public InetSocketAddress toSocketAddress() {
        if (this.hostname != null) {
            return InetSocketAddress.createUnresolved(this.hostname, this.port);
        }
        return new InetSocketAddress(this.addr, this.port);
    }

    private static enum NetworkId {
        IPV4(1),
        IPV6(2),
        TORV2(3),
        TORV3(4),
        I2P(5),
        CJDNS(6);

        final int value;

        private NetworkId(int value) {
            this.value = value;
        }

        static Optional<NetworkId> of(int value) {
            return Stream.of(NetworkId.values()).filter(id -> id.value == value).findFirst();
        }
    }
}

