/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.base.MoreObjects;
import com.google.common.math.IntMath;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import javax.annotation.Nullable;
import org.bitcoinj.base.Address;
import org.bitcoinj.base.Coin;
import org.bitcoinj.base.Network;
import org.bitcoinj.base.ScriptType;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.VarInt;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.base.internal.TimeUtils;
import org.bitcoinj.core.AbstractBlockChain;
import org.bitcoinj.core.BaseMessage;
import org.bitcoinj.core.Context;
import org.bitcoinj.core.LockTime;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.ProtocolVersion;
import org.bitcoinj.core.StoredBlock;
import org.bitcoinj.core.TransactionBag;
import org.bitcoinj.core.TransactionConfidence;
import org.bitcoinj.core.TransactionInput;
import org.bitcoinj.core.TransactionOutPoint;
import org.bitcoinj.core.TransactionOutput;
import org.bitcoinj.core.TransactionWitness;
import org.bitcoinj.core.TxConfidenceTable;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.crypto.AesKey;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.crypto.TransactionSignature;
import org.bitcoinj.script.Script;
import org.bitcoinj.script.ScriptBuilder;
import org.bitcoinj.script.ScriptError;
import org.bitcoinj.script.ScriptException;
import org.bitcoinj.script.ScriptPattern;
import org.bitcoinj.utils.ExchangeRate;
import org.bitcoinj.wallet.WalletTransaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Transaction
extends BaseMessage {
    private static final Comparator<Transaction> SORT_TX_BY_ID = Comparator.comparing(Transaction::getTxId);
    public static final Comparator<Transaction> SORT_TX_BY_UPDATE_TIME = Comparator.comparing(Transaction::sortableUpdateTime, Comparator.reverseOrder()).thenComparing(SORT_TX_BY_ID);
    public static final Comparator<Transaction> SORT_TX_BY_HEIGHT = Comparator.comparing(Transaction::sortableBlockHeight, Comparator.reverseOrder()).thenComparing(SORT_TX_BY_ID);
    private static final Logger log = LoggerFactory.getLogger(Transaction.class);
    public static final int SERIALIZE_TRANSACTION_NO_WITNESS = 0x40000000;
    @Deprecated
    public static final int LOCKTIME_THRESHOLD = 500000000;
    public static final int MAX_STANDARD_TX_SIZE = 100000;
    public static final Coin REFERENCE_DEFAULT_MIN_TX_FEE = Coin.valueOf(1000L);
    public static final Coin DEFAULT_TX_FEE = Coin.valueOf(100000L);
    private final int protocolVersion;
    private long version;
    private List<TransactionInput> inputs;
    private List<TransactionOutput> outputs;
    private volatile LockTime vLockTime;
    @Nullable
    private Instant updateTime = null;
    @Nullable
    private TransactionConfidence confidence;
    private Map<Sha256Hash, Integer> appearsInHashes;
    private Purpose purpose = Purpose.UNKNOWN;
    @Nullable
    private ExchangeRate exchangeRate;
    @Nullable
    private String memo;
    private Sha256Hash cachedTxId;
    private Sha256Hash cachedWTxId;
    public static final byte SIGHASH_ANYONECANPAY_VALUE = -128;

    private static Instant sortableUpdateTime(Transaction tx) {
        return tx.updateTime().orElse(Instant.EPOCH);
    }

    private static int sortableBlockHeight(Transaction tx) {
        TransactionConfidence confidence = tx.getConfidence();
        return confidence.getConfidenceType() == TransactionConfidence.ConfidenceType.BUILDING ? confidence.getAppearedAtChainHeight() : -1;
    }

    public static Transaction coinbase() {
        Transaction tx = new Transaction();
        tx.addInput(TransactionInput.coinbaseInput(tx, new byte[2]));
        return tx;
    }

    public static Transaction coinbase(byte[] inputScriptBytes) {
        Transaction tx = new Transaction();
        tx.addInput(TransactionInput.coinbaseInput(tx, inputScriptBytes));
        return tx;
    }

    public static Transaction read(ByteBuffer payload) throws BufferUnderflowException, ProtocolException {
        return Transaction.read(payload, ProtocolVersion.CURRENT.intValue());
    }

    public static Transaction read(ByteBuffer payload, int protocolVersion) throws BufferUnderflowException, ProtocolException {
        Transaction tx = new Transaction(protocolVersion);
        boolean allowWitness = Transaction.allowWitness(protocolVersion);
        tx.version = ByteUtils.readUint32(payload);
        byte flags = 0;
        tx.readInputs(payload);
        if (tx.inputs.size() == 0 && allowWitness) {
            flags = payload.get();
            if (flags != 0) {
                tx.readInputs(payload);
                tx.readOutputs(payload);
            } else {
                tx.outputs = new ArrayList<TransactionOutput>(0);
            }
        } else {
            tx.readOutputs(payload);
        }
        if ((flags & 1) != 0 && allowWitness) {
            flags = (byte)(flags ^ 1);
            tx.readWitnesses(payload);
            if (!tx.hasWitnesses()) {
                throw new ProtocolException("Superfluous witness record");
            }
        }
        if (flags != 0) {
            throw new ProtocolException("Unknown transaction optional data");
        }
        tx.vLockTime = LockTime.of(ByteUtils.readUint32(payload));
        return tx;
    }

    private Transaction(int protocolVersion) {
        this.protocolVersion = protocolVersion;
    }

    public Transaction() {
        this.protocolVersion = ProtocolVersion.CURRENT.intValue();
        this.version = 1L;
        this.inputs = new ArrayList<TransactionInput>();
        this.outputs = new ArrayList<TransactionOutput>();
        this.vLockTime = LockTime.unset();
    }

    @Deprecated
    public Transaction(NetworkParameters params) {
        this();
    }

    public Sha256Hash getTxId() {
        if (this.cachedTxId == null) {
            if (!this.hasWitnesses() && this.cachedWTxId != null) {
                this.cachedTxId = this.cachedWTxId;
            } else {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                try {
                    this.bitcoinSerializeToStream(baos, false);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                this.cachedTxId = Sha256Hash.wrapReversed(Sha256Hash.hashTwice(baos.toByteArray()));
            }
        }
        return this.cachedTxId;
    }

    private static boolean allowWitness(int protocolVersion) {
        return (protocolVersion & 0x40000000) == 0 && protocolVersion >= ProtocolVersion.WITNESS_VERSION.intValue();
    }

    public Sha256Hash getWTxId() {
        if (this.cachedWTxId == null) {
            if (!this.hasWitnesses() && this.cachedTxId != null) {
                this.cachedWTxId = this.cachedTxId;
            } else {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                try {
                    this.bitcoinSerializeToStream(baos, this.hasWitnesses());
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                this.cachedWTxId = Sha256Hash.wrapReversed(Sha256Hash.hashTwice(baos.toByteArray()));
            }
        }
        return this.cachedWTxId;
    }

    private void invalidateCachedTxIds() {
        this.cachedTxId = null;
        this.cachedWTxId = null;
    }

    public int getWeight() {
        int n;
        if (!this.hasWitnesses()) {
            return this.messageSize() * 4;
        }
        ByteArrayOutputStream stream = new ByteArrayOutputStream(255);
        try {
            this.bitcoinSerializeToStream(stream, false);
            int baseSize = stream.size();
            stream.reset();
            this.bitcoinSerializeToStream(stream, true);
            int totalSize = stream.size();
            n = baseSize * 3 + totalSize;
        }
        catch (Throwable throwable) {
            try {
                try {
                    stream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        stream.close();
        return n;
    }

    public int getVsize() {
        if (!this.hasWitnesses()) {
            return this.messageSize();
        }
        return IntMath.divide((int)this.getWeight(), (int)4, (RoundingMode)RoundingMode.CEILING);
    }

    public Coin getInputSum() {
        return this.inputs.stream().map(TransactionInput::getValue).filter(Objects::nonNull).reduce(Coin.ZERO, Coin::add);
    }

    public Coin getValueSentToMe(TransactionBag transactionBag) {
        Coin v = Coin.ZERO;
        for (TransactionOutput o : this.outputs) {
            if (!o.isMineOrWatched(transactionBag)) continue;
            v = v.add(o.getValue());
        }
        return v;
    }

    @Nullable
    public Map<Sha256Hash, Integer> getAppearsInHashes() {
        return this.appearsInHashes != null ? Collections.unmodifiableMap(new HashMap<Sha256Hash, Integer>(this.appearsInHashes)) : null;
    }

    public boolean isPending() {
        return this.getConfidence().getConfidenceType() == TransactionConfidence.ConfidenceType.PENDING;
    }

    public void setBlockAppearance(StoredBlock block, boolean bestChain, int relativityOffset) {
        Instant blockTime = block.getHeader().time();
        if (bestChain && (this.updateTime == null || this.updateTime.equals(Instant.EPOCH) || this.updateTime.isAfter(blockTime))) {
            this.updateTime = blockTime;
        }
        this.addBlockAppearance(block.getHeader().getHash(), relativityOffset);
        if (bestChain) {
            TransactionConfidence transactionConfidence = this.getConfidence();
            transactionConfidence.setAppearedAtChainHeight(block.getHeight());
        }
    }

    public void addBlockAppearance(Sha256Hash blockHash, int relativityOffset) {
        if (this.appearsInHashes == null) {
            this.appearsInHashes = new TreeMap<Sha256Hash, Integer>();
        }
        this.appearsInHashes.put(blockHash, relativityOffset);
    }

    public Coin getValueSentFromMe(TransactionBag wallet) throws ScriptException {
        Coin v = Coin.ZERO;
        for (TransactionInput input : this.inputs) {
            TransactionOutput connected = input.getConnectedOutput(wallet.getTransactionPool(WalletTransaction.Pool.UNSPENT));
            if (connected == null) {
                connected = input.getConnectedOutput(wallet.getTransactionPool(WalletTransaction.Pool.SPENT));
            }
            if (connected == null) {
                connected = input.getConnectedOutput(wallet.getTransactionPool(WalletTransaction.Pool.PENDING));
            }
            if (connected == null || !connected.isMineOrWatched(wallet)) continue;
            v = v.add(connected.getValue());
        }
        return v;
    }

    public Coin getOutputSum() {
        return this.outputs.stream().map(TransactionOutput::getValue).reduce(Coin.ZERO, Coin::add);
    }

    public Coin getValue(TransactionBag wallet) throws ScriptException {
        return this.getValueSentToMe(wallet).subtract(this.getValueSentFromMe(wallet));
    }

    public Coin getFee() {
        Coin fee = Coin.ZERO;
        if (this.inputs.isEmpty() || this.outputs.isEmpty()) {
            return null;
        }
        for (TransactionInput input : this.inputs) {
            if (input.getValue() == null) {
                return null;
            }
            fee = fee.add(input.getValue());
        }
        for (TransactionOutput output : this.outputs) {
            fee = fee.subtract(output.getValue());
        }
        return fee;
    }

    public boolean isAnyOutputSpent() {
        for (TransactionOutput output : this.outputs) {
            if (output.isAvailableForSpending()) continue;
            return true;
        }
        return false;
    }

    public boolean isEveryOwnedOutputSpent(TransactionBag transactionBag) {
        for (TransactionOutput output : this.outputs) {
            if (!output.isAvailableForSpending() || !output.isMineOrWatched(transactionBag)) continue;
            return false;
        }
        return true;
    }

    public Optional<Instant> updateTime() {
        return Optional.ofNullable(this.updateTime);
    }

    @Deprecated
    public Date getUpdateTime() {
        return Date.from(this.updateTime().orElse(Instant.EPOCH));
    }

    public void setUpdateTime(Instant updateTime) {
        this.updateTime = Objects.requireNonNull(updateTime);
    }

    public void clearUpdateTime() {
        this.updateTime = null;
    }

    @Deprecated
    public void setUpdateTime(Date updateTime) {
        if (updateTime != null && updateTime.getTime() > 0L) {
            this.setUpdateTime(updateTime.toInstant());
        } else {
            this.clearUpdateTime();
        }
    }

    private void readInputs(ByteBuffer payload) throws BufferUnderflowException, ProtocolException {
        VarInt numInputsVarInt = VarInt.read(payload);
        Preconditions.check(numInputsVarInt.fitsInt(), BufferUnderflowException::new);
        int numInputs = numInputsVarInt.intValue();
        this.inputs = new ArrayList<TransactionInput>(Math.min(numInputs, 20));
        for (long i2 = 0L; i2 < (long)numInputs; ++i2) {
            this.inputs.add(TransactionInput.read(payload, this));
        }
        this.invalidateCachedTxIds();
    }

    private void readOutputs(ByteBuffer payload) throws BufferUnderflowException, ProtocolException {
        VarInt numOutputsVarInt = VarInt.read(payload);
        Preconditions.check(numOutputsVarInt.fitsInt(), BufferUnderflowException::new);
        int numOutputs = numOutputsVarInt.intValue();
        this.outputs = new ArrayList<TransactionOutput>(Math.min(numOutputs, 20));
        for (long i2 = 0L; i2 < (long)numOutputs; ++i2) {
            this.outputs.add(TransactionOutput.read(payload, this));
        }
        this.invalidateCachedTxIds();
    }

    private void readWitnesses(ByteBuffer payload) throws BufferUnderflowException, ProtocolException {
        int numInputs = this.inputs.size();
        for (int i2 = 0; i2 < numInputs; ++i2) {
            TransactionWitness witness = TransactionWitness.read(payload);
            this.replaceInput(i2, this.getInput(i2).withWitness(witness));
        }
    }

    public boolean hasWitnesses() {
        return this.inputs.stream().anyMatch(TransactionInput::hasWitness);
    }

    public int getMessageSizeForPriorityCalc() {
        int size = this.messageSize();
        for (TransactionInput input : this.inputs) {
            int benefit = 41 + Math.min(110, input.getScriptSig().program().length);
            if (size <= benefit) continue;
            size -= benefit;
        }
        return size;
    }

    public boolean isCoinBase() {
        return this.inputs.size() == 1 && this.inputs.get(0).isCoinBase();
    }

    public String toString() {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this);
        helper.addValue((Object)this.toString(null, null));
        return helper.toString();
    }

    public String toString(@Nullable AbstractBlockChain chain, @Nullable Network network) {
        return this.toString(chain, network, "");
    }

    public String toString(@Nullable AbstractBlockChain chain, @Nullable Network network, CharSequence indent) {
        Objects.requireNonNull(indent);
        StringBuilder s = new StringBuilder();
        Sha256Hash txId = this.getTxId();
        Sha256Hash wTxId = this.getWTxId();
        s.append(indent).append(txId);
        if (!wTxId.equals(txId)) {
            s.append(", wtxid ").append(wTxId);
        }
        s.append('\n');
        int weight = this.getWeight();
        int size = this.messageSize();
        int vsize = this.getVsize();
        s.append(indent).append("weight: ").append(weight).append(" wu, ");
        if (size != vsize) {
            s.append(vsize).append(" virtual bytes, ");
        }
        s.append(size).append(" bytes\n");
        this.updateTime().ifPresent(time -> s.append(indent).append("updated: ").append(TimeUtils.dateTimeFormat(time)).append('\n'));
        if (this.version != 1L) {
            s.append(indent).append("version ").append(this.version).append('\n');
        }
        if (this.isTimeLocked()) {
            s.append(indent).append("time locked until ");
            LockTime locktime = this.lockTime();
            s.append(locktime);
            if (locktime instanceof LockTime.HeightLock && chain != null) {
                s.append(" (estimated to be reached at ").append(TimeUtils.dateTimeFormat(chain.estimateBlockTimeInstant(((LockTime.HeightLock)locktime).blockHeight()))).append(')');
            }
            s.append('\n');
        }
        if (this.hasRelativeLockTime()) {
            s.append(indent).append("has relative lock time\n");
        }
        if (this.isOptInFullRBF()) {
            s.append(indent).append("opts into full replace-by-fee\n");
        }
        if (this.purpose != null) {
            s.append(indent).append("purpose: ").append((Object)this.purpose).append('\n');
        }
        if (this.isCoinBase()) {
            s.append(indent).append("coinbase\n");
        } else if (!this.inputs.isEmpty()) {
            int i2 = 0;
            for (TransactionInput in : this.inputs) {
                s.append(indent).append("   ");
                s.append("in   ");
                try {
                    s.append(in.getScriptSig());
                    Coin value = in.getValue();
                    if (value != null) {
                        s.append("  ").append(value.toFriendlyString());
                    }
                    s.append('\n');
                    if (in.hasWitness()) {
                        s.append(indent).append("        witness:");
                        s.append(in.getWitness());
                        s.append('\n');
                    }
                    TransactionOutPoint outpoint = in.getOutpoint();
                    TransactionOutput connectedOutput = outpoint.getConnectedOutput();
                    s.append(indent).append("        ");
                    if (connectedOutput != null) {
                        Script scriptPubKey = connectedOutput.getScriptPubKey();
                        ScriptType scriptType = scriptPubKey.getScriptType();
                        if (scriptType != null) {
                            s.append((Object)scriptType);
                            if (network != null) {
                                s.append(" addr:").append(scriptPubKey.getToAddress(network));
                            }
                        } else {
                            s.append("unknown script type");
                        }
                    } else {
                        s.append("unconnected");
                    }
                    s.append("  outpoint:").append(outpoint).append('\n');
                    if (in.hasSequence()) {
                        s.append(indent).append("        sequence:").append(Long.toHexString(in.getSequenceNumber()));
                        if (in.isOptInFullRBF()) {
                            s.append(", opts into full RBF");
                        }
                        if (this.version >= 2L && in.hasRelativeLockTime()) {
                            s.append(", has RLT");
                        }
                        s.append('\n');
                    }
                }
                catch (Exception e) {
                    s.append("[exception: ").append(e.getMessage()).append("]\n");
                }
                ++i2;
            }
        } else {
            s.append(indent).append("   ");
            s.append("INCOMPLETE: No inputs!\n");
        }
        for (TransactionOutput out : this.outputs) {
            s.append(indent).append("   ");
            s.append("out  ");
            try {
                Script scriptPubKey = out.getScriptPubKey();
                s.append(scriptPubKey.chunks().size() > 0 ? scriptPubKey.toString() : "<no scriptPubKey>");
                s.append("  ");
                s.append(out.getValue().toFriendlyString());
                s.append('\n');
                s.append(indent).append("        ");
                ScriptType scriptType = scriptPubKey.getScriptType();
                if (scriptType != null) {
                    s.append((Object)scriptType);
                    if (network != null) {
                        s.append(" addr:").append(scriptPubKey.getToAddress(network));
                    }
                } else {
                    s.append("unknown script type");
                }
                if (!out.isAvailableForSpending()) {
                    s.append("  spent");
                    TransactionInput spentBy = out.getSpentBy();
                    if (spentBy != null) {
                        s.append(" by:");
                        s.append(spentBy.getParentTransaction().getTxId()).append(':').append(spentBy.getIndex());
                    }
                }
                s.append('\n');
            }
            catch (Exception e) {
                s.append("[exception: ").append(e.getMessage()).append("]\n");
            }
        }
        Coin fee = this.getFee();
        if (fee != null) {
            s.append(indent).append("   fee  ");
            s.append(fee.multiply(1000L).divide(weight).toFriendlyString()).append("/wu, ");
            if (size != vsize) {
                s.append(fee.multiply(1000L).divide(vsize).toFriendlyString()).append("/vkB, ");
            }
            s.append(fee.multiply(1000L).divide(size).toFriendlyString()).append("/kB  ");
            s.append(fee.toFriendlyString()).append('\n');
        }
        return s.toString();
    }

    public void clearInputs() {
        for (TransactionInput input : this.inputs) {
            input.setParent(null);
        }
        this.inputs.clear();
        this.invalidateCachedTxIds();
    }

    public TransactionInput addInput(TransactionOutput from) {
        TransactionInput input = this.addInput(new TransactionInput(this, from));
        this.invalidateCachedTxIds();
        return input;
    }

    public TransactionInput addInput(TransactionInput input) {
        input.setParent(this);
        this.inputs.add(input);
        this.invalidateCachedTxIds();
        return input;
    }

    public TransactionInput addInput(Sha256Hash spendTxHash, long outputIndex, Script script) {
        TransactionInput input = this.addInput(new TransactionInput(this, script.program(), new TransactionOutPoint(outputIndex, spendTxHash)));
        this.invalidateCachedTxIds();
        return input;
    }

    public TransactionInput addSignedInput(TransactionOutPoint prevOut, Script scriptPubKey, Coin amount, ECKey sigKey, SigHash sigHash, boolean anyoneCanPay) throws ScriptException {
        Preconditions.checkState(!this.outputs.isEmpty(), () -> "attempting to sign tx without outputs");
        if (amount == null || amount.value <= 0L) {
            log.warn("Illegal amount value. Amount is required for SegWit transactions.");
        }
        TransactionInput input = new TransactionInput(this, new byte[0], prevOut, amount);
        this.addInput(input);
        int inputIndex = this.inputs.size() - 1;
        if (ScriptPattern.isP2PK(scriptPubKey)) {
            TransactionSignature signature = this.calculateSignature(inputIndex, sigKey, scriptPubKey, sigHash, anyoneCanPay);
            input = input.withScriptSig(ScriptBuilder.createInputScript(signature));
            input = input.withoutWitness();
            this.replaceInput(inputIndex, input);
        } else if (ScriptPattern.isP2PKH(scriptPubKey)) {
            TransactionSignature signature = this.calculateSignature(inputIndex, sigKey, scriptPubKey, sigHash, anyoneCanPay);
            input = input.withScriptSig(ScriptBuilder.createInputScript(signature, sigKey));
            input = input.withoutWitness();
            this.replaceInput(inputIndex, input);
        } else if (ScriptPattern.isP2WPKH(scriptPubKey)) {
            Script scriptCode = ScriptBuilder.createP2PKHOutputScript(sigKey);
            TransactionSignature signature = this.calculateWitnessSignature(inputIndex, sigKey, scriptCode, input.getValue(), sigHash, anyoneCanPay);
            input = input.withScriptSig(ScriptBuilder.createEmpty());
            input = input.withWitness(TransactionWitness.redeemP2WPKH(signature, sigKey));
            this.replaceInput(inputIndex, input);
        } else {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Don't know how to sign for this kind of scriptPubKey: " + scriptPubKey);
        }
        return input;
    }

    @Deprecated
    public TransactionInput addSignedInput(TransactionOutPoint prevOut, Script scriptPubKey, ECKey sigKey, SigHash sigHash, boolean anyoneCanPay) throws ScriptException {
        return this.addSignedInput(prevOut, scriptPubKey, null, sigKey, sigHash, anyoneCanPay);
    }

    public TransactionInput addSignedInput(TransactionOutPoint prevOut, Script scriptPubKey, Coin amount, ECKey sigKey) throws ScriptException {
        return this.addSignedInput(prevOut, scriptPubKey, amount, sigKey, SigHash.ALL, false);
    }

    @Deprecated
    public TransactionInput addSignedInput(TransactionOutPoint prevOut, Script scriptPubKey, ECKey sigKey) throws ScriptException {
        return this.addSignedInput(prevOut, scriptPubKey, null, sigKey);
    }

    public TransactionInput addSignedInput(TransactionOutput output, ECKey sigKey) {
        return this.addSignedInput(output, sigKey, SigHash.ALL, false);
    }

    public TransactionInput addSignedInput(TransactionOutput output, ECKey sigKey, SigHash sigHash, boolean anyoneCanPay) {
        Objects.requireNonNull(output.getValue(), "TransactionOutput.getValue() must not be null");
        Preconditions.checkState(output.getValue().value > 0L, () -> "transactionOutput.getValue() must not be greater than zero");
        return this.addSignedInput(output.getOutPointFor(), output.getScriptPubKey(), output.getValue(), sigKey, sigHash, anyoneCanPay);
    }

    public void replaceInput(int index, TransactionInput input) {
        TransactionInput oldInput = this.inputs.remove(index);
        oldInput.setParent(null);
        input.setParent(this);
        this.inputs.add(index, input);
        this.invalidateCachedTxIds();
    }

    public void clearOutputs() {
        for (TransactionOutput output : this.outputs) {
            output.setParent(null);
        }
        this.outputs.clear();
        this.invalidateCachedTxIds();
    }

    public TransactionOutput addOutput(TransactionOutput to) {
        to.setParent(this);
        this.outputs.add(to);
        this.invalidateCachedTxIds();
        return to;
    }

    public void replaceOutput(int index, TransactionOutput output) {
        TransactionOutput oldOutput = this.outputs.remove(index);
        Preconditions.checkState(oldOutput.isAvailableForSpending(), () -> "output to be replaced is buried in a wallet: " + oldOutput);
        oldOutput.setParent(null);
        output.setParent(this);
        this.outputs.add(index, output);
        this.invalidateCachedTxIds();
    }

    public TransactionOutput addOutput(Coin value, Address address) {
        return this.addOutput(new TransactionOutput(this, value, address));
    }

    public TransactionOutput addOutput(Coin value, ECKey pubkey) {
        return this.addOutput(new TransactionOutput(this, value, pubkey));
    }

    public TransactionOutput addOutput(Coin value, Script script) {
        return this.addOutput(new TransactionOutput(this, value, script.program()));
    }

    public TransactionSignature calculateSignature(int inputIndex, ECKey key, byte[] redeemScript, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForSignature(inputIndex, redeemScript, hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash), hashType, anyoneCanPay);
    }

    public TransactionSignature calculateSignature(int inputIndex, ECKey key, Script redeemScript, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForSignature(inputIndex, redeemScript.program(), hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash), hashType, anyoneCanPay);
    }

    public TransactionSignature calculateSignature(int inputIndex, ECKey key, @Nullable AesKey aesKey, byte[] redeemScript, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForSignature(inputIndex, redeemScript, hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash, aesKey), hashType, anyoneCanPay);
    }

    public TransactionSignature calculateSignature(int inputIndex, ECKey key, @Nullable AesKey aesKey, Script redeemScript, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForSignature(inputIndex, redeemScript.program(), hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash, aesKey), hashType, anyoneCanPay);
    }

    public Sha256Hash hashForSignature(int inputIndex, byte[] redeemScript, SigHash type, boolean anyoneCanPay) {
        byte sigHashType = (byte)TransactionSignature.calcSigHashValue(type, anyoneCanPay);
        return this.hashForSignature(inputIndex, redeemScript, sigHashType);
    }

    public Sha256Hash hashForSignature(int inputIndex, Script redeemScript, SigHash type, boolean anyoneCanPay) {
        int sigHash = TransactionSignature.calcSigHashValue(type, anyoneCanPay);
        return this.hashForSignature(inputIndex, redeemScript.program(), (byte)sigHash);
    }

    public Sha256Hash hashForSignature(int inputIndex, byte[] connectedScript, byte sigHashType) {
        try {
            Transaction tx = Transaction.read(ByteBuffer.wrap(this.serialize()));
            for (int i2 = 0; i2 < tx.inputs.size(); ++i2) {
                TransactionInput input = tx.getInput(i2);
                input = input.withoutScriptBytes();
                input = input.withoutWitness();
                tx.replaceInput(i2, input);
            }
            connectedScript = Script.removeAllInstancesOfOp(connectedScript, 171);
            TransactionInput input = tx.getInput(inputIndex);
            input = input.withScriptBytes(connectedScript);
            tx.replaceInput(inputIndex, input);
            if ((sigHashType & 0x1F) == SigHash.NONE.value) {
                tx.outputs = new ArrayList<TransactionOutput>(0);
                for (int i3 = 0; i3 < tx.inputs.size(); ++i3) {
                    if (i3 == inputIndex) continue;
                    tx.replaceInput(i3, tx.getInput(i3).withSequence(0L));
                }
            } else if ((sigHashType & 0x1F) == SigHash.SINGLE.value) {
                int i4;
                if (inputIndex >= tx.outputs.size()) {
                    return Sha256Hash.wrap("0100000000000000000000000000000000000000000000000000000000000000");
                }
                tx.outputs = new ArrayList<TransactionOutput>(tx.outputs.subList(0, inputIndex + 1));
                for (i4 = 0; i4 < inputIndex; ++i4) {
                    tx.outputs.set(i4, new TransactionOutput(tx, Coin.NEGATIVE_SATOSHI, new byte[0]));
                }
                for (i4 = 0; i4 < tx.inputs.size(); ++i4) {
                    if (i4 == inputIndex) continue;
                    tx.replaceInput(i4, tx.getInput(i4).withSequence(0L));
                }
            }
            if ((sigHashType & SigHash.ANYONECANPAY.value) == SigHash.ANYONECANPAY.value) {
                tx.inputs = new ArrayList<TransactionInput>();
                tx.inputs.add(input);
            }
            ByteArrayOutputStream bos = new ByteArrayOutputStream(255);
            tx.bitcoinSerializeToStream(bos, false);
            ByteUtils.writeInt32LE(0xFF & sigHashType, (OutputStream)bos);
            Sha256Hash hash = Sha256Hash.twiceOf(bos.toByteArray());
            bos.close();
            return hash;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public TransactionSignature calculateWitnessSignature(int inputIndex, ECKey key, byte[] scriptCode, Coin value, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForWitnessSignature(inputIndex, scriptCode, value, hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash), hashType, anyoneCanPay);
    }

    public TransactionSignature calculateWitnessSignature(int inputIndex, ECKey key, Script scriptCode, Coin value, SigHash hashType, boolean anyoneCanPay) {
        return this.calculateWitnessSignature(inputIndex, key, scriptCode.program(), value, hashType, anyoneCanPay);
    }

    public TransactionSignature calculateWitnessSignature(int inputIndex, ECKey key, @Nullable AesKey aesKey, byte[] scriptCode, Coin value, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForWitnessSignature(inputIndex, scriptCode, value, hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash, aesKey), hashType, anyoneCanPay);
    }

    public TransactionSignature calculateWitnessSignature(int inputIndex, ECKey key, @Nullable AesKey aesKey, Script scriptCode, Coin value, SigHash hashType, boolean anyoneCanPay) {
        return this.calculateWitnessSignature(inputIndex, key, aesKey, scriptCode.program(), value, hashType, anyoneCanPay);
    }

    public synchronized Sha256Hash hashForWitnessSignature(int inputIndex, byte[] scriptCode, Coin prevValue, SigHash type, boolean anyoneCanPay) {
        int sigHash = TransactionSignature.calcSigHashValue(type, anyoneCanPay);
        return this.hashForWitnessSignature(inputIndex, scriptCode, prevValue, (byte)sigHash);
    }

    public synchronized Sha256Hash hashForWitnessSignature(int inputIndex, Script scriptCode, Coin prevValue, SigHash type, boolean anyoneCanPay) {
        return this.hashForWitnessSignature(inputIndex, scriptCode.program(), prevValue, type, anyoneCanPay);
    }

    public synchronized Sha256Hash hashForWitnessSignature(int inputIndex, byte[] scriptCode, Coin prevValue, byte sigHashType) {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(255);
        try {
            ByteArrayOutputStream bosHashOutputs;
            boolean signAll;
            byte[] hashPrevouts = new byte[32];
            byte[] hashSequence = new byte[32];
            byte[] hashOutputs = new byte[32];
            int basicSigHashType = sigHashType & 0x1F;
            boolean anyoneCanPay = (sigHashType & SigHash.ANYONECANPAY.value) == SigHash.ANYONECANPAY.value;
            boolean bl = signAll = basicSigHashType != SigHash.SINGLE.value && basicSigHashType != SigHash.NONE.value;
            if (!anyoneCanPay) {
                ByteArrayOutputStream bosHashPrevouts = new ByteArrayOutputStream(256);
                for (TransactionInput input : this.inputs) {
                    bosHashPrevouts.write(input.getOutpoint().hash().serialize());
                    ByteUtils.writeInt32LE(input.getOutpoint().index(), (OutputStream)bosHashPrevouts);
                }
                hashPrevouts = Sha256Hash.hashTwice(bosHashPrevouts.toByteArray());
            }
            if (!anyoneCanPay && signAll) {
                ByteArrayOutputStream bosSequence = new ByteArrayOutputStream(256);
                for (TransactionInput input : this.inputs) {
                    ByteUtils.writeInt32LE(input.getSequenceNumber(), (OutputStream)bosSequence);
                }
                hashSequence = Sha256Hash.hashTwice(bosSequence.toByteArray());
            }
            if (signAll) {
                bosHashOutputs = new ByteArrayOutputStream(256);
                for (TransactionOutput output : this.outputs) {
                    ByteUtils.writeInt64LE(BigInteger.valueOf(output.getValue().getValue()), (OutputStream)bosHashOutputs);
                    byte[] scriptBytes = output.getScriptBytes();
                    bosHashOutputs.write(VarInt.of(scriptBytes.length).serialize());
                    bosHashOutputs.write(scriptBytes);
                }
                hashOutputs = Sha256Hash.hashTwice(bosHashOutputs.toByteArray());
            } else if (basicSigHashType == SigHash.SINGLE.value && inputIndex < this.outputs.size()) {
                bosHashOutputs = new ByteArrayOutputStream(256);
                ByteUtils.writeInt64LE(BigInteger.valueOf(this.outputs.get(inputIndex).getValue().getValue()), (OutputStream)bosHashOutputs);
                byte[] scriptBytes = this.outputs.get(inputIndex).getScriptBytes();
                bosHashOutputs.write(VarInt.of(scriptBytes.length).serialize());
                bosHashOutputs.write(scriptBytes);
                hashOutputs = Sha256Hash.hashTwice(bosHashOutputs.toByteArray());
            }
            ByteUtils.writeInt32LE(this.version, (OutputStream)bos);
            bos.write(hashPrevouts);
            bos.write(hashSequence);
            bos.write(this.inputs.get(inputIndex).getOutpoint().hash().serialize());
            ByteUtils.writeInt32LE(this.inputs.get(inputIndex).getOutpoint().index(), (OutputStream)bos);
            bos.write(VarInt.of(scriptCode.length).serialize());
            bos.write(scriptCode);
            ByteUtils.writeInt64LE(BigInteger.valueOf(prevValue.getValue()), (OutputStream)bos);
            ByteUtils.writeInt32LE(this.inputs.get(inputIndex).getSequenceNumber(), (OutputStream)bos);
            bos.write(hashOutputs);
            ByteUtils.writeInt32LE(this.vLockTime.rawValue(), (OutputStream)bos);
            ByteUtils.writeInt32LE(0xFF & sigHashType, (OutputStream)bos);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return Sha256Hash.twiceOf(bos.toByteArray());
    }

    @Override
    public int messageSize() {
        boolean useSegwit = this.hasWitnesses() && Transaction.allowWitness(this.protocolVersion);
        int size = 4;
        if (useSegwit) {
            size += 2;
        }
        size += VarInt.sizeOf(this.inputs.size());
        for (TransactionInput in : this.inputs) {
            size += in.messageSize();
        }
        size += VarInt.sizeOf(this.outputs.size());
        for (TransactionOutput out : this.outputs) {
            size += out.messageSize();
        }
        if (useSegwit) {
            for (TransactionInput in : this.inputs) {
                size += in.getWitness().messageSize();
            }
        }
        return size += 4;
    }

    @Override
    protected void bitcoinSerializeToStream(OutputStream stream) throws IOException {
        boolean useSegwit = this.hasWitnesses() && Transaction.allowWitness(this.protocolVersion);
        this.bitcoinSerializeToStream(stream, useSegwit);
    }

    protected void bitcoinSerializeToStream(OutputStream stream, boolean useSegwit) throws IOException {
        ByteUtils.writeInt32LE(this.version, stream);
        if (useSegwit) {
            stream.write(0);
            stream.write(1);
        }
        stream.write(VarInt.of(this.inputs.size()).serialize());
        for (TransactionInput in : this.inputs) {
            stream.write(in.serialize());
        }
        stream.write(VarInt.of(this.outputs.size()).serialize());
        for (TransactionOutput out : this.outputs) {
            stream.write(out.serialize());
        }
        if (useSegwit) {
            for (TransactionInput in : this.inputs) {
                stream.write(in.getWitness().serialize());
            }
        }
        ByteUtils.writeInt32LE(this.vLockTime.rawValue(), stream);
    }

    public LockTime lockTime() {
        return this.vLockTime;
    }

    @Deprecated
    public long getLockTime() {
        return this.lockTime().rawValue();
    }

    public void setLockTime(long lockTime) {
        boolean seqNumSet = false;
        for (TransactionInput input : this.inputs) {
            if (input.getSequenceNumber() == 0xFFFFFFFFL) continue;
            seqNumSet = true;
            break;
        }
        if (lockTime != 0L && (!seqNumSet || this.inputs.isEmpty())) {
            log.warn("You are setting the lock time on a transaction but none of the inputs have non-default sequence numbers. This will not do what you expect!");
        }
        this.vLockTime = LockTime.of(lockTime);
        this.invalidateCachedTxIds();
    }

    public long getVersion() {
        return this.version;
    }

    public void setVersion(int version) {
        this.version = version;
        this.invalidateCachedTxIds();
    }

    public List<TransactionInput> getInputs() {
        return Collections.unmodifiableList(this.inputs);
    }

    public List<TransactionOutput> getOutputs() {
        return Collections.unmodifiableList(this.outputs);
    }

    public List<TransactionOutput> getWalletOutputs(TransactionBag transactionBag) {
        LinkedList<TransactionOutput> walletOutputs = new LinkedList<TransactionOutput>();
        for (TransactionOutput o : this.outputs) {
            if (!o.isMineOrWatched(transactionBag)) continue;
            walletOutputs.add(o);
        }
        return walletOutputs;
    }

    public void shuffleOutputs() {
        Collections.shuffle(this.outputs);
    }

    public TransactionInput getInput(long index) {
        return this.inputs.get((int)index);
    }

    public TransactionOutput getOutput(long index) {
        return this.outputs.get((int)index);
    }

    public TransactionOutput getOutput(TransactionOutPoint outpoint) {
        Preconditions.checkArgument(outpoint.hash().equals(this.getTxId()), () -> "outpoint points to a different transaction");
        return this.getOutput(outpoint.index());
    }

    public TransactionConfidence getConfidence() {
        return this.getConfidence(Context.get());
    }

    TransactionConfidence getConfidence(Context context) {
        return this.getConfidence(context.getConfidenceTable());
    }

    TransactionConfidence getConfidence(TxConfidenceTable table) {
        if (this.confidence == null) {
            this.confidence = table.getOrCreate(this.getTxId());
        }
        return this.confidence;
    }

    public boolean hasConfidence() {
        return this.getConfidence().getConfidenceType() != TransactionConfidence.ConfidenceType.UNKNOWN;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        return this.getTxId().equals(((Transaction)o).getTxId());
    }

    public int hashCode() {
        return this.getTxId().hashCode();
    }

    public int getSigOpCount() throws ScriptException {
        int sigOps = 0;
        for (TransactionInput input : this.inputs) {
            sigOps += Script.getSigOpCount(input.getScriptBytes());
        }
        for (TransactionOutput output : this.outputs) {
            sigOps += Script.getSigOpCount(output.getScriptBytes());
        }
        return sigOps;
    }

    public void checkCoinBaseHeight(int height) throws VerificationException {
        Preconditions.checkArgument(height >= 0);
        Preconditions.checkState(this.isCoinBase());
        TransactionInput in = this.getInput(0L);
        ScriptBuilder builder = new ScriptBuilder();
        builder.number(height);
        byte[] expected = builder.build().program();
        byte[] actual = in.getScriptBytes();
        if (actual.length < expected.length) {
            throw new VerificationException.CoinbaseHeightMismatch("Block height mismatch in coinbase.");
        }
        for (int scriptIdx = 0; scriptIdx < expected.length; ++scriptIdx) {
            if (actual[scriptIdx] == expected[scriptIdx]) continue;
            throw new VerificationException.CoinbaseHeightMismatch("Block height mismatch in coinbase.");
        }
    }

    public Sha256Hash findWitnessCommitment() {
        Preconditions.checkState(this.isCoinBase());
        ArrayList<TransactionOutput> reversed = new ArrayList<TransactionOutput>(this.outputs);
        Collections.reverse(reversed);
        return reversed.stream().map(TransactionOutput::getScriptPubKey).filter(ScriptPattern::isWitnessCommitment).findFirst().map(ScriptPattern::extractWitnessCommitmentHash).orElse(null);
    }

    public boolean isTimeLocked() {
        if (!this.lockTime().isSet()) {
            return false;
        }
        for (TransactionInput input : this.getInputs()) {
            if (!input.hasSequence()) continue;
            return true;
        }
        return false;
    }

    public boolean hasRelativeLockTime() {
        if (this.version < 2L) {
            return false;
        }
        for (TransactionInput input : this.getInputs()) {
            if (!input.hasRelativeLockTime()) continue;
            return true;
        }
        return false;
    }

    public boolean isOptInFullRBF() {
        for (TransactionInput input : this.getInputs()) {
            if (!input.isOptInFullRBF()) continue;
            return true;
        }
        return false;
    }

    public boolean isFinal(int height, Instant blockTime) {
        LockTime locktime = this.lockTime();
        return locktime.rawValue() < (locktime instanceof LockTime.HeightLock ? (long)height : blockTime.getEpochSecond()) || !this.isTimeLocked();
    }

    @Deprecated
    public boolean isFinal(int height, long blockTimeSeconds) {
        return this.isFinal(height, Instant.ofEpochSecond(blockTimeSeconds));
    }

    public Instant estimateUnlockTime(AbstractBlockChain chain) {
        LockTime locktime = this.lockTime();
        return locktime instanceof LockTime.HeightLock ? chain.estimateBlockTimeInstant(((LockTime.HeightLock)locktime).blockHeight()) : ((LockTime.TimeLock)locktime).timestamp();
    }

    @Deprecated
    public Date estimateLockTime(AbstractBlockChain chain) {
        return Date.from(this.estimateUnlockTime(chain));
    }

    public Purpose getPurpose() {
        return this.purpose;
    }

    public void setPurpose(Purpose purpose) {
        this.purpose = purpose;
    }

    @Nullable
    public ExchangeRate getExchangeRate() {
        return this.exchangeRate;
    }

    public void setExchangeRate(ExchangeRate exchangeRate) {
        this.exchangeRate = exchangeRate;
    }

    @Nullable
    public String getMemo() {
        return this.memo;
    }

    public void setMemo(String memo) {
        this.memo = memo;
    }

    public static void verify(Network network, Transaction tx) throws VerificationException {
        if (tx.inputs.size() == 0 || tx.outputs.size() == 0) {
            throw new VerificationException.EmptyInputsOrOutputs();
        }
        if (tx.messageSize() > 1000000) {
            throw new VerificationException.LargerThanMaxBlockSize();
        }
        HashSet<TransactionOutPoint> outpoints = new HashSet<TransactionOutPoint>();
        for (TransactionInput input : tx.inputs) {
            if (outpoints.contains(input.getOutpoint())) {
                throw new VerificationException.DuplicatedOutPoint();
            }
            outpoints.add(input.getOutpoint());
        }
        Coin valueOut = Coin.ZERO;
        for (TransactionOutput output : tx.outputs) {
            Coin value = output.getValue();
            if (value.signum() < 0) {
                throw new VerificationException.NegativeValueOutput();
            }
            try {
                valueOut = valueOut.add(value);
            }
            catch (ArithmeticException e) {
                throw new VerificationException.ExcessiveValue();
            }
            if (!network.exceedsMaxMoney(valueOut)) continue;
            throw new VerificationException.ExcessiveValue();
        }
        if (tx.isCoinBase()) {
            if (tx.inputs.get(0).getScriptBytes().length < 2 || tx.inputs.get(0).getScriptBytes().length > 100) {
                throw new VerificationException.CoinbaseScriptSizeOutOfRange();
            }
        } else {
            for (TransactionInput input : tx.inputs) {
                if (!input.isCoinBase()) continue;
                throw new VerificationException.UnexpectedCoinbaseInput();
            }
        }
    }

    @Deprecated
    public static void verify(NetworkParameters params, Transaction tx) throws VerificationException {
        Transaction.verify(params.network(), tx);
    }

    public static enum Purpose {
        UNKNOWN,
        USER_PAYMENT,
        KEY_ROTATION,
        ASSURANCE_CONTRACT_CLAIM,
        ASSURANCE_CONTRACT_PLEDGE,
        ASSURANCE_CONTRACT_STUB,
        RAISE_FEE;

    }

    public static enum SigHash {
        ALL(1),
        NONE(2),
        SINGLE(3),
        ANYONECANPAY(128),
        ANYONECANPAY_ALL(129),
        ANYONECANPAY_NONE(130),
        ANYONECANPAY_SINGLE(131),
        UNSET(0);

        public final int value;

        private SigHash(int value) {
            this.value = value;
        }

        public byte byteValue() {
            return (byte)this.value;
        }
    }
}

