/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto;

import com.google.common.base.MoreObjects;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nullable;
import org.bitcoinj.base.Base58;
import org.bitcoinj.base.Network;
import org.bitcoinj.base.ScriptType;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.crypto.AesKey;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.HDKeyDerivation;
import org.bitcoinj.crypto.HDPath;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.LazyECPoint;
import org.bitcoinj.crypto.internal.CryptoUtils;
import org.bouncycastle.math.ec.ECPoint;

public class DeterministicKey
extends ECKey {
    public static final Comparator<ECKey> CHILDNUM_ORDER = (k1, k2) -> {
        ChildNumber cn1 = ((DeterministicKey)k1).getChildNumber();
        ChildNumber cn2 = ((DeterministicKey)k2).getChildNumber();
        return cn1.compareTo(cn2);
    };
    private final DeterministicKey parent;
    private final HDPath childNumberPath;
    private final int depth;
    private int parentFingerprint;
    private final byte[] chainCode;

    public DeterministicKey(List<ChildNumber> childNumberPath, byte[] chainCode, LazyECPoint publicAsPoint, @Nullable BigInteger priv, @Nullable DeterministicKey parent) {
        super(priv, publicAsPoint.compress());
        Preconditions.checkArgument(chainCode.length == 32);
        this.parent = parent;
        this.childNumberPath = HDPath.M(Objects.requireNonNull(childNumberPath));
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = parent == null ? 0 : parent.depth + 1;
        this.parentFingerprint = parent != null ? parent.getFingerprint() : 0;
    }

    public DeterministicKey(List<ChildNumber> childNumberPath, byte[] chainCode, ECPoint publicAsPoint, boolean compressed, @Nullable BigInteger priv, @Nullable DeterministicKey parent) {
        this(childNumberPath, chainCode, new LazyECPoint(publicAsPoint, compressed), priv, parent);
    }

    public DeterministicKey(HDPath hdPath, byte[] chainCode, BigInteger priv, @Nullable DeterministicKey parent) {
        super(priv, ECKey.publicPointFromPrivate(priv), true);
        Preconditions.checkArgument(chainCode.length == 32);
        this.parent = parent;
        this.childNumberPath = Objects.requireNonNull(hdPath);
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = parent == null ? 0 : parent.depth + 1;
        this.parentFingerprint = parent != null ? parent.getFingerprint() : 0;
    }

    public DeterministicKey(List<ChildNumber> childNumberPath, byte[] chainCode, KeyCrypter crypter, LazyECPoint pub, EncryptedData priv, @Nullable DeterministicKey parent) {
        this(childNumberPath, chainCode, pub, null, parent);
        this.encryptedPrivateKey = Objects.requireNonNull(priv);
        this.keyCrypter = Objects.requireNonNull(crypter);
    }

    private int ascertainParentFingerprint(int parentFingerprint) throws IllegalArgumentException {
        if (parentFingerprint != 0) {
            if (this.parent != null) {
                Preconditions.checkArgument(this.parent.getFingerprint() == parentFingerprint, () -> "parent fingerprint mismatch: " + Integer.toHexString(this.parent.getFingerprint()) + " vs " + Integer.toHexString(parentFingerprint));
            }
            return parentFingerprint;
        }
        return 0;
    }

    public DeterministicKey(List<ChildNumber> childNumberPath, byte[] chainCode, LazyECPoint publicAsPoint, @Nullable DeterministicKey parent, int depth, int parentFingerprint) {
        super(null, publicAsPoint.compress());
        Preconditions.checkArgument(chainCode.length == 32);
        this.parent = parent;
        this.childNumberPath = HDPath.M(Objects.requireNonNull(childNumberPath));
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = depth;
        this.parentFingerprint = this.ascertainParentFingerprint(parentFingerprint);
    }

    public DeterministicKey(List<ChildNumber> childNumberPath, byte[] chainCode, BigInteger priv, @Nullable DeterministicKey parent, int depth, int parentFingerprint) {
        super(priv, ECKey.publicPointFromPrivate(priv), true);
        Preconditions.checkArgument(chainCode.length == 32);
        this.parent = parent;
        this.childNumberPath = HDPath.M(Objects.requireNonNull(childNumberPath));
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = depth;
        this.parentFingerprint = this.ascertainParentFingerprint(parentFingerprint);
    }

    public DeterministicKey(DeterministicKey keyToClone, DeterministicKey newParent) {
        super(keyToClone.priv, keyToClone.pub.get(), keyToClone.pub.isCompressed());
        this.parent = newParent;
        this.childNumberPath = keyToClone.childNumberPath;
        this.chainCode = keyToClone.chainCode;
        this.encryptedPrivateKey = keyToClone.encryptedPrivateKey;
        this.depth = this.childNumberPath.size();
        this.parentFingerprint = this.parent.getFingerprint();
    }

    public HDPath getPath() {
        return this.childNumberPath;
    }

    public String getPathAsString() {
        return this.getPath().toString();
    }

    public int getDepth() {
        return this.depth;
    }

    public ChildNumber getChildNumber() {
        return this.childNumberPath.size() == 0 ? ChildNumber.ZERO : this.childNumberPath.get(this.childNumberPath.size() - 1);
    }

    public byte[] getChainCode() {
        return this.chainCode;
    }

    public byte[] getIdentifier() {
        return CryptoUtils.sha256hash160(this.getPubKey());
    }

    public int getFingerprint() {
        return ByteBuffer.wrap(Arrays.copyOfRange(this.getIdentifier(), 0, 4)).getInt();
    }

    @Nullable
    public DeterministicKey getParent() {
        return this.parent;
    }

    public int getParentFingerprint() {
        return this.parentFingerprint;
    }

    public byte[] getPrivKeyBytes33() {
        byte[] bytes33 = new byte[33];
        byte[] priv = this.getPrivKeyBytes();
        System.arraycopy(priv, 0, bytes33, 33 - priv.length, priv.length);
        return bytes33;
    }

    public DeterministicKey dropPrivateBytes() {
        if (this.isPubKeyOnly()) {
            return this;
        }
        return new DeterministicKey(this.getPath(), this.getChainCode(), this.pub, null, this.parent);
    }

    public DeterministicKey dropParent() {
        DeterministicKey key = new DeterministicKey(this.getPath(), this.getChainCode(), this.pub, this.priv, null);
        key.parentFingerprint = this.parentFingerprint;
        return key;
    }

    static byte[] addChecksum(byte[] input) {
        int inputLength = input.length;
        byte[] checksummed = new byte[inputLength + 4];
        System.arraycopy(input, 0, checksummed, 0, inputLength);
        byte[] checksum = Sha256Hash.hashTwice(input);
        System.arraycopy(checksum, 0, checksummed, inputLength, 4);
        return checksummed;
    }

    @Override
    public DeterministicKey encrypt(KeyCrypter keyCrypter, AesKey aesKey) throws KeyCrypterException {
        throw new UnsupportedOperationException("Must supply a new parent for encryption");
    }

    public DeterministicKey encrypt(KeyCrypter keyCrypter, AesKey aesKey, @Nullable DeterministicKey newParent) throws KeyCrypterException {
        byte[] privKeyBytes;
        Objects.requireNonNull(keyCrypter);
        if (newParent != null) {
            Preconditions.checkArgument(newParent.isEncrypted());
        }
        Preconditions.checkState((privKeyBytes = this.getPrivKeyBytes()) != null, () -> "Private key is not available");
        EncryptedData encryptedPrivateKey = keyCrypter.encrypt(privKeyBytes, aesKey);
        DeterministicKey key = new DeterministicKey((List<ChildNumber>)this.childNumberPath, this.chainCode, keyCrypter, this.pub, encryptedPrivateKey, newParent);
        if (newParent == null) {
            Optional<Instant> creationTime = this.getCreationTime();
            if (creationTime.isPresent()) {
                key.setCreationTime(creationTime.get());
            } else {
                key.clearCreationTime();
            }
        }
        return key;
    }

    @Override
    public boolean isPubKeyOnly() {
        return super.isPubKeyOnly() && (this.parent == null || this.parent.isPubKeyOnly());
    }

    @Override
    public boolean hasPrivKey() {
        return this.findParentWithPrivKey() != null;
    }

    @Override
    @Nullable
    public byte[] getSecretBytes() {
        return this.priv != null ? this.getPrivKeyBytes() : null;
    }

    @Override
    public boolean isEncrypted() {
        return this.priv == null && (super.isEncrypted() || this.parent != null && this.parent.isEncrypted());
    }

    @Override
    @Nullable
    public KeyCrypter getKeyCrypter() {
        if (this.keyCrypter != null) {
            return this.keyCrypter;
        }
        if (this.parent != null) {
            return this.parent.getKeyCrypter();
        }
        return null;
    }

    @Override
    public ECKey.ECDSASignature sign(Sha256Hash input, @Nullable AesKey aesKey) throws KeyCrypterException {
        if (this.isEncrypted()) {
            return super.sign(input, aesKey);
        }
        BigInteger privateKey = this.findOrDerivePrivateKey();
        if (privateKey == null) {
            throw new ECKey.MissingPrivateKeyException();
        }
        return super.doSign(input, privateKey);
    }

    @Override
    public DeterministicKey decrypt(KeyCrypter keyCrypter, AesKey aesKey) throws KeyCrypterException {
        Objects.requireNonNull(keyCrypter);
        if (this.keyCrypter != null && !this.keyCrypter.equals(keyCrypter)) {
            throw new KeyCrypterException("The keyCrypter being used to decrypt the key is different to the one that was used to encrypt it");
        }
        BigInteger privKey = this.findOrDeriveEncryptedPrivateKey(keyCrypter, aesKey);
        DeterministicKey key = new DeterministicKey(this.childNumberPath, this.chainCode, privKey, this.parent);
        if (!Arrays.equals(key.getPubKey(), this.getPubKey())) {
            throw new KeyCrypterException.PublicPrivateMismatch("Provided AES key is wrong");
        }
        if (this.parent == null) {
            Optional<Instant> creationTime = this.getCreationTime();
            if (creationTime.isPresent()) {
                key.setCreationTime(creationTime.get());
            } else {
                key.clearCreationTime();
            }
        }
        return key;
    }

    @Override
    public DeterministicKey decrypt(AesKey aesKey) throws KeyCrypterException {
        return (DeterministicKey)super.decrypt(aesKey);
    }

    private BigInteger findOrDeriveEncryptedPrivateKey(KeyCrypter keyCrypter, AesKey aesKey) {
        if (this.encryptedPrivateKey != null) {
            byte[] decryptedKey = keyCrypter.decrypt(this.encryptedPrivateKey, aesKey);
            if (decryptedKey.length != 32) {
                throw new KeyCrypterException.InvalidCipherText("Decrypted key must be 32 bytes long, but is " + decryptedKey.length);
            }
            return ByteUtils.bytesToBigInteger(decryptedKey);
        }
        DeterministicKey cursor = this.parent;
        while (cursor != null && cursor.encryptedPrivateKey == null) {
            cursor = cursor.parent;
        }
        if (cursor == null) {
            throw new KeyCrypterException("Neither this key nor its parents have an encrypted private key");
        }
        byte[] parentalPrivateKeyBytes = keyCrypter.decrypt(cursor.encryptedPrivateKey, aesKey);
        if (parentalPrivateKeyBytes.length != 32) {
            throw new KeyCrypterException.InvalidCipherText("Decrypted key must be 32 bytes long, but is " + parentalPrivateKeyBytes.length);
        }
        return this.derivePrivateKeyDownwards(cursor, parentalPrivateKeyBytes);
    }

    private DeterministicKey findParentWithPrivKey() {
        DeterministicKey cursor = this;
        while (cursor != null && cursor.priv == null) {
            cursor = cursor.parent;
        }
        return cursor;
    }

    @Nullable
    private BigInteger findOrDerivePrivateKey() {
        DeterministicKey cursor = this.findParentWithPrivKey();
        if (cursor == null) {
            return null;
        }
        return this.derivePrivateKeyDownwards(cursor, cursor.priv.toByteArray());
    }

    private BigInteger derivePrivateKeyDownwards(DeterministicKey cursor, byte[] parentalPrivateKeyBytes) {
        DeterministicKey downCursor = new DeterministicKey(cursor.childNumberPath, cursor.chainCode, cursor.pub, ByteUtils.bytesToBigInteger(parentalPrivateKeyBytes), cursor.parent);
        List path = this.childNumberPath.subList(cursor.getPath().size(), this.childNumberPath.size());
        for (ChildNumber num : path) {
            downCursor = HDKeyDerivation.deriveChildKey(downCursor, num);
        }
        if (!downCursor.pub.equals(this.pub)) {
            throw new KeyCrypterException.PublicPrivateMismatch("Could not decrypt bytes");
        }
        return Objects.requireNonNull(downCursor.priv);
    }

    public DeterministicKey derive(int child) {
        return HDKeyDerivation.deriveChildKey(this, new ChildNumber(child, true));
    }

    @Override
    public BigInteger getPrivKey() {
        BigInteger key = this.findOrDerivePrivateKey();
        Preconditions.checkState(key != null, () -> "private key bytes not available");
        return key;
    }

    byte[] serialize(Network network, boolean pub) {
        return this.serialize(network, pub, ScriptType.P2PKH);
    }

    @Deprecated
    byte[] serialize(NetworkParameters params, boolean pub) {
        return this.serialize(params.network(), pub);
    }

    private byte[] serialize(Network network, boolean pub, ScriptType outputScriptType) {
        NetworkParameters params = NetworkParameters.of(network);
        ByteBuffer ser = ByteBuffer.allocate(78);
        if (outputScriptType == ScriptType.P2PKH) {
            ser.putInt(pub ? params.getBip32HeaderP2PKHpub() : params.getBip32HeaderP2PKHpriv());
        } else if (outputScriptType == ScriptType.P2WPKH) {
            ser.putInt(pub ? params.getBip32HeaderP2WPKHpub() : params.getBip32HeaderP2WPKHpriv());
        } else {
            throw new IllegalStateException(outputScriptType.toString());
        }
        ser.put((byte)this.getDepth());
        ser.putInt(this.getParentFingerprint());
        ser.putInt(this.getChildNumber().i());
        ser.put(this.getChainCode());
        ser.put(pub ? this.getPubKey() : this.getPrivKeyBytes33());
        Preconditions.checkState(ser.position() == 78);
        return ser.array();
    }

    public String serializePubB58(Network network, ScriptType outputScriptType) {
        return DeterministicKey.toBase58(this.serialize(network, true, outputScriptType));
    }

    @Deprecated
    public String serializePubB58(NetworkParameters params, ScriptType outputScriptType) {
        return this.serializePubB58(params.network(), outputScriptType);
    }

    public String serializePrivB58(Network network, ScriptType outputScriptType) {
        return DeterministicKey.toBase58(this.serialize(network, false, outputScriptType));
    }

    @Deprecated
    public String serializePrivB58(NetworkParameters params, ScriptType outputScriptType) {
        return this.serializePrivB58(params.network(), outputScriptType);
    }

    public String serializePubB58(Network network) {
        return DeterministicKey.toBase58(this.serialize(network, true));
    }

    @Deprecated
    public String serializePubB58(NetworkParameters params) {
        return this.serializePubB58(params.network());
    }

    public String serializePrivB58(Network network) {
        return DeterministicKey.toBase58(this.serialize(network, false));
    }

    @Deprecated
    public String serializePrivB58(NetworkParameters params) {
        return this.serializePrivB58(params.network());
    }

    static String toBase58(byte[] ser) {
        return Base58.encode(DeterministicKey.addChecksum(ser));
    }

    public static DeterministicKey deserializeB58(String base58, Network network) {
        return DeterministicKey.deserializeB58(null, base58, network);
    }

    @Deprecated
    public static DeterministicKey deserializeB58(String base58, NetworkParameters params) {
        return DeterministicKey.deserializeB58(base58, params.network());
    }

    public static DeterministicKey deserializeB58(@Nullable DeterministicKey parent, String base58, Network network) {
        return DeterministicKey.deserialize(network, Base58.decodeChecked(base58), parent);
    }

    @Deprecated
    public static DeterministicKey deserializeB58(@Nullable DeterministicKey parent, String base58, NetworkParameters params) {
        return DeterministicKey.deserializeB58(parent, base58, params.network());
    }

    public static DeterministicKey deserialize(Network network, byte[] serializedKey) {
        return DeterministicKey.deserialize(network, serializedKey, null);
    }

    @Deprecated
    public static DeterministicKey deserialize(NetworkParameters params, byte[] serializedKey) {
        return DeterministicKey.deserialize(params.network(), serializedKey);
    }

    public static DeterministicKey deserialize(Network network, byte[] serializedKey, @Nullable DeterministicKey parent) {
        HDPath path;
        boolean priv;
        NetworkParameters params;
        ByteBuffer buffer = ByteBuffer.wrap(serializedKey);
        int header = buffer.getInt();
        boolean pub = header == (params = NetworkParameters.of(network)).getBip32HeaderP2PKHpub() || header == params.getBip32HeaderP2WPKHpub();
        boolean bl = priv = header == params.getBip32HeaderP2PKHpriv() || header == params.getBip32HeaderP2WPKHpriv();
        if (!pub && !priv) {
            throw new IllegalArgumentException("Unknown header bytes: " + DeterministicKey.toBase58(serializedKey).substring(0, 4));
        }
        int depth = buffer.get() & 0xFF;
        int parentFingerprint = buffer.getInt();
        int i2 = buffer.getInt();
        ChildNumber childNumber = new ChildNumber(i2);
        if (parent != null) {
            if (parentFingerprint == 0) {
                throw new IllegalArgumentException("Parent was provided but this key doesn't have one");
            }
            if (parent.getFingerprint() != parentFingerprint) {
                throw new IllegalArgumentException("Parent fingerprints don't match");
            }
            path = parent.getPath().extend(childNumber, new ChildNumber[0]);
            if (path.size() != depth) {
                throw new IllegalArgumentException("Depth does not match");
            }
        } else {
            path = depth >= 1 ? HDPath.M(childNumber) : HDPath.M();
        }
        byte[] chainCode = new byte[32];
        buffer.get(chainCode);
        byte[] data = new byte[33];
        buffer.get(data);
        Preconditions.checkArgument(!buffer.hasRemaining(), () -> "found unexpected data in key");
        if (pub) {
            return new DeterministicKey((List<ChildNumber>)path, chainCode, new LazyECPoint(data), parent, depth, parentFingerprint);
        }
        return new DeterministicKey((List<ChildNumber>)path, chainCode, ByteUtils.bytesToBigInteger(data), parent, depth, parentFingerprint);
    }

    @Deprecated
    public static DeterministicKey deserialize(NetworkParameters params, byte[] serializedKey, @Nullable DeterministicKey parent) {
        return DeterministicKey.deserialize(params.network(), serializedKey, parent);
    }

    @Override
    public Optional<Instant> getCreationTime() {
        if (this.parent != null) {
            return this.parent.getCreationTime();
        }
        return super.getCreationTime();
    }

    @Override
    public void setCreationTime(Instant creationTime) {
        if (this.parent != null) {
            throw new IllegalStateException("Creation time can only be set on root keys.");
        }
        super.setCreationTime(creationTime);
    }

    @Override
    public void clearCreationTime() {
        if (this.parent != null) {
            throw new IllegalStateException("Creation time can only be cleared on root keys.");
        }
        super.clearCreationTime();
    }

    @Override
    @Deprecated
    public void setCreationTimeSeconds(long creationTimeSecs) {
        if (creationTimeSecs > 0L) {
            this.setCreationTime(Instant.ofEpochSecond(creationTimeSecs));
        } else if (creationTimeSecs == 0L) {
            this.clearCreationTime();
        } else {
            throw new IllegalArgumentException("Cannot set creation time to negative value: " + creationTimeSecs);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DeterministicKey other = (DeterministicKey)o;
        return super.equals(other) && Arrays.equals(this.chainCode, other.chainCode) && Objects.equals(this.childNumberPath, other.childNumberPath);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), Arrays.hashCode(this.chainCode), this.childNumberPath);
    }

    @Override
    public String toString() {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this).omitNullValues();
        helper.add("pub", (Object)ByteUtils.formatHex(this.pub.getEncoded()));
        helper.add("chainCode", (Object)ByteUtils.formatHex(this.chainCode));
        helper.add("path", (Object)this.getPathAsString());
        Optional<Instant> creationTime = this.getCreationTime();
        if (!creationTime.isPresent()) {
            helper.add("creationTimeSeconds", (Object)"unknown");
        } else if (this.parent != null) {
            helper.add("creationTimeSeconds", (Object)(creationTime.get().getEpochSecond() + " (inherited)"));
        } else {
            helper.add("creationTimeSeconds", creationTime.get().getEpochSecond());
        }
        helper.add("isEncrypted", this.isEncrypted());
        helper.add("isPubKeyOnly", this.isPubKeyOnly());
        return helper.toString();
    }

    @Override
    public void formatKeyWithAddress(boolean includePrivateKeys, @Nullable AesKey aesKey, StringBuilder builder, Network network, ScriptType outputScriptType, @Nullable String comment) {
        builder.append("  addr:").append(this.toAddress(outputScriptType, network).toString());
        builder.append("  hash160:").append(ByteUtils.formatHex(this.getPubKeyHash()));
        builder.append("  (").append(this.getPathAsString());
        if (comment != null) {
            builder.append(", ").append(comment);
        }
        builder.append(")\n");
        if (includePrivateKeys) {
            builder.append("  ").append(this.toStringWithPrivate(aesKey, network)).append("\n");
        }
    }

    @Override
    @Deprecated
    public void formatKeyWithAddress(boolean includePrivateKeys, @Nullable AesKey aesKey, StringBuilder builder, NetworkParameters params, ScriptType outputScriptType, @Nullable String comment) {
        this.formatKeyWithAddress(includePrivateKeys, aesKey, builder, params.network(), outputScriptType, comment);
    }
}

