/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto;

import com.google.common.base.MoreObjects;
import java.io.ByteArrayOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nullable;
import org.bitcoinj.base.Address;
import org.bitcoinj.base.BitcoinNetwork;
import org.bitcoinj.base.LegacyAddress;
import org.bitcoinj.base.Network;
import org.bitcoinj.base.ScriptType;
import org.bitcoinj.base.SegwitAddress;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.VarInt;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.base.internal.TimeUtils;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.crypto.AesKey;
import org.bitcoinj.crypto.DumpedPrivateKey;
import org.bitcoinj.crypto.EncryptableItem;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.LazyECPoint;
import org.bitcoinj.crypto.SignatureDecodeException;
import org.bitcoinj.crypto.internal.CryptoUtils;
import org.bitcoinj.protobuf.wallet.Protos;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequenceGenerator;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.asn1.x9.X9IntegerConverter;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.ec.CustomNamedCurves;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.signers.ECDSASigner;
import org.bouncycastle.crypto.signers.HMacDSAKCalculator;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.math.ec.FixedPointUtil;
import org.bouncycastle.math.ec.custom.sec.SecP256K1Curve;
import org.bouncycastle.util.Properties;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ECKey
implements EncryptableItem {
    private static final Logger log = LoggerFactory.getLogger(ECKey.class);
    private static final Comparator<byte[]> LEXICOGRAPHICAL_COMPARATOR = ByteUtils.arrayUnsignedComparator();
    public static final Comparator<ECKey> AGE_COMPARATOR = Comparator.comparing(ecKey -> ecKey.getCreationTime().orElse(Instant.EPOCH));
    public static final Comparator<ECKey> PUBKEY_COMPARATOR = Comparator.comparing(ECKey::getPubKey, LEXICOGRAPHICAL_COMPARATOR);
    private static final X9ECParameters CURVE_PARAMS = CustomNamedCurves.getByName("secp256k1");
    static final ECDomainParameters CURVE;
    static final BigInteger HALF_CURVE_ORDER;
    private static final SecureRandom secureRandom;
    @Nullable
    protected final BigInteger priv;
    protected final LazyECPoint pub;
    @Nullable
    protected Instant creationTime = null;
    protected KeyCrypter keyCrypter;
    protected EncryptedData encryptedPrivateKey;
    private byte[] pubKeyHash;
    private static final String BITCOIN_SIGNED_MESSAGE_HEADER = "Bitcoin Signed Message:\n";
    private static final byte[] BITCOIN_SIGNED_MESSAGE_HEADER_BYTES;

    public static ECDomainParameters ecDomainParameters() {
        return CURVE;
    }

    public ECKey() {
        this(secureRandom);
    }

    public ECKey(SecureRandom secureRandom) {
        ECKeyPairGenerator generator = new ECKeyPairGenerator();
        ECKeyGenerationParameters keygenParams = new ECKeyGenerationParameters(CURVE, secureRandom);
        generator.init(keygenParams);
        AsymmetricCipherKeyPair keypair = generator.generateKeyPair();
        ECPrivateKeyParameters privParams = (ECPrivateKeyParameters)keypair.getPrivate();
        ECPublicKeyParameters pubParams = (ECPublicKeyParameters)keypair.getPublic();
        this.priv = privParams.getD();
        this.pub = new LazyECPoint(pubParams.getQ(), true);
        this.creationTime = TimeUtils.currentTime().truncatedTo(ChronoUnit.SECONDS);
    }

    protected ECKey(@Nullable BigInteger priv, ECPoint pub, boolean compressed) {
        this(priv, new LazyECPoint(Objects.requireNonNull(pub), compressed));
    }

    protected ECKey(@Nullable BigInteger priv, LazyECPoint pub) {
        if (priv != null) {
            Preconditions.checkArgument(priv.bitLength() <= 256, () -> "private key exceeds 32 bytes: " + priv.bitLength() + " bits");
            Preconditions.checkArgument(!priv.equals(BigInteger.ZERO));
            Preconditions.checkArgument(!priv.equals(BigInteger.ONE));
        }
        this.priv = priv;
        this.pub = Objects.requireNonNull(pub);
    }

    public static ECKey fromASN1(byte[] asn1privkey) {
        return ECKey.extractKeyFromASN1(asn1privkey);
    }

    public static ECKey fromPrivate(BigInteger privKey) {
        return ECKey.fromPrivate(privKey, true);
    }

    public static ECKey fromPrivate(BigInteger privKey, boolean compressed) {
        ECPoint point = ECKey.publicPointFromPrivate(privKey);
        return new ECKey(privKey, new LazyECPoint(point, compressed));
    }

    public static ECKey fromPrivate(byte[] privKeyBytes) {
        return ECKey.fromPrivate(ByteUtils.bytesToBigInteger(privKeyBytes));
    }

    public static ECKey fromPrivate(byte[] privKeyBytes, boolean compressed) {
        return ECKey.fromPrivate(ByteUtils.bytesToBigInteger(privKeyBytes), compressed);
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(BigInteger priv, ECPoint pub, boolean compressed) {
        return new ECKey(priv, pub, compressed);
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(byte[] priv, byte[] pub) {
        Objects.requireNonNull(priv);
        Objects.requireNonNull(pub);
        return new ECKey(ByteUtils.bytesToBigInteger(priv), new LazyECPoint(pub));
    }

    public static ECKey fromPublicOnly(ECPoint pub, boolean compressed) {
        return new ECKey(null, pub, compressed);
    }

    public static ECKey fromPublicOnly(byte[] pub) {
        return new ECKey(null, new LazyECPoint(pub));
    }

    public static ECKey fromPublicOnly(ECKey key) {
        return ECKey.fromPublicOnly(key.getPubKeyPoint(), key.isCompressed());
    }

    public ECKey decompress() {
        if (!this.pub.isCompressed()) {
            return this;
        }
        return new ECKey(this.priv, new LazyECPoint(this.pub.get(), false));
    }

    public static ECKey fromEncrypted(EncryptedData encryptedPrivateKey, KeyCrypter crypter, byte[] pubKey) {
        ECKey key = ECKey.fromPublicOnly(pubKey);
        key.encryptedPrivateKey = Objects.requireNonNull(encryptedPrivateKey);
        key.keyCrypter = Objects.requireNonNull(crypter);
        return key;
    }

    public boolean isPubKeyOnly() {
        return this.priv == null;
    }

    public boolean hasPrivKey() {
        return this.priv != null;
    }

    public boolean isWatching() {
        return this.isPubKeyOnly() && !this.isEncrypted();
    }

    public byte[] toASN1() {
        try {
            byte[] privKeyBytes = this.getPrivKeyBytes();
            ByteArrayOutputStream baos = new ByteArrayOutputStream(400);
            DERSequenceGenerator seq = new DERSequenceGenerator(baos);
            seq.addObject(new ASN1Integer(1L));
            seq.addObject(new DEROctetString(privKeyBytes));
            seq.addObject(new DERTaggedObject(0, CURVE_PARAMS.toASN1Primitive()));
            seq.addObject(new DERTaggedObject(1, new DERBitString(this.getPubKey())));
            seq.close();
            return baos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] publicKeyFromPrivate(BigInteger privKey, boolean compressed) {
        ECPoint point = ECKey.publicPointFromPrivate(privKey);
        return point.getEncoded(compressed);
    }

    public static ECPoint publicPointFromPrivate(BigInteger privKey) {
        if (privKey.bitLength() > CURVE.getN().bitLength()) {
            privKey = privKey.mod(CURVE.getN());
        }
        return new FixedPointCombMultiplier().multiply(CURVE.getG(), privKey);
    }

    public byte[] getPubKeyHash() {
        if (this.pubKeyHash == null) {
            this.pubKeyHash = CryptoUtils.sha256hash160(this.pub.getEncoded());
        }
        return this.pubKeyHash;
    }

    public byte[] getPubKey() {
        return this.pub.getEncoded();
    }

    public ECPoint getPubKeyPoint() {
        return this.pub.get();
    }

    public BigInteger getPrivKey() {
        if (this.priv == null) {
            throw new MissingPrivateKeyException();
        }
        return this.priv;
    }

    public boolean isCompressed() {
        return this.pub.isCompressed();
    }

    public Address toAddress(ScriptType scriptType, Network network) {
        if (scriptType == ScriptType.P2PKH) {
            return LegacyAddress.fromPubKeyHash(network, this.getPubKeyHash());
        }
        if (scriptType == ScriptType.P2WPKH) {
            Preconditions.checkArgument(this.isCompressed(), () -> "only compressed keys allowed");
            return SegwitAddress.fromHash(network, this.getPubKeyHash());
        }
        throw new IllegalArgumentException(scriptType.toString());
    }

    public ECDSASignature sign(Sha256Hash input) throws KeyCrypterException {
        return this.sign(input, null);
    }

    public ECDSASignature sign(Sha256Hash input, @Nullable AesKey aesKey) throws KeyCrypterException {
        KeyCrypter crypter = this.getKeyCrypter();
        if (crypter != null) {
            if (aesKey == null) {
                throw new KeyIsEncryptedException();
            }
            return this.decrypt(aesKey).sign(input);
        }
        if (this.priv == null) {
            throw new MissingPrivateKeyException();
        }
        return this.doSign(input, this.priv);
    }

    protected ECDSASignature doSign(Sha256Hash input, BigInteger privateKeyForSigning) {
        Objects.requireNonNull(privateKeyForSigning);
        ECDSASigner signer = new ECDSASigner(new HMacDSAKCalculator(new SHA256Digest()));
        ECPrivateKeyParameters privKey = new ECPrivateKeyParameters(privateKeyForSigning, CURVE);
        signer.init(true, privKey);
        BigInteger[] components = signer.generateSignature(input.getBytes());
        return new ECDSASignature(components[0], components[1]).toCanonicalised();
    }

    public static boolean verify(byte[] data, ECDSASignature signature, byte[] pub) {
        ECDSASigner signer = new ECDSASigner();
        ECPublicKeyParameters params = new ECPublicKeyParameters(CURVE.getCurve().decodePoint(pub), CURVE);
        signer.init(false, params);
        try {
            return signer.verifySignature(data, signature.r, signature.s);
        }
        catch (NullPointerException e) {
            log.error("Caught NPE inside bouncy castle", e);
            return false;
        }
    }

    public static boolean verify(byte[] data, byte[] signature, byte[] pub) throws SignatureDecodeException {
        return ECKey.verify(data, ECDSASignature.decodeFromDER(signature), pub);
    }

    public boolean verify(byte[] hash, byte[] signature) throws SignatureDecodeException {
        return ECKey.verify(hash, signature, this.getPubKey());
    }

    public boolean verify(Sha256Hash sigHash, ECDSASignature signature) {
        return ECKey.verify(sigHash.getBytes(), signature, this.getPubKey());
    }

    public void verifyOrThrow(byte[] hash, byte[] signature) throws SignatureDecodeException, SignatureException {
        if (!this.verify(hash, signature)) {
            throw new SignatureException();
        }
    }

    public void verifyOrThrow(Sha256Hash sigHash, ECDSASignature signature) throws SignatureException {
        if (!ECKey.verify(sigHash.getBytes(), signature, this.getPubKey())) {
            throw new SignatureException();
        }
    }

    public static boolean isPubKeyCanonical(byte[] pubkey) {
        if (pubkey.length < 33) {
            return false;
        }
        if (pubkey[0] == 4) {
            if (pubkey.length != 65) {
                return false;
            }
        } else if (pubkey[0] == 2 || pubkey[0] == 3) {
            if (pubkey.length != 33) {
                return false;
            }
        } else {
            return false;
        }
        return true;
    }

    public static boolean isPubKeyCompressed(byte[] encoded) {
        if (encoded.length == 33 && (encoded[0] == 2 || encoded[0] == 3)) {
            return true;
        }
        if (encoded.length == 65 && encoded[0] == 4) {
            return false;
        }
        throw new IllegalArgumentException(ByteUtils.formatHex(encoded));
    }

    private static ECKey extractKeyFromASN1(byte[] asn1privkey) {
        try {
            ASN1InputStream decoder = new ASN1InputStream(asn1privkey);
            DLSequence seq = (DLSequence)decoder.readObject();
            Preconditions.checkArgument(decoder.readObject() == null, () -> "input contains extra bytes");
            decoder.close();
            Preconditions.checkArgument(seq.size() == 4, () -> "input does not appear to be an ASN.1 OpenSSL EC private key");
            Preconditions.checkArgument(((ASN1Integer)seq.getObjectAt(0)).getValue().equals(BigInteger.ONE), () -> "input is of wrong version");
            byte[] privbits = ((ASN1OctetString)seq.getObjectAt(1)).getOctets();
            BigInteger privkey = ByteUtils.bytesToBigInteger(privbits);
            ASN1TaggedObject pubkey = (ASN1TaggedObject)seq.getObjectAt(3);
            Preconditions.checkArgument(pubkey.getTagNo() == 1, () -> "input has 'publicKey' with bad tag number");
            Preconditions.checkArgument(pubkey.getTagClass() == 128, () -> "input has 'publicKey' with bad tag class");
            byte[] pubbits = ((DERBitString)pubkey.getBaseObject()).getBytes();
            Preconditions.checkArgument(pubbits.length == 33 || pubbits.length == 65, () -> "input has 'publicKey' with invalid length");
            int encoding = pubbits[0] & 0xFF;
            Preconditions.checkArgument(encoding >= 2 && encoding <= 4, () -> "input has 'publicKey' with invalid encoding");
            ECKey key = ECKey.fromPrivate(privkey, ECKey.isPubKeyCompressed(pubbits));
            Preconditions.checkArgument(Arrays.equals(key.getPubKey(), pubbits), () -> "public key in ASN.1 structure does not match private key.");
            return key;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Deprecated
    public String signMessage(String message) throws KeyCrypterException {
        return this.signMessage(message, null, ScriptType.P2PKH);
    }

    public String signMessage(String message, ScriptType scriptType) throws KeyCrypterException {
        return this.signMessage(message, null, scriptType);
    }

    @Deprecated
    public String signMessage(String message, @Nullable AesKey aesKey) throws KeyCrypterException {
        return this.signMessage(message, aesKey, ScriptType.P2PKH);
    }

    public String signMessage(String message, @Nullable AesKey aesKey, ScriptType scriptType) throws KeyCrypterException {
        int headerByte;
        if (!(this.isCompressed() || scriptType != ScriptType.P2WPKH && scriptType != ScriptType.P2SH)) {
            throw new IllegalArgumentException("Segwit P2WPKH and P2SH-P2WPKH script types only can be used with compressed keys. See BIP 141.");
        }
        byte[] data = ECKey.formatMessageForSigning(message);
        Sha256Hash hash = Sha256Hash.twiceOf(data);
        ECDSASignature sig = this.sign(hash, aesKey);
        byte recId = this.findRecoveryId(hash, sig);
        switch (scriptType) {
            case P2PKH: {
                headerByte = recId + 27 + (this.isCompressed() ? 4 : 0);
                break;
            }
            case P2SH: {
                headerByte = recId + 35;
                break;
            }
            case P2WPKH: {
                headerByte = recId + 39;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported script type for message signing.");
            }
        }
        byte[] sigData = new byte[65];
        sigData[0] = (byte)headerByte;
        System.arraycopy(ByteUtils.bigIntegerToBytes(sig.r, 32), 0, sigData, 1, 32);
        System.arraycopy(ByteUtils.bigIntegerToBytes(sig.s, 32), 0, sigData, 33, 32);
        return new String(Base64.encode(sigData), StandardCharsets.UTF_8);
    }

    public static ECKey signedMessageToKey(String message, String signatureBase64) throws SignatureException {
        byte[] signatureEncoded;
        try {
            signatureEncoded = Base64.decode(signatureBase64);
        }
        catch (RuntimeException e) {
            throw new SignatureException("Could not decode base64", e);
        }
        if (signatureEncoded.length < 65) {
            throw new SignatureException("Signature truncated, expected 65 bytes and got " + signatureEncoded.length);
        }
        int header = signatureEncoded[0] & 0xFF;
        if (header < 27 || header > 42) {
            throw new SignatureException("Header byte out of range: " + header);
        }
        BigInteger r = ByteUtils.bytesToBigInteger(Arrays.copyOfRange(signatureEncoded, 1, 33));
        BigInteger s = ByteUtils.bytesToBigInteger(Arrays.copyOfRange(signatureEncoded, 33, 65));
        ECDSASignature sig = new ECDSASignature(r, s);
        byte[] messageBytes = ECKey.formatMessageForSigning(message);
        Sha256Hash messageHash = Sha256Hash.twiceOf(messageBytes);
        boolean compressed = false;
        if (header >= 39) {
            header -= 12;
            compressed = true;
        } else if (header >= 35) {
            header -= 8;
            compressed = true;
        } else if (header >= 31) {
            compressed = true;
            header -= 4;
        }
        int recId = header - 27;
        ECKey key = ECKey.recoverFromSignature(recId, sig, messageHash, compressed);
        if (key == null) {
            throw new SignatureException("Could not recover public key from signature");
        }
        return key;
    }

    @Deprecated
    public void verifyMessage(String message, String signatureBase64) throws SignatureException {
        ECKey key = ECKey.signedMessageToKey(message, signatureBase64);
        if (!key.pub.equals(this.pub)) {
            throw new SignatureException("Signature did not match for message");
        }
    }

    public byte findRecoveryId(Sha256Hash hash, ECDSASignature sig) {
        int recId = -1;
        for (int i2 = 0; i2 < 4; i2 = (int)((byte)(i2 + 1))) {
            ECKey k = ECKey.recoverFromSignature(i2, sig, hash, this.isCompressed());
            if (k == null || !k.pub.equals(this.pub)) continue;
            recId = i2;
            break;
        }
        if (recId == -1) {
            throw new RuntimeException("Could not construct a recoverable key. This should never happen.");
        }
        return (byte)recId;
    }

    @Nullable
    public static ECKey recoverFromSignature(int recId, ECDSASignature sig, Sha256Hash message, boolean compressed) {
        Preconditions.checkArgument(recId >= 0, () -> "recId must be positive");
        Preconditions.checkArgument(sig.r.signum() >= 0, () -> "r must be positive");
        Preconditions.checkArgument(sig.s.signum() >= 0, () -> "s must be positive");
        Objects.requireNonNull(message);
        BigInteger n = CURVE.getN();
        BigInteger i2 = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i2.multiply(n));
        BigInteger prime = SecP256K1Curve.q;
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = ECKey.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = message.toBigInteger();
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint q = ECAlgorithms.sumOfTwoMultiplies(CURVE.getG(), eInvrInv, R, srInv);
        return ECKey.fromPublicOnly(q, compressed);
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength(CURVE.getCurve()));
        compEnc[0] = (byte)(yBit ? 3 : 2);
        return CURVE.getCurve().decodePoint(compEnc);
    }

    public byte[] getPrivKeyBytes() {
        return ByteUtils.bigIntegerToBytes(this.getPrivKey(), 32);
    }

    public DumpedPrivateKey getPrivateKeyEncoded(Network network) {
        return new DumpedPrivateKey(network, this.getPrivKeyBytes(), this.isCompressed());
    }

    @Deprecated
    public DumpedPrivateKey getPrivateKeyEncoded(NetworkParameters params) {
        return this.getPrivateKeyEncoded(params.network());
    }

    @Override
    public Optional<Instant> getCreationTime() {
        return Optional.ofNullable(this.creationTime);
    }

    public void setCreationTime(Instant creationTime) {
        this.creationTime = Objects.requireNonNull(creationTime);
    }

    public void clearCreationTime() {
        this.creationTime = null;
    }

    @Deprecated
    public void setCreationTimeSeconds(long creationTimeSecs) {
        if (creationTimeSecs > 0L) {
            this.setCreationTime(Instant.ofEpochSecond(creationTimeSecs));
        } else if (creationTimeSecs == 0L) {
            this.clearCreationTime();
        } else {
            throw new IllegalArgumentException("Cannot set creation time to negative value: " + creationTimeSecs);
        }
    }

    public ECKey encrypt(KeyCrypter keyCrypter, AesKey aesKey) throws KeyCrypterException {
        Objects.requireNonNull(keyCrypter);
        byte[] privKeyBytes = this.getPrivKeyBytes();
        EncryptedData encryptedPrivateKey = keyCrypter.encrypt(privKeyBytes, aesKey);
        ECKey result = ECKey.fromEncrypted(encryptedPrivateKey, keyCrypter, this.getPubKey());
        if (this.creationTime != null) {
            result.setCreationTime(this.creationTime);
        } else {
            result.clearCreationTime();
        }
        return result;
    }

    public ECKey decrypt(KeyCrypter keyCrypter, AesKey aesKey) throws KeyCrypterException {
        Objects.requireNonNull(keyCrypter);
        if (this.keyCrypter != null && !this.keyCrypter.equals(keyCrypter)) {
            throw new KeyCrypterException("The keyCrypter being used to decrypt the key is different to the one that was used to encrypt it");
        }
        Preconditions.checkState(this.encryptedPrivateKey != null, () -> "this key is not encrypted");
        byte[] unencryptedPrivateKey = keyCrypter.decrypt(this.encryptedPrivateKey, aesKey);
        if (unencryptedPrivateKey.length != 32) {
            throw new KeyCrypterException.InvalidCipherText("Decrypted key must be 32 bytes long, but is " + unencryptedPrivateKey.length);
        }
        ECKey key = ECKey.fromPrivate(unencryptedPrivateKey, this.isCompressed());
        if (!Arrays.equals(key.getPubKey(), this.getPubKey())) {
            throw new KeyCrypterException("Provided AES key is wrong");
        }
        if (this.creationTime != null) {
            key.setCreationTime(this.creationTime);
        } else {
            key.clearCreationTime();
        }
        return key;
    }

    public ECKey decrypt(AesKey aesKey) throws KeyCrypterException {
        KeyCrypter crypter = this.getKeyCrypter();
        if (crypter == null) {
            throw new KeyCrypterException("No key crypter available");
        }
        return this.decrypt(crypter, aesKey);
    }

    public ECKey maybeDecrypt(@Nullable AesKey aesKey) throws KeyCrypterException {
        return this.isEncrypted() && aesKey != null ? this.decrypt(aesKey) : this;
    }

    public static boolean encryptionIsReversible(ECKey originalKey, ECKey encryptedKey, KeyCrypter keyCrypter, AesKey aesKey) {
        try {
            ECKey rebornUnencryptedKey = encryptedKey.decrypt(keyCrypter, aesKey);
            byte[] originalPrivateKeyBytes = originalKey.getPrivKeyBytes();
            byte[] rebornKeyBytes = rebornUnencryptedKey.getPrivKeyBytes();
            if (!Arrays.equals(originalPrivateKeyBytes, rebornKeyBytes)) {
                log.error("The check that encryption could be reversed failed for {}", (Object)originalKey);
                return false;
            }
            return true;
        }
        catch (KeyCrypterException kce) {
            log.error(kce.getMessage());
            return false;
        }
    }

    @Override
    public boolean isEncrypted() {
        return this.keyCrypter != null && this.encryptedPrivateKey != null && this.encryptedPrivateKey.encryptedBytes.length > 0;
    }

    @Override
    @Nullable
    public Protos.Wallet.EncryptionType getEncryptionType() {
        return this.keyCrypter != null ? this.keyCrypter.getUnderstoodEncryptionType() : Protos.Wallet.EncryptionType.UNENCRYPTED;
    }

    @Override
    @Nullable
    public byte[] getSecretBytes() {
        if (this.hasPrivKey()) {
            return this.getPrivKeyBytes();
        }
        return null;
    }

    @Override
    @Nullable
    public EncryptedData getEncryptedData() {
        return this.getEncryptedPrivateKey();
    }

    @Nullable
    public EncryptedData getEncryptedPrivateKey() {
        return this.encryptedPrivateKey;
    }

    @Nullable
    public KeyCrypter getKeyCrypter() {
        return this.keyCrypter;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof ECKey)) {
            return false;
        }
        ECKey other = (ECKey)o;
        return Objects.equals(this.priv, other.priv) && Objects.equals(this.pub, other.pub) && Objects.equals(this.creationTime, other.creationTime) && Objects.equals(this.keyCrypter, other.keyCrypter) && Objects.equals(this.encryptedPrivateKey, other.encryptedPrivateKey);
    }

    public int hashCode() {
        return this.pub.hashCode();
    }

    public String toString() {
        return this.toString(false, null, BitcoinNetwork.MAINNET);
    }

    public String toStringWithPrivate(@Nullable AesKey aesKey, Network network) {
        return this.toString(true, aesKey, network);
    }

    @Deprecated
    public String toStringWithPrivate(@Nullable AesKey aesKey, NetworkParameters params) {
        return this.toStringWithPrivate(aesKey, params.network());
    }

    public String getPrivateKeyAsHex() {
        return ByteUtils.formatHex(this.getPrivKeyBytes());
    }

    public String getPublicKeyAsHex() {
        return ByteUtils.formatHex(this.pub.getEncoded());
    }

    public String getPrivateKeyAsWiF(Network network) {
        return this.getPrivateKeyEncoded(network).toString();
    }

    @Deprecated
    public String getPrivateKeyAsWiF(NetworkParameters params) {
        return this.getPrivateKeyAsWiF(params.network());
    }

    private String toString(boolean includePrivate, @Nullable AesKey aesKey, Network network) {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this).omitNullValues();
        helper.add("pub HEX", (Object)this.getPublicKeyAsHex());
        if (includePrivate) {
            ECKey decryptedKey = this.isEncrypted() ? this.decrypt(Objects.requireNonNull(aesKey)) : this;
            try {
                helper.add("priv HEX", (Object)decryptedKey.getPrivateKeyAsHex());
                helper.add("priv WIF", (Object)decryptedKey.getPrivateKeyAsWiF(network));
            }
            catch (IllegalStateException illegalStateException) {
            }
            catch (Exception e) {
                String message = e.getMessage();
                helper.add("priv EXCEPTION", (Object)(e.getClass().getName() + (message != null ? ": " + message : "")));
            }
        }
        if (this.creationTime != null) {
            helper.add("creationTime", (Object)this.creationTime);
        }
        helper.add("keyCrypter", (Object)this.keyCrypter);
        if (includePrivate) {
            helper.add("encryptedPrivateKey", (Object)this.encryptedPrivateKey);
        }
        helper.add("isEncrypted", this.isEncrypted());
        helper.add("isPubKeyOnly", this.isPubKeyOnly());
        return helper.toString();
    }

    @Deprecated
    private String toString(boolean includePrivate, @Nullable AesKey aesKey, @Nullable NetworkParameters params) {
        Network network = params != null ? params.network() : BitcoinNetwork.MAINNET;
        return this.toString(includePrivate, aesKey, network);
    }

    public void formatKeyWithAddress(boolean includePrivateKeys, @Nullable AesKey aesKey, StringBuilder builder, Network network, ScriptType outputScriptType, @Nullable String comment) {
        builder.append("  addr:");
        if (outputScriptType != null) {
            builder.append(this.toAddress(outputScriptType, network));
        } else {
            builder.append(this.toAddress(ScriptType.P2PKH, network));
            if (this.isCompressed()) {
                builder.append(',').append(this.toAddress(ScriptType.P2WPKH, network));
            }
        }
        if (!this.isCompressed()) {
            builder.append("  UNCOMPRESSED");
        }
        builder.append("  hash160:");
        builder.append(ByteUtils.formatHex(this.getPubKeyHash()));
        if (this.creationTime != null) {
            builder.append("  creationTime:").append(this.creationTime).append(" [").append(TimeUtils.dateTimeFormat(this.creationTime)).append("]");
        }
        if (comment != null) {
            builder.append("  (").append(comment).append(")");
        }
        builder.append("\n");
        if (includePrivateKeys) {
            builder.append("  ");
            builder.append(this.toStringWithPrivate(aesKey, network));
            builder.append("\n");
        }
    }

    @Deprecated
    public void formatKeyWithAddress(boolean includePrivateKeys, @Nullable AesKey aesKey, StringBuilder builder, NetworkParameters params, ScriptType outputScriptType, @Nullable String comment) {
        this.formatKeyWithAddress(includePrivateKeys, aesKey, builder, params.network(), outputScriptType, comment);
    }

    private static byte[] formatMessageForSigning(String message) {
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            bos.write(BITCOIN_SIGNED_MESSAGE_HEADER_BYTES.length);
            bos.write(BITCOIN_SIGNED_MESSAGE_HEADER_BYTES);
            byte[] messageBytes = message.getBytes(StandardCharsets.UTF_8);
            VarInt size = VarInt.of(messageBytes.length);
            bos.write(size.serialize());
            bos.write(messageBytes);
            return bos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static {
        FixedPointUtil.precompute(CURVE_PARAMS.getG());
        CURVE = new ECDomainParameters(CURVE_PARAMS.getCurve(), CURVE_PARAMS.getG(), CURVE_PARAMS.getN(), CURVE_PARAMS.getH());
        HALF_CURVE_ORDER = CURVE_PARAMS.getN().shiftRight(1);
        secureRandom = new SecureRandom();
        BITCOIN_SIGNED_MESSAGE_HEADER_BYTES = BITCOIN_SIGNED_MESSAGE_HEADER.getBytes(StandardCharsets.UTF_8);
    }

    public static class MissingPrivateKeyException
    extends RuntimeException {
    }

    public static class ECDSASignature {
        public final BigInteger r;
        public final BigInteger s;

        public ECDSASignature(BigInteger r, BigInteger s) {
            this.r = r;
            this.s = s;
        }

        public boolean isCanonical() {
            return this.s.compareTo(HALF_CURVE_ORDER) <= 0;
        }

        public ECDSASignature toCanonicalised() {
            if (!this.isCanonical()) {
                return new ECDSASignature(this.r, CURVE.getN().subtract(this.s));
            }
            return this;
        }

        public byte[] encodeToDER() {
            try {
                return this.derByteStream().toByteArray();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        public static ECDSASignature decodeFromDER(byte[] bytes) throws SignatureDecodeException {
            FilterInputStream decoder = null;
            try {
                ASN1Integer s;
                ASN1Integer r;
                Properties.setThreadOverride("org.bouncycastle.asn1.allow_unsafe_integer", true);
                decoder = new ASN1InputStream(bytes);
                ASN1Primitive seqObj = ((ASN1InputStream)decoder).readObject();
                if (seqObj == null) {
                    throw new SignatureDecodeException("Reached past end of ASN.1 stream.");
                }
                if (!(seqObj instanceof DLSequence)) {
                    throw new SignatureDecodeException("Read unexpected class: " + seqObj.getClass().getName());
                }
                DLSequence seq = (DLSequence)seqObj;
                try {
                    r = (ASN1Integer)seq.getObjectAt(0);
                    s = (ASN1Integer)seq.getObjectAt(1);
                }
                catch (ClassCastException e) {
                    throw new SignatureDecodeException(e);
                }
                ECDSASignature eCDSASignature = new ECDSASignature(r.getPositiveValue(), s.getPositiveValue());
                return eCDSASignature;
            }
            catch (IOException e) {
                throw new SignatureDecodeException(e);
            }
            finally {
                if (decoder != null) {
                    try {
                        decoder.close();
                    }
                    catch (IOException iOException) {}
                }
                Properties.removeThreadOverride("org.bouncycastle.asn1.allow_unsafe_integer");
            }
        }

        protected ByteArrayOutputStream derByteStream() throws IOException {
            ByteArrayOutputStream bos = new ByteArrayOutputStream(72);
            DERSequenceGenerator seq = new DERSequenceGenerator(bos);
            seq.addObject(new ASN1Integer(this.r));
            seq.addObject(new ASN1Integer(this.s));
            seq.close();
            return bos;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ECDSASignature other = (ECDSASignature)o;
            return this.r.equals(other.r) && this.s.equals(other.s);
        }

        public int hashCode() {
            return Objects.hash(this.r, this.s);
        }
    }

    public static class KeyIsEncryptedException
    extends MissingPrivateKeyException {
    }
}

