/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto.utils;

import java.security.SignatureException;
import java.util.Arrays;
import org.bitcoinj.base.Address;
import org.bitcoinj.base.LegacyAddress;
import org.bitcoinj.base.ScriptType;
import org.bitcoinj.base.SegwitAddress;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.crypto.internal.CryptoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageVerifyUtils {
    private static final Logger log = LoggerFactory.getLogger(MessageVerifyUtils.class);
    public static final String SIGNATURE_FAILED_ERROR_MESSAGE = "Signature did not match for message";

    private MessageVerifyUtils() {
    }

    public static void verifyMessage(Address address, String message, String signatureBase64) throws SignatureException {
        try {
            ScriptType scriptType = address.getOutputScriptType();
            switch (scriptType) {
                case P2PKH: 
                case P2WPKH: {
                    MessageVerifyUtils.comparePubKeyHash(address, message, signatureBase64);
                    break;
                }
                case P2SH: {
                    MessageVerifyUtils.compareP2SHScriptHashDerivedFromPubKey((LegacyAddress)address, message, signatureBase64);
                    break;
                }
                default: {
                    throw new SignatureException(SIGNATURE_FAILED_ERROR_MESSAGE);
                }
            }
        }
        catch (SignatureException se) {
            throw se;
        }
        catch (Exception e) {
            log.warn("verifying of message signature failed with exception", e);
            throw new SignatureException(SIGNATURE_FAILED_ERROR_MESSAGE);
        }
    }

    private static void comparePubKeyHash(Address address, String message, String signatureBase64) throws SignatureException {
        if (address instanceof SegwitAddress && ((SegwitAddress)address).getWitnessVersion() > 0) {
            throw new SignatureException("Message verification currently only supports segwit version 0");
        }
        byte[] pubKeyHashFromSignature = ECKey.signedMessageToKey(message, signatureBase64).getPubKeyHash();
        byte[] pubKeyHashFromAddress = address.getHash();
        if (!Arrays.equals(pubKeyHashFromAddress, pubKeyHashFromSignature)) {
            throw new SignatureException(SIGNATURE_FAILED_ERROR_MESSAGE);
        }
    }

    private static void compareP2SHScriptHashDerivedFromPubKey(LegacyAddress address, String message, String signatureBase64) throws SignatureException {
        byte[] segwitScriptFromSignaturePubKey;
        byte[] scriptHashDerivedFromSig;
        byte[] pubKeyHashFromSignature = ECKey.signedMessageToKey(message, signatureBase64).getPubKeyHash();
        byte[] scriptHashFromAddress = address.getHash();
        if (!Arrays.equals(scriptHashFromAddress, scriptHashDerivedFromSig = CryptoUtils.sha256hash160(segwitScriptFromSignaturePubKey = MessageVerifyUtils.wrapPubKeyHashInSegwitScript(pubKeyHashFromSignature)))) {
            throw new SignatureException(SIGNATURE_FAILED_ERROR_MESSAGE);
        }
    }

    private static byte[] wrapPubKeyHashInSegwitScript(byte[] pubKeyHash) {
        byte[] scriptPubKeyPrefix = new byte[]{0, 20};
        return ByteUtils.concat(scriptPubKeyPrefix, pubKeyHash);
    }
}

