/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.script;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import javax.annotation.Nullable;
import org.bitcoinj.base.Address;
import org.bitcoinj.base.Coin;
import org.bitcoinj.base.LegacyAddress;
import org.bitcoinj.base.Network;
import org.bitcoinj.base.ScriptType;
import org.bitcoinj.base.SegwitAddress;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.VarInt;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.InternalUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.core.LockTime;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Transaction;
import org.bitcoinj.core.TransactionWitness;
import org.bitcoinj.core.Utils;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.crypto.SignatureDecodeException;
import org.bitcoinj.crypto.TransactionSignature;
import org.bitcoinj.crypto.internal.CryptoUtils;
import org.bitcoinj.script.ScriptBuilder;
import org.bitcoinj.script.ScriptChunk;
import org.bitcoinj.script.ScriptError;
import org.bitcoinj.script.ScriptException;
import org.bitcoinj.script.ScriptOpCodes;
import org.bitcoinj.script.ScriptPattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Script {
    public static final EnumSet<VerifyFlag> ALL_VERIFY_FLAGS = EnumSet.allOf(VerifyFlag.class);
    private static final BigInteger LOCKTIME_THRESHOLD_BIG = BigInteger.valueOf(500000000L);
    private static final Logger log = LoggerFactory.getLogger(Script.class);
    public static final int MAX_SCRIPT_ELEMENT_SIZE = 520;
    private static final int MAX_OPS_PER_SCRIPT = 201;
    private static final int MAX_STACK_SIZE = 1000;
    private static final int MAX_PUBKEYS_PER_MULTISIG = 20;
    private static final int MAX_SCRIPT_SIZE = 10000;
    public static final int SIG_SIZE = 75;
    public static final int MAX_P2SH_SIGOPS = 15;
    private final List<ScriptChunk> chunks;
    @Nullable
    private final byte[] program;
    @Nullable
    private final Instant creationTime;
    private static final ScriptChunk[] STANDARD_TRANSACTION_SCRIPT_CHUNKS = new ScriptChunk[]{new ScriptChunk(118, null), new ScriptChunk(169, null), new ScriptChunk(136, null), new ScriptChunk(172, null)};

    public static Script of(List<ScriptChunk> chunks) {
        return Script.of(chunks, null);
    }

    public static Script of(List<ScriptChunk> chunks, Instant creationTime) {
        return new Script(chunks, creationTime);
    }

    public static Script parse(byte[] program) throws ScriptException {
        return Script.parse(program, null);
    }

    public static Script parse(byte[] program, Instant creationTime) throws ScriptException {
        return new Script(program, creationTime);
    }

    private static List<ScriptChunk> parseIntoChunks(byte[] program) throws ScriptException {
        ArrayList<ScriptChunk> chunks = new ArrayList<ScriptChunk>();
        Script.parseIntoChunksPartial(program, chunks);
        return Collections.unmodifiableList(chunks);
    }

    private static void parseIntoChunksPartial(byte[] program, List<ScriptChunk> chunks) throws ScriptException {
        ByteArrayInputStream bis = new ByteArrayInputStream(program);
        while (bis.available() > 0) {
            ScriptChunk chunk;
            int opcode = bis.read();
            long dataToRead = -1L;
            if (opcode >= 0 && opcode < 76) {
                dataToRead = opcode;
            } else if (opcode == 76) {
                if (bis.available() < 1) {
                    throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Unexpected end of script");
                }
                dataToRead = bis.read();
            } else if (opcode == 77) {
                if (bis.available() < 2) {
                    throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Unexpected end of script");
                }
                dataToRead = ByteUtils.readUint16(bis);
            } else if (opcode == 78) {
                if (bis.available() < 4) {
                    throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Unexpected end of script");
                }
                dataToRead = ByteUtils.readUint32(bis);
            }
            if (dataToRead == -1L) {
                chunk = new ScriptChunk(opcode, null);
            } else {
                if (dataToRead > (long)bis.available()) {
                    throw new ScriptException(ScriptError.SCRIPT_ERR_BAD_OPCODE, "Push of data element that is larger than remaining data: " + dataToRead + " vs " + bis.available());
                }
                byte[] data = new byte[(int)dataToRead];
                Preconditions.checkState(dataToRead == 0L || (long)bis.read(data, 0, (int)dataToRead) == dataToRead);
                chunk = new ScriptChunk(opcode, data);
            }
            for (ScriptChunk c : STANDARD_TRANSACTION_SCRIPT_CHUNKS) {
                if (!c.equals(chunk)) continue;
                chunk = c;
            }
            chunks.add(chunk);
        }
    }

    @Deprecated
    public Script(byte[] program) {
        this(program, null);
    }

    @Deprecated
    public Script(byte[] program, long creationTimeInSeconds) {
        this(program, creationTimeInSeconds != 0L ? Instant.ofEpochSecond(creationTimeInSeconds) : null);
    }

    private Script(byte[] program, @Nullable Instant creationTime) {
        Objects.requireNonNull(program);
        this.program = Arrays.copyOf(program, program.length);
        this.chunks = Script.parseIntoChunks(this.program);
        this.creationTime = creationTime;
    }

    private Script(List<ScriptChunk> chunks, @Nullable Instant creationTime) {
        Objects.requireNonNull(chunks);
        this.program = null;
        this.chunks = Collections.unmodifiableList(new ArrayList<ScriptChunk>(chunks));
        this.creationTime = creationTime;
    }

    public byte[] program() {
        if (this.program != null) {
            return Arrays.copyOf(this.program, this.program.length);
        }
        int size = this.chunks.stream().mapToInt(ScriptChunk::size).sum();
        ByteBuffer buf = ByteBuffer.allocate(size);
        this.chunks.forEach(chunk -> buf.put(chunk.toByteArray()));
        return buf.array();
    }

    @Deprecated
    public byte[] getProgram() {
        return this.program();
    }

    public List<ScriptChunk> chunks() {
        return Collections.unmodifiableList(this.chunks);
    }

    @Deprecated
    public List<ScriptChunk> getChunks() {
        return this.chunks();
    }

    public Optional<Instant> creationTime() {
        return Optional.ofNullable(this.creationTime);
    }

    @Deprecated
    public long getCreationTimeSeconds() {
        return this.creationTime().orElse(Instant.EPOCH).getEpochSecond();
    }

    public String toString() {
        if (!this.chunks.isEmpty()) {
            return InternalUtils.SPACE_JOINER.join(this.chunks);
        }
        return "<empty>";
    }

    public byte[] getPubKeyHash() throws ScriptException {
        if (ScriptPattern.isP2PKH(this)) {
            return ScriptPattern.extractHashFromP2PKH(this);
        }
        if (ScriptPattern.isP2SH(this)) {
            return ScriptPattern.extractHashFromP2SH(this);
        }
        if (ScriptPattern.isP2WH(this)) {
            return ScriptPattern.extractHashFromP2WH(this);
        }
        throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Script not in the standard scriptPubKey form");
    }

    public Address getToAddress(Network network) throws ScriptException {
        return this.getToAddress(network, false);
    }

    @Deprecated
    public Address getToAddress(NetworkParameters params) throws ScriptException {
        return this.getToAddress(params.network(), false);
    }

    public Address getToAddress(Network network, boolean forcePayToPubKey) throws ScriptException {
        if (ScriptPattern.isP2PKH(this)) {
            return LegacyAddress.fromPubKeyHash(network, ScriptPattern.extractHashFromP2PKH(this));
        }
        if (ScriptPattern.isP2SH(this)) {
            return LegacyAddress.fromScriptHash(network, ScriptPattern.extractHashFromP2SH(this));
        }
        if (forcePayToPubKey && ScriptPattern.isP2PK(this)) {
            return ECKey.fromPublicOnly(ScriptPattern.extractKeyFromP2PK(this)).toAddress(ScriptType.P2PKH, network);
        }
        if (ScriptPattern.isP2WH(this)) {
            return SegwitAddress.fromHash(network, ScriptPattern.extractHashFromP2WH(this));
        }
        if (ScriptPattern.isP2TR(this)) {
            return SegwitAddress.fromProgram(network, 1, ScriptPattern.extractOutputKeyFromP2TR(this));
        }
        throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Cannot cast this script to an address");
    }

    @Deprecated
    public Address getToAddress(NetworkParameters params, boolean forcePayToPubKey) throws ScriptException {
        return this.getToAddress(params.network(), forcePayToPubKey);
    }

    public static void writeBytes(OutputStream os, byte[] buf) throws IOException {
        if (buf.length < 76) {
            os.write(buf.length);
            os.write(buf);
        } else if (buf.length < 256) {
            os.write(76);
            os.write(buf.length);
            os.write(buf);
        } else if (buf.length < 65536) {
            os.write(77);
            ByteUtils.writeInt16LE(buf.length, os);
            os.write(buf);
        } else {
            throw new RuntimeException("Unimplemented");
        }
    }

    public static byte[] createMultiSigOutputScript(int threshold, List<ECKey> pubkeys) {
        Preconditions.checkArgument(threshold > 0);
        Preconditions.checkArgument(threshold <= pubkeys.size());
        Preconditions.checkArgument(pubkeys.size() <= 16);
        if (pubkeys.size() > 3) {
            log.warn("Creating a multi-signature output that is non-standard: {} pubkeys, should be <= 3", (Object)pubkeys.size());
        }
        try {
            ByteArrayOutputStream bits = new ByteArrayOutputStream();
            bits.write(Script.encodeToOpN(threshold));
            for (ECKey key : pubkeys) {
                Script.writeBytes(bits, key.getPubKey());
            }
            bits.write(Script.encodeToOpN(pubkeys.size()));
            bits.write(174);
            return bits.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] createInputScript(byte[] signature, byte[] pubkey) {
        try {
            ByteArrayOutputStream bits = new ByteArrayOutputStream(signature.length + pubkey.length + 2);
            Script.writeBytes(bits, signature);
            Script.writeBytes(bits, pubkey);
            return bits.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] createInputScript(byte[] signature) {
        try {
            ByteArrayOutputStream bits = new ByteArrayOutputStream(signature.length + 2);
            Script.writeBytes(bits, signature);
            return bits.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public Script createEmptyInputScript(@Nullable ECKey key, @Nullable Script redeemScript) {
        if (ScriptPattern.isP2PKH(this)) {
            Preconditions.checkArgument(key != null, () -> "key required to create P2PKH input script");
            return ScriptBuilder.createInputScript(null, key);
        }
        if (ScriptPattern.isP2WPKH(this)) {
            return ScriptBuilder.createEmpty();
        }
        if (ScriptPattern.isP2PK(this)) {
            return ScriptBuilder.createInputScript(null);
        }
        if (ScriptPattern.isP2SH(this)) {
            Preconditions.checkArgument(redeemScript != null, () -> "redeem script required to create P2SH input script");
            return ScriptBuilder.createP2SHMultiSigInputScript(null, redeemScript);
        }
        throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Do not understand script type: " + this);
    }

    public Script getScriptSigWithSignature(Script scriptSig, byte[] sigBytes, int index) {
        int sigsPrefixCount = 0;
        int sigsSuffixCount = 0;
        if (ScriptPattern.isP2SH(this)) {
            sigsPrefixCount = 1;
            sigsSuffixCount = 1;
        } else if (ScriptPattern.isSentToMultisig(this)) {
            sigsPrefixCount = 1;
        } else if (ScriptPattern.isP2PKH(this)) {
            sigsSuffixCount = 1;
        }
        return ScriptBuilder.updateScriptWithSignature(scriptSig, sigBytes, index, sigsPrefixCount, sigsSuffixCount);
    }

    public int getSigInsertionIndex(Sha256Hash hash, ECKey signingKey) {
        List<ScriptChunk> existingChunks = this.chunks.subList(1, this.chunks.size() - 1);
        ScriptChunk redeemScriptChunk = this.chunks.get(this.chunks.size() - 1);
        Objects.requireNonNull(redeemScriptChunk.data);
        Script redeemScript = Script.parse(redeemScriptChunk.data);
        int sigCount = 0;
        int myIndex = redeemScript.findKeyInRedeem(signingKey);
        for (ScriptChunk chunk : existingChunks) {
            if (chunk.opcode == 0) continue;
            Objects.requireNonNull(chunk.data);
            try {
                if (myIndex < redeemScript.findSigInRedeem(chunk.data, hash)) {
                    return sigCount;
                }
            }
            catch (SignatureDecodeException signatureDecodeException) {
                // empty catch block
            }
            ++sigCount;
        }
        return sigCount;
    }

    private int findKeyInRedeem(ECKey key) {
        Preconditions.checkArgument(this.chunks.get(0).isOpCode());
        int numKeys = Script.decodeFromOpN(this.chunks.get((int)(this.chunks.size() - 2)).opcode);
        for (int i2 = 0; i2 < numKeys; ++i2) {
            if (!Arrays.equals(this.chunks.get((int)(1 + i2)).data, key.getPubKey())) continue;
            return i2;
        }
        throw new IllegalStateException("Could not find matching key " + key.toString() + " in script " + this);
    }

    public List<ECKey> getPubKeys() {
        if (!ScriptPattern.isSentToMultisig(this)) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Only usable for multisig scripts.");
        }
        ArrayList<ECKey> result = new ArrayList<ECKey>();
        int numKeys = Script.decodeFromOpN(this.chunks.get((int)(this.chunks.size() - 2)).opcode);
        for (int i2 = 0; i2 < numKeys; ++i2) {
            result.add(ECKey.fromPublicOnly(this.chunks.get((int)(1 + i2)).data));
        }
        return result;
    }

    private int findSigInRedeem(byte[] signatureBytes, Sha256Hash hash) throws SignatureDecodeException {
        Preconditions.checkArgument(this.chunks.get(0).isOpCode());
        int numKeys = Script.decodeFromOpN(this.chunks.get((int)(this.chunks.size() - 2)).opcode);
        TransactionSignature signature = TransactionSignature.decodeFromBitcoin(signatureBytes, true, false);
        for (int i2 = 0; i2 < numKeys; ++i2) {
            if (!ECKey.fromPublicOnly(this.chunks.get((int)(i2 + 1)).data).verify(hash, signature)) continue;
            return i2;
        }
        throw new IllegalStateException("Could not find matching key for signature on " + hash.toString() + " sig " + ByteUtils.formatHex(signatureBytes));
    }

    private static int getSigOpCount(List<ScriptChunk> chunks, boolean accurate) throws ScriptException {
        int sigOps = 0;
        int lastOpCode = 255;
        for (ScriptChunk chunk : chunks) {
            if (!chunk.isOpCode()) continue;
            switch (chunk.opcode) {
                case 172: 
                case 173: {
                    ++sigOps;
                    break;
                }
                case 174: 
                case 175: {
                    if (accurate && lastOpCode >= 81 && lastOpCode <= 96) {
                        sigOps += Script.decodeFromOpN(lastOpCode);
                        break;
                    }
                    sigOps += 20;
                    break;
                }
            }
            lastOpCode = chunk.opcode;
        }
        return sigOps;
    }

    public static int decodeFromOpN(int opcode) {
        Preconditions.checkArgument(opcode == 0 || opcode == 79 || opcode >= 81 && opcode <= 96, () -> "decodeFromOpN called on non OP_N opcode: " + ScriptOpCodes.getOpCodeName(opcode));
        if (opcode == 0) {
            return 0;
        }
        if (opcode == 79) {
            return -1;
        }
        return opcode + 1 - 81;
    }

    public static int encodeToOpN(int value) {
        Preconditions.checkArgument(value >= -1 && value <= 16, () -> "encodeToOpN called for " + value + " which we cannot encode in an opcode");
        if (value == 0) {
            return 0;
        }
        if (value == -1) {
            return 79;
        }
        return value - 1 + 81;
    }

    public static int getSigOpCount(byte[] program) throws ScriptException {
        ArrayList<ScriptChunk> chunks = new ArrayList<ScriptChunk>(5);
        try {
            Script.parseIntoChunksPartial(program, chunks);
        }
        catch (ScriptException scriptException) {
            // empty catch block
        }
        return Script.getSigOpCount(chunks, false);
    }

    public static long getP2SHSigOpCount(byte[] scriptSig) throws ScriptException {
        ArrayList<ScriptChunk> chunks = new ArrayList<ScriptChunk>(5);
        try {
            Script.parseIntoChunksPartial(scriptSig, chunks);
        }
        catch (ScriptException scriptException) {
            // empty catch block
        }
        Collections.reverse(chunks);
        for (ScriptChunk chunk : chunks) {
            if (chunk.isOpCode()) continue;
            Script subScript = Script.parse(chunk.data);
            return Script.getSigOpCount(subScript.chunks, true);
        }
        return 0L;
    }

    public int getNumberOfSignaturesRequiredToSpend() {
        if (ScriptPattern.isSentToMultisig(this)) {
            ScriptChunk nChunk = this.chunks.get(0);
            return Script.decodeFromOpN(nChunk.opcode);
        }
        if (ScriptPattern.isP2PKH(this) || ScriptPattern.isP2PK(this)) {
            return 1;
        }
        if (ScriptPattern.isP2SH(this)) {
            throw new IllegalStateException("For P2SH number of signatures depends on redeem script");
        }
        throw new IllegalStateException("Unsupported script type");
    }

    public int getNumberOfBytesRequiredToSpend(@Nullable ECKey pubKey, @Nullable Script redeemScript) {
        if (ScriptPattern.isP2SH(this)) {
            Preconditions.checkArgument(redeemScript != null, () -> "P2SH script requires redeemScript to be spent");
            return redeemScript.getNumberOfSignaturesRequiredToSpend() * 75 + redeemScript.program().length;
        }
        if (ScriptPattern.isSentToMultisig(this)) {
            return this.getNumberOfSignaturesRequiredToSpend() * 75 + 1;
        }
        if (ScriptPattern.isP2PK(this)) {
            return 75;
        }
        if (ScriptPattern.isP2PKH(this)) {
            int uncompressedPubKeySize = 65;
            return 75 + (pubKey != null ? pubKey.getPubKey().length : uncompressedPubKeySize);
        }
        if (ScriptPattern.isP2WPKH(this)) {
            int compressedPubKeySize = 33;
            int publicKeyLength = pubKey != null ? pubKey.getPubKey().length : compressedPubKeySize;
            return VarInt.sizeOf(2L) + VarInt.sizeOf(75L) + 75 + VarInt.sizeOf(publicKeyLength) + publicKeyLength;
        }
        throw new IllegalStateException("Unsupported script type");
    }

    private static boolean equalsRange(byte[] a, int start, byte[] b) {
        if (start + b.length > a.length) {
            return false;
        }
        for (int i2 = 0; i2 < b.length; ++i2) {
            if (a[i2 + start] == b[i2]) continue;
            return false;
        }
        return true;
    }

    public static byte[] removeAllInstancesOf(byte[] inputScript, byte[] chunkToRemove) {
        int additionalBytes;
        ByteArrayOutputStream bos = new ByteArrayOutputStream(inputScript.length);
        for (int cursor = 0; cursor < inputScript.length; cursor += additionalBytes) {
            boolean skip = Script.equalsRange(inputScript, cursor, chunkToRemove);
            int opcode = inputScript[cursor++] & 0xFF;
            additionalBytes = 0;
            if (opcode >= 0 && opcode < 76) {
                additionalBytes = opcode;
            } else if (opcode == 76) {
                additionalBytes = (0xFF & inputScript[cursor]) + 1;
            } else if (opcode == 77) {
                additionalBytes = ByteUtils.readUint16(inputScript, cursor) + 2;
            } else if (opcode == 78) {
                additionalBytes = (int)ByteUtils.readUint32(inputScript, cursor) + 4;
            }
            if (skip) continue;
            try {
                bos.write(opcode);
                bos.write(Arrays.copyOfRange(inputScript, cursor, cursor + additionalBytes));
                continue;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return bos.toByteArray();
    }

    public static byte[] removeAllInstancesOfOp(byte[] inputScript, int opCode) {
        return Script.removeAllInstancesOf(inputScript, new byte[]{(byte)opCode});
    }

    private static boolean castToBool(byte[] data) {
        for (int i2 = 0; i2 < data.length; ++i2) {
            if (data[i2] == 0) continue;
            return i2 != data.length - 1 || (data[i2] & 0xFF) != 128;
        }
        return false;
    }

    private static BigInteger castToBigInteger(byte[] chunk, boolean requireMinimal) throws ScriptException {
        return Script.castToBigInteger(chunk, 4, requireMinimal);
    }

    static BigInteger castToBigInteger(byte[] chunk, int maxLength, boolean requireMinimal) throws ScriptException {
        if (chunk.length > maxLength) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "Script attempted to use an integer larger than " + maxLength + " bytes");
        }
        if (requireMinimal && chunk.length > 0 && (chunk[chunk.length - 1] & 0x7F) == 0 && (chunk.length <= 1 || (chunk[chunk.length - 2] & 0x80) == 0)) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNKNOWN_ERROR, "non-minimally encoded script number");
        }
        return ByteUtils.decodeMPI(ByteUtils.reverseBytes(chunk), false);
    }

    public static void executeScript(@Nullable Transaction txContainingThis, long index, Script script, LinkedList<byte[]> stack, Set<VerifyFlag> verifyFlags) throws ScriptException {
        int opCount = 0;
        int lastCodeSepLocation = 0;
        LinkedList<byte[]> altstack = new LinkedList<byte[]>();
        LinkedList<Boolean> ifStack = new LinkedList<Boolean>();
        int nextLocationInScript = 0;
        block70: for (ScriptChunk chunk : script.chunks) {
            boolean shouldExecute = !ifStack.contains(false);
            int opcode = chunk.opcode;
            nextLocationInScript += chunk.size();
            if (chunk.data != null && chunk.data.length > 520) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_PUSH_SIZE, "Attempted to push a data string larger than 520 bytes");
            }
            if (opcode > 96 && ++opCount > 201) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_OP_COUNT, "More script operations than is allowed");
            }
            if (opcode == 126 || opcode == 127 || opcode == 128 || opcode == 129 || opcode == 131 || opcode == 132 || opcode == 133 || opcode == 134 || opcode == 141 || opcode == 142 || opcode == 149 || opcode == 150 || opcode == 151 || opcode == 152 || opcode == 153) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_DISABLED_OPCODE, "Script included disabled Script Op " + ScriptOpCodes.getOpCodeName(opcode));
            }
            if (shouldExecute && 0 <= opcode && opcode <= 78) {
                if (verifyFlags.contains((Object)VerifyFlag.MINIMALDATA) && !chunk.isShortestPossiblePushData()) {
                    throw new ScriptException(ScriptError.SCRIPT_ERR_MINIMALDATA, "Script included a not minimal push operation.");
                }
                if (opcode == 0) {
                    stack.add(new byte[0]);
                } else {
                    stack.add(chunk.data);
                }
            } else if (shouldExecute || 99 <= opcode && opcode <= 104) {
                switch (opcode) {
                    case 99: {
                        if (!shouldExecute) {
                            ifStack.add(false);
                            continue block70;
                        }
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_UNBALANCED_CONDITIONAL, "Attempted OP_IF on an empty stack");
                        }
                        ifStack.add(Script.castToBool(stack.pollLast()));
                        continue block70;
                    }
                    case 100: {
                        if (!shouldExecute) {
                            ifStack.add(false);
                            continue block70;
                        }
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_UNBALANCED_CONDITIONAL, "Attempted OP_NOTIF on an empty stack");
                        }
                        ifStack.add(!Script.castToBool(stack.pollLast()));
                        continue block70;
                    }
                    case 103: {
                        if (ifStack.isEmpty()) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_UNBALANCED_CONDITIONAL, "Attempted OP_ELSE without OP_IF/NOTIF");
                        }
                        ifStack.add((Boolean)ifStack.pollLast() == false);
                        continue block70;
                    }
                    case 104: {
                        if (ifStack.isEmpty()) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_UNBALANCED_CONDITIONAL, "Attempted OP_ENDIF without OP_IF/NOTIF");
                        }
                        ifStack.pollLast();
                        continue block70;
                    }
                    case 79: {
                        stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(BigInteger.ONE.negate(), false)));
                        break;
                    }
                    case 81: 
                    case 82: 
                    case 83: 
                    case 84: 
                    case 85: 
                    case 86: 
                    case 87: 
                    case 88: 
                    case 89: 
                    case 90: 
                    case 91: 
                    case 92: 
                    case 93: 
                    case 94: 
                    case 95: 
                    case 96: {
                        stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(BigInteger.valueOf(Script.decodeFromOpN(opcode)), false)));
                        break;
                    }
                    case 97: {
                        break;
                    }
                    case 105: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_VERIFY on an empty stack");
                        }
                        if (Script.castToBool(stack.pollLast())) break;
                        throw new ScriptException(ScriptError.SCRIPT_ERR_VERIFY, "OP_VERIFY failed");
                    }
                    case 106: {
                        throw new ScriptException(ScriptError.SCRIPT_ERR_OP_RETURN, "Script called OP_RETURN");
                    }
                    case 107: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_TOALTSTACK on an empty stack");
                        }
                        altstack.add(stack.pollLast());
                        break;
                    }
                    case 108: {
                        if (altstack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_ALTSTACK_OPERATION, "Attempted OP_FROMALTSTACK on an empty altstack");
                        }
                        stack.add((byte[])altstack.pollLast());
                        break;
                    }
                    case 109: {
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_2DROP on a stack with size < 2");
                        }
                        stack.pollLast();
                        stack.pollLast();
                        break;
                    }
                    case 110: {
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_2DUP on a stack with size < 2");
                        }
                        Iterator<byte[]> it2DUP = stack.descendingIterator();
                        byte[] OP2DUPtmpChunk2 = it2DUP.next();
                        stack.add(it2DUP.next());
                        stack.add(OP2DUPtmpChunk2);
                        break;
                    }
                    case 111: {
                        if (stack.size() < 3) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_3DUP on a stack with size < 3");
                        }
                        Iterator<byte[]> it3DUP = stack.descendingIterator();
                        byte[] OP3DUPtmpChunk3 = it3DUP.next();
                        byte[] OP3DUPtmpChunk2 = it3DUP.next();
                        stack.add(it3DUP.next());
                        stack.add(OP3DUPtmpChunk2);
                        stack.add(OP3DUPtmpChunk3);
                        break;
                    }
                    case 112: {
                        if (stack.size() < 4) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_2OVER on a stack with size < 4");
                        }
                        Iterator<byte[]> it2OVER = stack.descendingIterator();
                        it2OVER.next();
                        it2OVER.next();
                        byte[] OP2OVERtmpChunk2 = it2OVER.next();
                        stack.add(it2OVER.next());
                        stack.add(OP2OVERtmpChunk2);
                        break;
                    }
                    case 113: {
                        if (stack.size() < 6) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_2ROT on a stack with size < 6");
                        }
                        byte[] OP2ROTtmpChunk6 = stack.pollLast();
                        byte[] OP2ROTtmpChunk5 = stack.pollLast();
                        byte[] OP2ROTtmpChunk4 = stack.pollLast();
                        byte[] OP2ROTtmpChunk3 = stack.pollLast();
                        byte[] OP2ROTtmpChunk2 = stack.pollLast();
                        byte[] OP2ROTtmpChunk1 = stack.pollLast();
                        stack.add(OP2ROTtmpChunk3);
                        stack.add(OP2ROTtmpChunk4);
                        stack.add(OP2ROTtmpChunk5);
                        stack.add(OP2ROTtmpChunk6);
                        stack.add(OP2ROTtmpChunk1);
                        stack.add(OP2ROTtmpChunk2);
                        break;
                    }
                    case 114: {
                        if (stack.size() < 4) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_2SWAP on a stack with size < 4");
                        }
                        byte[] OP2SWAPtmpChunk4 = stack.pollLast();
                        byte[] OP2SWAPtmpChunk3 = stack.pollLast();
                        byte[] OP2SWAPtmpChunk2 = stack.pollLast();
                        byte[] OP2SWAPtmpChunk1 = stack.pollLast();
                        stack.add(OP2SWAPtmpChunk3);
                        stack.add(OP2SWAPtmpChunk4);
                        stack.add(OP2SWAPtmpChunk1);
                        stack.add(OP2SWAPtmpChunk2);
                        break;
                    }
                    case 115: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_IFDUP on an empty stack");
                        }
                        if (!Script.castToBool(stack.getLast())) break;
                        stack.add(stack.getLast());
                        break;
                    }
                    case 116: {
                        stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(BigInteger.valueOf(stack.size()), false)));
                        break;
                    }
                    case 117: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_DROP on an empty stack");
                        }
                        stack.pollLast();
                        break;
                    }
                    case 118: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_DUP on an empty stack");
                        }
                        stack.add(stack.getLast());
                        break;
                    }
                    case 119: {
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_NIP on a stack with size < 2");
                        }
                        byte[] OPNIPtmpChunk = stack.pollLast();
                        stack.pollLast();
                        stack.add(OPNIPtmpChunk);
                        break;
                    }
                    case 120: {
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_OVER on a stack with size < 2");
                        }
                        Iterator<byte[]> itOVER = stack.descendingIterator();
                        itOVER.next();
                        stack.add(itOVER.next());
                        break;
                    }
                    case 121: 
                    case 122: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_PICK/OP_ROLL on an empty stack");
                        }
                        long val = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA)).longValue();
                        if (val < 0L || val >= (long)stack.size()) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "OP_PICK/OP_ROLL attempted to get data deeper than stack size");
                        }
                        Iterator<byte[]> itPICK = stack.descendingIterator();
                        for (long i2 = 0L; i2 < val; ++i2) {
                            itPICK.next();
                        }
                        byte[] OPROLLtmpChunk = itPICK.next();
                        if (opcode == 122) {
                            itPICK.remove();
                        }
                        stack.add(OPROLLtmpChunk);
                        break;
                    }
                    case 123: {
                        if (stack.size() < 3) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_ROT on a stack with size < 3");
                        }
                        byte[] OPROTtmpChunk3 = stack.pollLast();
                        byte[] OPROTtmpChunk2 = stack.pollLast();
                        byte[] OPROTtmpChunk1 = stack.pollLast();
                        stack.add(OPROTtmpChunk2);
                        stack.add(OPROTtmpChunk3);
                        stack.add(OPROTtmpChunk1);
                        break;
                    }
                    case 124: 
                    case 125: {
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_SWAP on a stack with size < 2");
                        }
                        byte[] OPSWAPtmpChunk2 = stack.pollLast();
                        byte[] OPSWAPtmpChunk1 = stack.pollLast();
                        stack.add(OPSWAPtmpChunk2);
                        stack.add(OPSWAPtmpChunk1);
                        if (opcode != 125) break;
                        stack.add(OPSWAPtmpChunk2);
                        break;
                    }
                    case 130: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_SIZE on an empty stack");
                        }
                        stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(BigInteger.valueOf(stack.getLast().length), false)));
                        break;
                    }
                    case 135: {
                        byte[] byArray;
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_EQUAL on a stack with size < 2");
                        }
                        if (Arrays.equals(stack.pollLast(), stack.pollLast())) {
                            byte[] byArray2 = new byte[1];
                            byArray = byArray2;
                            byArray2[0] = 1;
                        } else {
                            byArray = new byte[]{};
                        }
                        stack.add(byArray);
                        break;
                    }
                    case 136: {
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_EQUALVERIFY on a stack with size < 2");
                        }
                        if (Arrays.equals(stack.pollLast(), stack.pollLast())) break;
                        throw new ScriptException(ScriptError.SCRIPT_ERR_EQUALVERIFY, "OP_EQUALVERIFY: non-equal data");
                    }
                    case 139: 
                    case 140: 
                    case 143: 
                    case 144: 
                    case 145: 
                    case 146: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted a numeric op on an empty stack");
                        }
                        BigInteger numericOPnum = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
                        switch (opcode) {
                            case 139: {
                                numericOPnum = numericOPnum.add(BigInteger.ONE);
                                break;
                            }
                            case 140: {
                                numericOPnum = numericOPnum.subtract(BigInteger.ONE);
                                break;
                            }
                            case 143: {
                                numericOPnum = numericOPnum.negate();
                                break;
                            }
                            case 144: {
                                if (numericOPnum.signum() >= 0) break;
                                numericOPnum = numericOPnum.negate();
                                break;
                            }
                            case 145: {
                                if (numericOPnum.equals(BigInteger.ZERO)) {
                                    numericOPnum = BigInteger.ONE;
                                    break;
                                }
                                numericOPnum = BigInteger.ZERO;
                                break;
                            }
                            case 146: {
                                if (numericOPnum.equals(BigInteger.ZERO)) {
                                    numericOPnum = BigInteger.ZERO;
                                    break;
                                }
                                numericOPnum = BigInteger.ONE;
                                break;
                            }
                            default: {
                                throw new AssertionError((Object)"Unreachable");
                            }
                        }
                        stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(numericOPnum, false)));
                        break;
                    }
                    case 147: 
                    case 148: 
                    case 154: 
                    case 155: 
                    case 156: 
                    case 158: 
                    case 159: 
                    case 160: 
                    case 161: 
                    case 162: 
                    case 163: 
                    case 164: {
                        BigInteger numericOPresult;
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted a numeric op on a stack with size < 2");
                        }
                        BigInteger numericOPnum2 = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
                        BigInteger numericOPnum1 = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
                        switch (opcode) {
                            case 147: {
                                numericOPresult = numericOPnum1.add(numericOPnum2);
                                break;
                            }
                            case 148: {
                                numericOPresult = numericOPnum1.subtract(numericOPnum2);
                                break;
                            }
                            case 154: {
                                if (!numericOPnum1.equals(BigInteger.ZERO) && !numericOPnum2.equals(BigInteger.ZERO)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 155: {
                                if (!numericOPnum1.equals(BigInteger.ZERO) || !numericOPnum2.equals(BigInteger.ZERO)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 156: {
                                if (numericOPnum1.equals(numericOPnum2)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 158: {
                                if (!numericOPnum1.equals(numericOPnum2)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 159: {
                                if (numericOPnum1.compareTo(numericOPnum2) < 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 160: {
                                if (numericOPnum1.compareTo(numericOPnum2) > 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 161: {
                                if (numericOPnum1.compareTo(numericOPnum2) <= 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 162: {
                                if (numericOPnum1.compareTo(numericOPnum2) >= 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 163: {
                                if (numericOPnum1.compareTo(numericOPnum2) < 0) {
                                    numericOPresult = numericOPnum1;
                                    break;
                                }
                                numericOPresult = numericOPnum2;
                                break;
                            }
                            case 164: {
                                if (numericOPnum1.compareTo(numericOPnum2) > 0) {
                                    numericOPresult = numericOPnum1;
                                    break;
                                }
                                numericOPresult = numericOPnum2;
                                break;
                            }
                            default: {
                                throw new RuntimeException("Opcode switched at runtime?");
                            }
                        }
                        stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(numericOPresult, false)));
                        break;
                    }
                    case 157: {
                        if (stack.size() < 2) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_NUMEQUALVERIFY on a stack with size < 2");
                        }
                        BigInteger OPNUMEQUALVERIFYnum2 = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
                        BigInteger OPNUMEQUALVERIFYnum1 = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
                        if (OPNUMEQUALVERIFYnum1.equals(OPNUMEQUALVERIFYnum2)) break;
                        throw new ScriptException(ScriptError.SCRIPT_ERR_NUMEQUALVERIFY, "OP_NUMEQUALVERIFY failed");
                    }
                    case 165: {
                        BigInteger OPWITHINnum1;
                        if (stack.size() < 3) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_WITHIN on a stack with size < 3");
                        }
                        BigInteger OPWITHINnum3 = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
                        BigInteger OPWITHINnum2 = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
                        if (OPWITHINnum2.compareTo(OPWITHINnum1 = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA))) <= 0 && OPWITHINnum1.compareTo(OPWITHINnum3) < 0) {
                            stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(BigInteger.ONE, false)));
                            break;
                        }
                        stack.add(ByteUtils.reverseBytes(ByteUtils.encodeMPI(BigInteger.ZERO, false)));
                        break;
                    }
                    case 166: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_RIPEMD160 on an empty stack");
                        }
                        byte[] dataToHash = stack.pollLast();
                        byte[] ripmeMdHash = CryptoUtils.digestRipeMd160(dataToHash);
                        stack.add(ripmeMdHash);
                        break;
                    }
                    case 167: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_SHA1 on an empty stack");
                        }
                        try {
                            stack.add(MessageDigest.getInstance("SHA-1").digest(stack.pollLast()));
                            break;
                        }
                        catch (NoSuchAlgorithmException e) {
                            throw new RuntimeException(e);
                        }
                    }
                    case 168: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_SHA256 on an empty stack");
                        }
                        stack.add(Sha256Hash.hash(stack.pollLast()));
                        break;
                    }
                    case 169: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_HASH160 on an empty stack");
                        }
                        stack.add(CryptoUtils.sha256hash160(stack.pollLast()));
                        break;
                    }
                    case 170: {
                        if (stack.size() < 1) {
                            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_SHA256 on an empty stack");
                        }
                        stack.add(Sha256Hash.hashTwice(stack.pollLast()));
                        break;
                    }
                    case 171: {
                        lastCodeSepLocation = nextLocationInScript;
                        break;
                    }
                    case 172: 
                    case 173: {
                        if (txContainingThis == null) {
                            throw new IllegalStateException("Script attempted signature check but no tx was provided");
                        }
                        Script.executeCheckSig(txContainingThis, (int)index, script, stack, lastCodeSepLocation, opcode, verifyFlags);
                        break;
                    }
                    case 174: 
                    case 175: {
                        if (txContainingThis == null) {
                            throw new IllegalStateException("Script attempted signature check but no tx was provided");
                        }
                        opCount = Script.executeMultiSig(txContainingThis, (int)index, script, stack, opCount, lastCodeSepLocation, opcode, verifyFlags);
                        break;
                    }
                    case 177: {
                        if (!verifyFlags.contains((Object)VerifyFlag.CHECKLOCKTIMEVERIFY)) {
                            if (!verifyFlags.contains((Object)VerifyFlag.DISCOURAGE_UPGRADABLE_NOPS)) break;
                            throw new ScriptException(ScriptError.SCRIPT_ERR_DISCOURAGE_UPGRADABLE_NOPS, "Script used a reserved opcode " + opcode);
                        }
                        Script.executeCheckLockTimeVerify(txContainingThis, (int)index, stack, verifyFlags);
                        break;
                    }
                    case 178: {
                        if (!verifyFlags.contains((Object)VerifyFlag.CHECKSEQUENCEVERIFY)) {
                            if (!verifyFlags.contains((Object)VerifyFlag.DISCOURAGE_UPGRADABLE_NOPS)) break;
                            throw new ScriptException(ScriptError.SCRIPT_ERR_DISCOURAGE_UPGRADABLE_NOPS, "Script used a reserved opcode " + opcode);
                        }
                        Script.executeCheckSequenceVerify(txContainingThis, (int)index, stack, verifyFlags);
                        break;
                    }
                    case 176: 
                    case 179: 
                    case 180: 
                    case 181: 
                    case 182: 
                    case 183: 
                    case 184: 
                    case 185: {
                        if (!verifyFlags.contains((Object)VerifyFlag.DISCOURAGE_UPGRADABLE_NOPS)) break;
                        throw new ScriptException(ScriptError.SCRIPT_ERR_DISCOURAGE_UPGRADABLE_NOPS, "Script used a reserved opcode " + opcode);
                    }
                    default: {
                        throw new ScriptException(ScriptError.SCRIPT_ERR_BAD_OPCODE, "Script used a reserved or disabled opcode: " + opcode);
                    }
                }
            }
            if (stack.size() + altstack.size() <= 1000 && stack.size() + altstack.size() >= 0) continue;
            throw new ScriptException(ScriptError.SCRIPT_ERR_STACK_SIZE, "Stack size exceeded range");
        }
        if (!ifStack.isEmpty()) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNBALANCED_CONDITIONAL, "OP_IF/OP_NOTIF without OP_ENDIF");
        }
    }

    private static void executeCheckLockTimeVerify(Transaction txContainingThis, int index, LinkedList<byte[]> stack, Set<VerifyFlag> verifyFlags) throws ScriptException {
        if (stack.size() < 1) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_CHECKLOCKTIMEVERIFY on a stack with size < 1");
        }
        BigInteger nLockTime = Script.castToBigInteger(stack.getLast(), 5, verifyFlags.contains((Object)VerifyFlag.MINIMALDATA));
        if (nLockTime.compareTo(BigInteger.ZERO) < 0) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_NEGATIVE_LOCKTIME, "Negative locktime");
        }
        LockTime txContainingThisLockTime = txContainingThis.lockTime();
        if (!(txContainingThisLockTime instanceof LockTime.HeightLock && nLockTime.compareTo(LOCKTIME_THRESHOLD_BIG) < 0 || txContainingThisLockTime instanceof LockTime.TimeLock && nLockTime.compareTo(LOCKTIME_THRESHOLD_BIG) >= 0)) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNSATISFIED_LOCKTIME, "Lock time requirement type mismatch");
        }
        if (nLockTime.compareTo(BigInteger.valueOf(txContainingThisLockTime.rawValue())) > 0) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNSATISFIED_LOCKTIME, "Lock time requirement not satisfied");
        }
        if (!txContainingThis.getInput(index).hasSequence()) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNSATISFIED_LOCKTIME, "Transaction contains a final transaction input for a CHECKLOCKTIMEVERIFY script.");
        }
    }

    private static void executeCheckSequenceVerify(Transaction txContainingThis, int index, LinkedList<byte[]> stack, Set<VerifyFlag> verifyFlags) throws ScriptException {
        if (stack.size() < 1) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_CHECKSEQUENCEVERIFY on a stack with size < 1");
        }
        long nSequence = Script.castToBigInteger(stack.getLast(), 5, verifyFlags.contains((Object)VerifyFlag.MINIMALDATA)).longValue();
        if (nSequence < 0L) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_NEGATIVE_LOCKTIME, "Negative sequence");
        }
        if ((nSequence & 0x80000000L) != 0L) {
            return;
        }
        Script.checkSequence(nSequence, txContainingThis, index);
    }

    private static void checkSequence(long nSequence, Transaction txContainingThis, int index) {
        long txToSequence = txContainingThis.getInput(index).getSequenceNumber();
        if (txContainingThis.getVersion() < 2L) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNSATISFIED_LOCKTIME, "Transaction version is < 2");
        }
        if ((txToSequence & 0x80000000L) != 0L) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNSATISFIED_LOCKTIME, "Sequence disable flag is set");
        }
        long nLockTimeMask = 0x40FFFFL;
        long txToSequenceMasked = txToSequence & nLockTimeMask;
        long nSequenceMasked = nSequence & nLockTimeMask;
        if (!(txToSequenceMasked < 0x400000L && nSequenceMasked < 0x400000L || txToSequenceMasked >= 0x400000L && nSequenceMasked >= 0x400000L)) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNSATISFIED_LOCKTIME, "Relative locktime requirement type mismatch");
        }
        if (nSequenceMasked > txToSequenceMasked) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_UNSATISFIED_LOCKTIME, "Relative locktime requirement not satisfied");
        }
    }

    private static void executeCheckSig(Transaction txContainingThis, int index, Script script, LinkedList<byte[]> stack, int lastCodeSepLocation, int opcode, Set<VerifyFlag> verifyFlags) throws ScriptException {
        boolean requireCanonical;
        boolean bl = requireCanonical = verifyFlags.contains((Object)VerifyFlag.STRICTENC) || verifyFlags.contains((Object)VerifyFlag.DERSIG) || verifyFlags.contains((Object)VerifyFlag.LOW_S);
        if (stack.size() < 2) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_CHECKSIG(VERIFY) on a stack with size < 2");
        }
        byte[] pubKey = stack.pollLast();
        byte[] sigBytes = stack.pollLast();
        byte[] prog = script.program();
        byte[] connectedScript = Arrays.copyOfRange(prog, lastCodeSepLocation, prog.length);
        ByteArrayOutputStream outStream = new ByteArrayOutputStream(sigBytes.length + 1);
        try {
            Script.writeBytes(outStream, sigBytes);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        connectedScript = Script.removeAllInstancesOf(connectedScript, outStream.toByteArray());
        boolean sigValid = false;
        try {
            TransactionSignature sig = TransactionSignature.decodeFromBitcoin(sigBytes, requireCanonical, verifyFlags.contains((Object)VerifyFlag.LOW_S));
            Sha256Hash hash = txContainingThis.hashForSignature(index, connectedScript, (byte)sig.sighashFlags);
            sigValid = ECKey.verify(hash.getBytes(), sig, pubKey);
        }
        catch (VerificationException.NoncanonicalSignature e) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_DER, "Script contains non-canonical signature");
        }
        catch (SignatureDecodeException e) {
            if (e.getMessage() != null && !e.getMessage().contains("Reached past end of ASN.1 stream")) {
                log.warn("Signature parsing failed!", e);
            }
        }
        catch (Exception e) {
            log.warn("Signature checking failed!", e);
        }
        if (opcode == 172) {
            byte[] byArray;
            if (sigValid) {
                byte[] byArray2 = new byte[1];
                byArray = byArray2;
                byArray2[0] = 1;
            } else {
                byArray = new byte[]{};
            }
            stack.add(byArray);
        } else if (opcode == 173 && !sigValid) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_CHECKSIGVERIFY, "Script failed OP_CHECKSIGVERIFY");
        }
    }

    private static int executeMultiSig(Transaction txContainingThis, int index, Script script, LinkedList<byte[]> stack, int opCount, int lastCodeSepLocation, int opcode, Set<VerifyFlag> verifyFlags) throws ScriptException {
        boolean requireCanonical;
        boolean bl = requireCanonical = verifyFlags.contains((Object)VerifyFlag.STRICTENC) || verifyFlags.contains((Object)VerifyFlag.DERSIG) || verifyFlags.contains((Object)VerifyFlag.LOW_S);
        if (stack.size() < 1) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_CHECKMULTISIG(VERIFY) on a stack with size < 2");
        }
        int pubKeyCount = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA)).intValue();
        if (pubKeyCount < 0 || pubKeyCount > 20) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_PUBKEY_COUNT, "OP_CHECKMULTISIG(VERIFY) with pubkey count out of range");
        }
        if ((opCount += pubKeyCount) > 201) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_OP_COUNT, "Total op count > 201 during OP_CHECKMULTISIG(VERIFY)");
        }
        if (stack.size() < pubKeyCount + 1) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_CHECKMULTISIG(VERIFY) on a stack with size < num_of_pubkeys + 2");
        }
        LinkedList<byte[]> pubkeys = new LinkedList<byte[]>();
        for (int i2 = 0; i2 < pubKeyCount; ++i2) {
            byte[] pubKey = stack.pollLast();
            pubkeys.add(pubKey);
        }
        int sigCount = Script.castToBigInteger(stack.pollLast(), verifyFlags.contains((Object)VerifyFlag.MINIMALDATA)).intValue();
        if (sigCount < 0 || sigCount > pubKeyCount) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_COUNT, "OP_CHECKMULTISIG(VERIFY) with sig count out of range");
        }
        if (stack.size() < sigCount + 1) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_INVALID_STACK_OPERATION, "Attempted OP_CHECKMULTISIG(VERIFY) on a stack with size < num_of_pubkeys + num_of_signatures + 3");
        }
        LinkedList<byte[]> sigs = new LinkedList<byte[]>();
        for (int i3 = 0; i3 < sigCount; ++i3) {
            byte[] sig = stack.pollLast();
            sigs.add(sig);
        }
        byte[] prog = script.program();
        byte[] connectedScript = Arrays.copyOfRange(prog, lastCodeSepLocation, prog.length);
        for (byte[] sig : sigs) {
            ByteArrayOutputStream outStream = new ByteArrayOutputStream(sig.length + 1);
            try {
                Script.writeBytes(outStream, sig);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            connectedScript = Script.removeAllInstancesOf(connectedScript, outStream.toByteArray());
        }
        boolean valid = true;
        while (sigs.size() > 0) {
            byte[] pubKey = (byte[])pubkeys.pollFirst();
            try {
                TransactionSignature sig = TransactionSignature.decodeFromBitcoin((byte[])sigs.getFirst(), requireCanonical, false);
                Sha256Hash hash = txContainingThis.hashForSignature(index, connectedScript, (byte)sig.sighashFlags);
                if (ECKey.verify(hash.getBytes(), sig, pubKey)) {
                    sigs.pollFirst();
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            if (sigs.size() <= pubkeys.size()) continue;
            valid = false;
            break;
        }
        byte[] nullDummy = stack.pollLast();
        if (verifyFlags.contains((Object)VerifyFlag.NULLDUMMY) && nullDummy.length > 0) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_NULLFAIL, "OP_CHECKMULTISIG(VERIFY) with non-null nulldummy: " + Arrays.toString(nullDummy));
        }
        if (opcode == 174) {
            byte[] byArray;
            if (valid) {
                byte[] byArray2 = new byte[1];
                byArray = byArray2;
                byArray2[0] = 1;
            } else {
                byArray = new byte[]{};
            }
            stack.add(byArray);
        } else if (opcode == 175 && !valid) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_NULLFAIL, "Script failed OP_CHECKMULTISIGVERIFY");
        }
        return opCount;
    }

    public void correctlySpends(Transaction txContainingThis, int scriptSigIndex, @Nullable TransactionWitness witness, @Nullable Coin value, Script scriptPubKey, Set<VerifyFlag> verifyFlags) throws ScriptException {
        if (ScriptPattern.isP2WPKH(scriptPubKey)) {
            TransactionSignature signature;
            if (witness.getPushCount() < 2) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_WITNESS_PROGRAM_WITNESS_EMPTY, witness.toString());
            }
            try {
                signature = TransactionSignature.decodeFromBitcoin(witness.getPush(0), true, true);
            }
            catch (SignatureDecodeException x) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_DER, "Cannot decode", x);
            }
            ECKey pubkey = ECKey.fromPublicOnly(witness.getPush(1));
            Script scriptCode = ScriptBuilder.createP2PKHOutputScript(pubkey);
            Sha256Hash sigHash = txContainingThis.hashForWitnessSignature(scriptSigIndex, scriptCode, value, signature.sigHashMode(), false);
            boolean validSig = pubkey.verify(sigHash, signature);
            if (!validSig) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_CHECKSIGVERIFY, "Invalid signature");
            }
        } else if (ScriptPattern.isP2PKH(scriptPubKey)) {
            TransactionSignature signature;
            if (this.chunks.size() != 2) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_SCRIPT_SIZE, "Invalid size: " + this.chunks.size());
            }
            try {
                signature = TransactionSignature.decodeFromBitcoin(this.chunks.get((int)0).data, true, true);
            }
            catch (SignatureDecodeException x) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_DER, "Cannot decode", x);
            }
            ECKey pubkey = ECKey.fromPublicOnly(this.chunks.get((int)1).data);
            Sha256Hash sigHash = txContainingThis.hashForSignature(scriptSigIndex, scriptPubKey, signature.sigHashMode(), false);
            boolean validSig = pubkey.verify(sigHash, signature);
            if (!validSig) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_CHECKSIGVERIFY, "Invalid signature");
            }
        } else if (ScriptPattern.isP2PK(scriptPubKey)) {
            TransactionSignature signature;
            if (this.chunks.size() != 1) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_SCRIPT_SIZE, "Invalid size: " + this.chunks.size());
            }
            try {
                signature = TransactionSignature.decodeFromBitcoin(this.chunks.get((int)0).data, false, false);
            }
            catch (SignatureDecodeException x) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_DER, "Cannot decode", x);
            }
            ECKey pubkey = ECKey.fromPublicOnly(ScriptPattern.extractKeyFromP2PK(scriptPubKey));
            Sha256Hash sigHash = txContainingThis.hashForSignature(scriptSigIndex, scriptPubKey, signature.sigHashMode(), false);
            boolean validSig = pubkey.verify(sigHash, signature);
            if (!validSig) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_CHECKSIGVERIFY, "Invalid signature");
            }
        } else {
            this.correctlySpends(txContainingThis, scriptSigIndex, scriptPubKey, verifyFlags);
        }
    }

    @Deprecated
    public void correctlySpends(Transaction txContainingThis, long scriptSigIndex, Script scriptPubKey, Set<VerifyFlag> verifyFlags) throws ScriptException {
        try {
            txContainingThis = Transaction.read(ByteBuffer.wrap(txContainingThis.serialize()));
        }
        catch (ProtocolException e) {
            throw new RuntimeException(e);
        }
        if (this.program().length > 10000 || scriptPubKey.program().length > 10000) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_SCRIPT_SIZE, "Script larger than 10,000 bytes");
        }
        LinkedList<byte[]> stack = new LinkedList<byte[]>();
        LinkedList<byte[]> p2shStack = null;
        Script.executeScript(txContainingThis, scriptSigIndex, this, stack, verifyFlags);
        if (verifyFlags.contains((Object)VerifyFlag.P2SH)) {
            p2shStack = new LinkedList<byte[]>(stack);
        }
        Script.executeScript(txContainingThis, scriptSigIndex, scriptPubKey, stack, verifyFlags);
        if (stack.size() == 0) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_EVAL_FALSE, "Stack empty at end of script execution.");
        }
        LinkedList<byte[]> stackCopy = new LinkedList<byte[]>(stack);
        if (!Script.castToBool(stack.pollLast())) {
            throw new ScriptException(ScriptError.SCRIPT_ERR_EVAL_FALSE, "Script resulted in a non-true stack: " + Utils.toString(stackCopy));
        }
        if (verifyFlags.contains((Object)VerifyFlag.P2SH) && ScriptPattern.isP2SH(scriptPubKey)) {
            for (ScriptChunk chunk : this.chunks) {
                if (chunk.isPushData()) continue;
                throw new ScriptException(ScriptError.SCRIPT_ERR_SIG_PUSHONLY, "Attempted to spend a P2SH scriptPubKey with a script that contained the script op " + chunk);
            }
            byte[] scriptPubKeyBytes = p2shStack.pollLast();
            Script scriptPubKeyP2SH = Script.parse(scriptPubKeyBytes);
            Script.executeScript(txContainingThis, scriptSigIndex, scriptPubKeyP2SH, p2shStack, verifyFlags);
            if (p2shStack.size() == 0) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_EVAL_FALSE, "P2SH stack empty at end of script execution.");
            }
            LinkedList<byte[]> p2shStackCopy = new LinkedList<byte[]>(p2shStack);
            if (!Script.castToBool(p2shStack.pollLast())) {
                throw new ScriptException(ScriptError.SCRIPT_ERR_EVAL_FALSE, "P2SH script execution resulted in a non-true stack: " + Utils.toString(p2shStackCopy));
            }
        }
    }

    private byte[] getQuickProgram() {
        if (this.program != null) {
            return this.program;
        }
        return this.program();
    }

    @Nullable
    public ScriptType getScriptType() {
        if (ScriptPattern.isP2PKH(this)) {
            return ScriptType.P2PKH;
        }
        if (ScriptPattern.isP2PK(this)) {
            return ScriptType.P2PK;
        }
        if (ScriptPattern.isP2SH(this)) {
            return ScriptType.P2SH;
        }
        if (ScriptPattern.isP2WPKH(this)) {
            return ScriptType.P2WPKH;
        }
        if (ScriptPattern.isP2WSH(this)) {
            return ScriptType.P2WSH;
        }
        if (ScriptPattern.isP2TR(this)) {
            return ScriptType.P2TR;
        }
        return null;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        return Arrays.equals(this.getQuickProgram(), ((Script)o).getQuickProgram());
    }

    public int hashCode() {
        return Arrays.hashCode(this.getQuickProgram());
    }

    public static enum VerifyFlag {
        P2SH,
        STRICTENC,
        DERSIG,
        LOW_S,
        NULLDUMMY,
        SIGPUSHONLY,
        MINIMALDATA,
        DISCOURAGE_UPGRADABLE_NOPS,
        CLEANSTACK,
        CHECKLOCKTIMEVERIFY,
        CHECKSEQUENCEVERIFY;

    }
}

