/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.script;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Stack;
import javax.annotation.Nullable;
import org.bitcoinj.base.Address;
import org.bitcoinj.base.LegacyAddress;
import org.bitcoinj.base.ScriptType;
import org.bitcoinj.base.SegwitAddress;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.crypto.TransactionSignature;
import org.bitcoinj.crypto.internal.CryptoUtils;
import org.bitcoinj.script.Script;
import org.bitcoinj.script.ScriptChunk;

public class ScriptBuilder {
    private final List<ScriptChunk> chunks;
    @Nullable
    private Instant creationTime = null;

    public ScriptBuilder() {
        this.chunks = new LinkedList<ScriptChunk>();
    }

    public ScriptBuilder(Script template) {
        this.chunks = new ArrayList<ScriptChunk>(template.chunks());
    }

    public ScriptBuilder creationTime(Instant creationTime) {
        this.creationTime = Objects.requireNonNull(creationTime);
        return this;
    }

    public ScriptBuilder addChunk(ScriptChunk chunk) {
        return this.addChunk(this.chunks.size(), chunk);
    }

    public ScriptBuilder addChunk(int index, ScriptChunk chunk) {
        this.chunks.add(index, chunk);
        return this;
    }

    public ScriptBuilder op(int opcode) {
        return this.op(this.chunks.size(), opcode);
    }

    public ScriptBuilder op(int index, int opcode) {
        Preconditions.checkArgument(opcode > 78);
        return this.addChunk(index, new ScriptChunk(opcode, null));
    }

    public ScriptBuilder data(byte[] data) {
        return this.data(this.chunks.size(), data);
    }

    public ScriptBuilder data(int index, byte[] data) {
        int opcode;
        byte[] copy = Arrays.copyOf(data, data.length);
        if (data.length == 0) {
            opcode = 0;
        } else if (data.length == 1) {
            byte b = data[0];
            opcode = b >= 1 && b <= 16 ? Script.encodeToOpN(b) : 1;
        } else if (data.length < 76) {
            opcode = data.length;
        } else if (data.length < 256) {
            opcode = 76;
        } else if (data.length < 65536) {
            opcode = 77;
        } else {
            throw new RuntimeException("Unimplemented");
        }
        return this.addChunk(index, new ScriptChunk(opcode, copy));
    }

    public ScriptBuilder number(long num) {
        return this.number(this.chunks.size(), num);
    }

    public ScriptBuilder number(int index, long num) {
        if (num == -1L) {
            return this.op(index, 79);
        }
        if (num >= 0L && num <= 16L) {
            return this.smallNum(index, (int)num);
        }
        return this.bigNum(index, num);
    }

    public ScriptBuilder smallNum(int num) {
        return this.smallNum(this.chunks.size(), num);
    }

    protected ScriptBuilder bigNum(long num) {
        return this.bigNum(this.chunks.size(), num);
    }

    public ScriptBuilder smallNum(int index, int num) {
        Preconditions.checkArgument(num >= 0, () -> "cannot encode negative numbers with smallNum");
        Preconditions.checkArgument(num <= 16, () -> "cannot encode numbers larger than 16 with smallNum");
        return this.addChunk(index, new ScriptChunk(Script.encodeToOpN(num), null));
    }

    protected ScriptBuilder bigNum(int index, long num) {
        byte[] data;
        if (num == 0L) {
            data = new byte[]{};
        } else {
            Stack<Byte> result = new Stack<Byte>();
            boolean neg = num < 0L;
            for (long absvalue = Math.abs(num); absvalue != 0L; absvalue >>= 8) {
                result.push((byte)(absvalue & 0xFFL));
            }
            if (((Byte)result.peek() & 0x80) != 0) {
                result.push((byte)(neg ? 128 : 0));
            } else if (neg) {
                result.push((byte)((Byte)result.pop() | 0x80));
            }
            data = new byte[result.size()];
            for (int byteIdx = 0; byteIdx < data.length; ++byteIdx) {
                data[byteIdx] = (Byte)result.get(byteIdx);
            }
        }
        return this.addChunk(index, new ScriptChunk(data.length, data));
    }

    public ScriptBuilder opTrue() {
        return this.number(1L);
    }

    public ScriptBuilder opTrue(int index) {
        return this.number(index, 1L);
    }

    public ScriptBuilder opFalse() {
        return this.number(0L);
    }

    public ScriptBuilder opFalse(int index) {
        return this.number(index, 0L);
    }

    public Script build() {
        if (this.creationTime != null) {
            return Script.of(this.chunks, this.creationTime);
        }
        return Script.of(this.chunks);
    }

    public static Script createEmpty() {
        return new ScriptBuilder().build();
    }

    public static Script createOutputScript(Address to, Instant creationTime) {
        return new ScriptBuilder().outputScript(to).creationTime(creationTime).build();
    }

    public static Script createOutputScript(Address to) {
        return new ScriptBuilder().outputScript(to).build();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private ScriptBuilder outputScript(Address to) {
        Preconditions.checkState(this.chunks.isEmpty());
        if (to instanceof LegacyAddress) {
            ScriptType scriptType = to.getOutputScriptType();
            if (scriptType == ScriptType.P2PKH) {
                this.p2pkhOutputScript(((LegacyAddress)to).getHash());
                return this;
            } else {
                if (scriptType != ScriptType.P2SH) throw new IllegalStateException("Cannot handle " + (Object)((Object)scriptType));
                this.p2shOutputScript(((LegacyAddress)to).getHash());
            }
            return this;
        } else {
            if (!(to instanceof SegwitAddress)) throw new IllegalStateException("Cannot handle " + to);
            this.p2whOutputScript((SegwitAddress)to);
        }
        return this;
    }

    private ScriptBuilder p2whOutputScript(SegwitAddress address) {
        Preconditions.checkState(this.chunks.isEmpty());
        return this.smallNum(address.getWitnessVersion()).data(address.getWitnessProgram());
    }

    public static Script createInputScript(@Nullable TransactionSignature signature, ECKey pubKey) {
        byte[] pubkeyBytes = pubKey.getPubKey();
        byte[] sigBytes = signature != null ? signature.encodeToBitcoin() : new byte[]{};
        return new ScriptBuilder().data(sigBytes).data(pubkeyBytes).build();
    }

    public static Script createInputScript(@Nullable TransactionSignature signature) {
        byte[] sigBytes = signature != null ? signature.encodeToBitcoin() : new byte[]{};
        return new ScriptBuilder().data(sigBytes).build();
    }

    public static Script createMultiSigOutputScript(int threshold, List<ECKey> pubkeys) {
        Preconditions.checkArgument(threshold > 0);
        Preconditions.checkArgument(threshold <= pubkeys.size());
        Preconditions.checkArgument(pubkeys.size() <= 16);
        ScriptBuilder builder = new ScriptBuilder();
        builder.smallNum(threshold);
        for (ECKey key : pubkeys) {
            builder.data(key.getPubKey());
        }
        builder.smallNum(pubkeys.size());
        builder.op(174);
        return builder.build();
    }

    public static Script createMultiSigInputScript(List<TransactionSignature> signatures) {
        ArrayList<byte[]> sigs = new ArrayList<byte[]>(signatures.size());
        for (TransactionSignature signature : signatures) {
            sigs.add(signature.encodeToBitcoin());
        }
        return ScriptBuilder.createMultiSigInputScriptBytes(sigs, null);
    }

    public static Script createMultiSigInputScript(TransactionSignature ... signatures) {
        return ScriptBuilder.createMultiSigInputScript(Arrays.asList(signatures));
    }

    public static Script createMultiSigInputScriptBytes(List<byte[]> signatures) {
        return ScriptBuilder.createMultiSigInputScriptBytes(signatures, null);
    }

    public static Script createP2SHMultiSigInputScript(@Nullable List<TransactionSignature> signatures, Script multisigProgram) {
        ArrayList<byte[]> sigs = new ArrayList<byte[]>();
        if (signatures == null) {
            int numSigs = multisigProgram.getNumberOfSignaturesRequiredToSpend();
            for (int i2 = 0; i2 < numSigs; ++i2) {
                sigs.add(new byte[0]);
            }
        } else {
            for (TransactionSignature signature : signatures) {
                sigs.add(signature.encodeToBitcoin());
            }
        }
        return ScriptBuilder.createMultiSigInputScriptBytes(sigs, multisigProgram.program());
    }

    public static Script createMultiSigInputScriptBytes(List<byte[]> signatures, @Nullable byte[] multisigProgramBytes) {
        Preconditions.checkArgument(signatures.size() <= 16);
        ScriptBuilder builder = new ScriptBuilder();
        builder.smallNum(0);
        for (byte[] signature : signatures) {
            builder.data(signature);
        }
        if (multisigProgramBytes != null) {
            builder.data(multisigProgramBytes);
        }
        return builder.build();
    }

    public static Script updateScriptWithSignature(Script scriptSig, byte[] signature, int targetIndex, int sigsPrefixCount, int sigsSuffixCount) {
        ScriptBuilder builder = new ScriptBuilder();
        List<ScriptChunk> inputChunks = scriptSig.chunks();
        int totalChunks = inputChunks.size();
        boolean hasMissingSigs = inputChunks.get(totalChunks - sigsSuffixCount - 1).equalsOpCode(0);
        Preconditions.checkArgument(hasMissingSigs, () -> "scriptSig is already filled with signatures");
        for (ScriptChunk chunk : inputChunks.subList(0, sigsPrefixCount)) {
            builder.addChunk(chunk);
        }
        int pos = 0;
        boolean inserted = false;
        for (ScriptChunk chunk : inputChunks.subList(sigsPrefixCount, totalChunks - sigsSuffixCount)) {
            if (pos == targetIndex) {
                inserted = true;
                builder.data(signature);
                ++pos;
            }
            if (chunk.equalsOpCode(0)) continue;
            builder.addChunk(chunk);
            ++pos;
        }
        while (pos < totalChunks - sigsPrefixCount - sigsSuffixCount) {
            if (pos == targetIndex) {
                inserted = true;
                builder.data(signature);
            } else {
                builder.addChunk(new ScriptChunk(0, null));
            }
            ++pos;
        }
        for (ScriptChunk chunk : inputChunks.subList(totalChunks - sigsSuffixCount, totalChunks)) {
            builder.addChunk(chunk);
        }
        Preconditions.checkState(inserted);
        return builder.build();
    }

    public static Script createP2PKOutputScript(byte[] pubKey) {
        return new ScriptBuilder().data(pubKey).op(172).build();
    }

    public static Script createP2PKOutputScript(ECKey pubKey) {
        return ScriptBuilder.createP2PKOutputScript(pubKey.getPubKey());
    }

    public static Script createP2PKHOutputScript(byte[] hash) {
        return new ScriptBuilder().p2pkhOutputScript(hash).build();
    }

    private ScriptBuilder p2pkhOutputScript(byte[] hash) {
        Preconditions.checkArgument(hash.length == 20);
        Preconditions.checkState(this.chunks.isEmpty());
        return this.op(118).op(169).data(hash).op(136).op(172);
    }

    public static Script createP2PKHOutputScript(ECKey key) {
        Preconditions.checkArgument(key.isCompressed());
        return ScriptBuilder.createP2PKHOutputScript(key.getPubKeyHash());
    }

    public static Script createP2WPKHOutputScript(byte[] hash) {
        Preconditions.checkArgument(hash.length == 20);
        return new ScriptBuilder().smallNum(0).data(hash).build();
    }

    public static Script createP2WPKHOutputScript(ECKey key) {
        Preconditions.checkArgument(key.isCompressed());
        return ScriptBuilder.createP2WPKHOutputScript(key.getPubKeyHash());
    }

    public static Script createP2SHOutputScript(byte[] hash) {
        return new ScriptBuilder().p2shOutputScript(hash).build();
    }

    private ScriptBuilder p2shOutputScript(byte[] hash) {
        Preconditions.checkArgument(hash.length == 20);
        Preconditions.checkState(this.chunks.isEmpty());
        return this.op(169).data(hash).op(135);
    }

    public static Script createP2SHOutputScript(Script redeemScript) {
        byte[] hash = CryptoUtils.sha256hash160(redeemScript.program());
        return ScriptBuilder.createP2SHOutputScript(hash);
    }

    public static Script createP2WSHOutputScript(byte[] hash) {
        Preconditions.checkArgument(hash.length == 32);
        return new ScriptBuilder().smallNum(0).data(hash).build();
    }

    public static Script createP2WSHOutputScript(Script redeemScript) {
        byte[] hash = Sha256Hash.hash(redeemScript.program());
        return ScriptBuilder.createP2WSHOutputScript(hash);
    }

    public static Script createP2SHOutputScript(int threshold, List<ECKey> pubkeys) {
        Script redeemScript = ScriptBuilder.createRedeemScript(threshold, pubkeys);
        return ScriptBuilder.createP2SHOutputScript(redeemScript);
    }

    public static Script createRedeemScript(int threshold, List<ECKey> pubkeys) {
        pubkeys = new ArrayList<ECKey>(pubkeys);
        Collections.sort(pubkeys, ECKey.PUBKEY_COMPARATOR);
        return ScriptBuilder.createMultiSigOutputScript(threshold, pubkeys);
    }

    public static Script createOpReturnScript(byte[] data) {
        Preconditions.checkArgument(data.length <= 80);
        return new ScriptBuilder().op(106).data(data).build();
    }
}

