/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.store;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.Buffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.bitcoinj.base.Sha256Hash;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.core.Block;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.StoredBlock;
import org.bitcoinj.store.BlockStore;
import org.bitcoinj.store.BlockStoreException;
import org.bitcoinj.store.ChainFileLockedException;
import org.bitcoinj.utils.Threading;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SPVBlockStore
implements BlockStore {
    private static final Logger log = LoggerFactory.getLogger(SPVBlockStore.class);
    protected final ReentrantLock lock = Threading.lock(SPVBlockStore.class);
    public static final int DEFAULT_CAPACITY = 10000;
    @Deprecated
    public static final String HEADER_MAGIC = "SPVB";
    static final byte[] HEADER_MAGIC_V1 = "SPVB".getBytes(StandardCharsets.US_ASCII);
    static final byte[] HEADER_MAGIC_V2 = "SPV2".getBytes(StandardCharsets.US_ASCII);
    protected volatile MappedByteBuffer buffer;
    protected final NetworkParameters params;
    protected LinkedHashMap<Sha256Hash, StoredBlock> blockCache = new LinkedHashMap<Sha256Hash, StoredBlock>(){

        @Override
        protected boolean removeEldestEntry(Map.Entry<Sha256Hash, StoredBlock> entry) {
            return this.size() > 2050;
        }
    };
    private static final Object NOT_FOUND_MARKER = new Object();
    protected LinkedHashMap<Sha256Hash, Object> notFoundCache = new LinkedHashMap<Sha256Hash, Object>(){

        @Override
        protected boolean removeEldestEntry(Map.Entry<Sha256Hash, Object> entry) {
            return this.size() > 100;
        }
    };
    protected FileLock fileLock = null;
    protected RandomAccessFile randomAccessFile = null;
    private final FileChannel channel;
    private int fileLength;
    protected StoredBlock lastChainHead = null;
    static final int RECORD_SIZE_V1 = 128;
    static final int RECORD_SIZE_V2 = 148;
    protected static final int FILE_PROLOGUE_BYTES = 1024;

    public SPVBlockStore(NetworkParameters params, File file) throws BlockStoreException {
        this(params, file, 10000, false);
    }

    public SPVBlockStore(NetworkParameters params, File file, int capacity, boolean grow) throws BlockStoreException {
        Objects.requireNonNull(file);
        this.params = Objects.requireNonNull(params);
        Preconditions.checkArgument(capacity > 0);
        try {
            long currentLength;
            boolean exists = file.exists();
            this.randomAccessFile = new RandomAccessFile(file, "rw");
            this.channel = this.randomAccessFile.getChannel();
            this.fileLock = this.channel.tryLock();
            if (this.fileLock == null) {
                throw new ChainFileLockedException("Store file is already locked by another process");
            }
            this.fileLength = SPVBlockStore.getFileSize(capacity);
            byte[] currentHeader = new byte[4];
            if (exists) {
                log.info("Using existing SPV block chain file: " + file);
                this.buffer = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.randomAccessFile.length());
                this.buffer.get(currentHeader);
            } else {
                log.info("Creating new SPV block chain file: " + file);
                this.randomAccessFile.setLength(this.fileLength);
                this.buffer = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.fileLength);
                this.initNewStore(params.getGenesisBlock());
            }
            if (Arrays.equals(HEADER_MAGIC_V1, currentHeader)) {
                log.info("Migrating SPV block chain file from V1 to V2 format: " + file);
                this.migrateV1toV2();
            }
            if (exists && (currentLength = this.randomAccessFile.length()) != (long)this.fileLength) {
                if ((currentLength - 1024L) % 148L != 0L) {
                    throw new BlockStoreException("File size on disk indicates this is not a V2 block store: " + currentLength);
                }
                if (!grow) {
                    throw new BlockStoreException("File size on disk does not match expected size: " + currentLength + " vs " + this.fileLength);
                }
                if ((long)this.fileLength < this.randomAccessFile.length()) {
                    throw new BlockStoreException("Shrinking is unsupported: " + currentLength + " vs " + this.fileLength);
                }
                this.randomAccessFile.setLength(this.fileLength);
                this.buffer.force();
                this.buffer = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.fileLength);
            }
            byte[] header = new byte[4];
            ((Buffer)this.buffer).rewind();
            this.buffer.get(currentHeader);
            if (!Arrays.equals(currentHeader, HEADER_MAGIC_V2)) {
                throw new BlockStoreException("Magic header V2 expected: " + new String(currentHeader, StandardCharsets.US_ASCII));
            }
        }
        catch (Exception e) {
            try {
                if (this.randomAccessFile != null) {
                    this.randomAccessFile.close();
                }
            }
            catch (IOException e2) {
                throw new BlockStoreException(e2);
            }
            throw new BlockStoreException(e);
        }
    }

    private void initNewStore(Block genesisBlock) throws Exception {
        ((Buffer)this.buffer).rewind();
        this.buffer.put(HEADER_MAGIC_V2);
        this.lock.lock();
        try {
            this.setRingCursor(1024);
        }
        finally {
            this.lock.unlock();
        }
        StoredBlock storedGenesis = new StoredBlock(genesisBlock.cloneAsHeader(), genesisBlock.getWork(), 0);
        this.put(storedGenesis);
        this.setChainHead(storedGenesis);
    }

    private void migrateV1toV2() throws BlockStoreException, IOException {
        long currentLength = this.randomAccessFile.length();
        long currentBlocksLength = currentLength - 1024L;
        if (currentBlocksLength % 128L != 0L) {
            throw new BlockStoreException("File size on disk indicates this is not a V1 block store: " + currentLength);
        }
        int currentCapacity = (int)(currentBlocksLength / 128L);
        this.randomAccessFile.setLength(this.fileLength);
        this.buffer.force();
        this.buffer = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.fileLength);
        ((Buffer)this.buffer).rewind();
        this.buffer.put(HEADER_MAGIC_V2);
        byte[] zeroPadding = new byte[20];
        for (int i2 = currentCapacity - 1; i2 >= 0; --i2) {
            byte[] record = new byte[128];
            ((Buffer)this.buffer).position(1024 + i2 * 128);
            this.buffer.get(record);
            ((Buffer)this.buffer).position(1024 + i2 * 148);
            this.buffer.put(record, 0, 32);
            this.buffer.put(zeroPadding);
            this.buffer.put(record, 32, 96);
        }
        int cursorRecord = (this.getRingCursor() - 1024) / 128;
        this.setRingCursor(1024 + cursorRecord * 148);
    }

    public static int getFileSize(int capacity) {
        return 148 * capacity + 1024;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void put(StoredBlock block) throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            int cursor = this.getRingCursor();
            if (cursor == this.fileLength) {
                cursor = 1024;
            }
            ((Buffer)buffer).position(cursor);
            Sha256Hash hash = block.getHeader().getHash();
            this.notFoundCache.remove(hash);
            buffer.put(hash.getBytes());
            block.serializeCompactV2(buffer);
            this.setRingCursor(buffer.position());
            this.blockCache.put(hash, block);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    @Nullable
    public StoredBlock get(Sha256Hash hash) throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            int cursor;
            StoredBlock cacheHit = this.blockCache.get(hash);
            if (cacheHit != null) {
                StoredBlock storedBlock = cacheHit;
                return storedBlock;
            }
            if (this.notFoundCache.get(hash) != null) {
                StoredBlock storedBlock = null;
                return storedBlock;
            }
            int startingPoint = cursor = this.getRingCursor();
            byte[] targetHashBytes = hash.getBytes();
            byte[] scratch = new byte[32];
            do {
                if ((cursor -= 148) < 1024) {
                    cursor = this.fileLength - 148;
                }
                ((Buffer)buffer).position(cursor);
                buffer.get(scratch);
                if (!Arrays.equals(scratch, targetHashBytes)) continue;
                StoredBlock storedBlock = StoredBlock.deserializeCompactV2(buffer);
                this.blockCache.put(hash, storedBlock);
                StoredBlock storedBlock2 = storedBlock;
                return storedBlock2;
            } while (cursor != startingPoint);
            this.notFoundCache.put(hash, NOT_FOUND_MARKER);
            StoredBlock storedBlock = null;
            return storedBlock;
        }
        catch (ProtocolException e) {
            throw new RuntimeException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public StoredBlock getChainHead() throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            if (this.lastChainHead == null) {
                byte[] headHash = new byte[32];
                ((Buffer)buffer).position(8);
                buffer.get(headHash);
                Sha256Hash hash = Sha256Hash.wrap(headHash);
                StoredBlock block = this.get(hash);
                if (block == null) {
                    throw new BlockStoreException("Corrupted block store: could not find chain head: " + hash);
                }
                this.lastChainHead = block;
            }
            StoredBlock storedBlock = this.lastChainHead;
            return storedBlock;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            this.lastChainHead = chainHead;
            byte[] headHash = chainHead.getHeader().getHash().getBytes();
            ((Buffer)buffer).position(8);
            buffer.put(headHash);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void close() throws BlockStoreException {
        try {
            this.buffer.force();
            this.buffer = null;
            this.fileLock.release();
            this.randomAccessFile.close();
            this.blockCache.clear();
        }
        catch (IOException e) {
            throw new BlockStoreException(e);
        }
    }

    int getRingCursor() {
        int c = this.buffer.getInt(4);
        Preconditions.checkState(c >= 1024, () -> "integer overflow");
        return c;
    }

    private void setRingCursor(int newCursor) {
        Preconditions.checkArgument(newCursor >= 0);
        this.buffer.putInt(4, newCursor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() throws Exception {
        this.lock.lock();
        try {
            this.blockCache.clear();
            this.notFoundCache.clear();
            ((Buffer)this.buffer).position(0);
            long fileLength = this.randomAccessFile.length();
            int i2 = 0;
            while ((long)i2 < fileLength) {
                this.buffer.put((byte)0);
                ++i2;
            }
            this.initNewStore(this.params.getGenesisBlock());
        }
        finally {
            this.lock.unlock();
        }
    }
}

