/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.wallet;

import com.google.protobuf.ByteString;
import com.google.protobuf.GeneratedMessageLite;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.bitcoinj.base.Network;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.core.BloomFilter;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.crypto.AesKey;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.crypto.EncryptableItem;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.KeyCrypterScrypt;
import org.bitcoinj.protobuf.wallet.Protos;
import org.bitcoinj.utils.ListenerRegistration;
import org.bitcoinj.utils.Threading;
import org.bitcoinj.wallet.EncryptableKeyChain;
import org.bitcoinj.wallet.KeyChain;
import org.bitcoinj.wallet.UnreadableWalletException;
import org.bitcoinj.wallet.listeners.KeyChainEventListener;

public class BasicKeyChain
implements EncryptableKeyChain {
    private final ReentrantLock lock = Threading.lock(BasicKeyChain.class);
    private final LinkedHashMap<ByteString, ECKey> hashToKeys;
    private final LinkedHashMap<ByteString, ECKey> pubkeyToKeys;
    @Nullable
    private final KeyCrypter keyCrypter;
    private boolean isWatching;
    private final CopyOnWriteArrayList<ListenerRegistration<KeyChainEventListener>> listeners;

    public BasicKeyChain() {
        this(null);
    }

    public BasicKeyChain(@Nullable KeyCrypter crypter) {
        this.keyCrypter = crypter;
        this.hashToKeys = new LinkedHashMap();
        this.pubkeyToKeys = new LinkedHashMap();
        this.listeners = new CopyOnWriteArrayList();
    }

    @Override
    @Nullable
    public KeyCrypter getKeyCrypter() {
        this.lock.lock();
        try {
            KeyCrypter keyCrypter = this.keyCrypter;
            return keyCrypter;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public ECKey getKey(@Nullable KeyChain.KeyPurpose ignored) {
        this.lock.lock();
        try {
            if (this.hashToKeys.isEmpty()) {
                Preconditions.checkState(this.keyCrypter == null);
                ECKey key = new ECKey();
                this.importKeyLocked(key);
                this.queueOnKeysAdded(Collections.singletonList(key));
            }
            ECKey eCKey = this.hashToKeys.values().iterator().next();
            return eCKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<ECKey> getKeys(@Nullable KeyChain.KeyPurpose purpose, int numberOfKeys) {
        Preconditions.checkArgument(numberOfKeys > 0);
        this.lock.lock();
        try {
            if (this.hashToKeys.size() < numberOfKeys) {
                Preconditions.checkState(this.keyCrypter == null);
                ArrayList<ECKey> keys2 = new ArrayList<ECKey>();
                for (int i2 = 0; i2 < numberOfKeys - this.hashToKeys.size(); ++i2) {
                    keys2.add(new ECKey());
                }
                List<ECKey> immutableKeys = Collections.unmodifiableList(keys2);
                this.importKeysLocked(immutableKeys);
                this.queueOnKeysAdded(immutableKeys);
            }
            ArrayList<ECKey> keysToReturn = new ArrayList<ECKey>();
            for (int count = 0; this.hashToKeys.values().iterator().hasNext() && numberOfKeys != count; ++count) {
                keysToReturn.add(this.hashToKeys.values().iterator().next());
            }
            ArrayList<ECKey> arrayList = keysToReturn;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    public List<ECKey> getKeys() {
        this.lock.lock();
        try {
            ArrayList<ECKey> arrayList = new ArrayList<ECKey>(this.hashToKeys.values());
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int importKeys(ECKey ... keys2) {
        return this.importKeys(Collections.unmodifiableList(Arrays.asList(keys2)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int importKeys(List<? extends ECKey> keys2) {
        this.lock.lock();
        try {
            for (ECKey eCKey : keys2) {
                this.checkKeyEncryptionStateMatches(eCKey);
            }
            ArrayList<ECKey> actuallyAdded = new ArrayList<ECKey>(keys2.size());
            for (ECKey eCKey : keys2) {
                if (this.hasKey(eCKey)) continue;
                actuallyAdded.add(eCKey);
                this.importKeyLocked(eCKey);
            }
            if (actuallyAdded.size() > 0) {
                this.queueOnKeysAdded(actuallyAdded);
            }
            int n = actuallyAdded.size();
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void checkKeyEncryptionStateMatches(ECKey key) {
        if (this.keyCrypter == null && key.isEncrypted()) {
            throw new KeyCrypterException("Key is encrypted but chain is not");
        }
        if (this.keyCrypter != null && !key.isEncrypted()) {
            throw new KeyCrypterException("Key is not encrypted but chain is");
        }
        if (this.keyCrypter != null && key.getKeyCrypter() != null && !key.getKeyCrypter().equals(this.keyCrypter)) {
            throw new KeyCrypterException("Key encrypted under different parameters to chain");
        }
    }

    private void importKeyLocked(ECKey key) {
        if (this.hashToKeys.isEmpty()) {
            this.isWatching = key.isWatching();
        } else {
            if (key.isWatching() && !this.isWatching) {
                throw new IllegalArgumentException("Key is watching but chain is not");
            }
            if (!key.isWatching() && this.isWatching) {
                throw new IllegalArgumentException("Key is not watching but chain is");
            }
        }
        ECKey previousKey = this.pubkeyToKeys.put(ByteString.copyFrom(key.getPubKey()), key);
        this.hashToKeys.put(ByteString.copyFrom(key.getPubKeyHash()), key);
        Preconditions.checkState(previousKey == null);
    }

    private void importKeysLocked(List<ECKey> keys2) {
        for (ECKey key : keys2) {
            this.importKeyLocked(key);
        }
    }

    public void importKey(ECKey key) {
        this.lock.lock();
        try {
            this.checkKeyEncryptionStateMatches(key);
            if (this.hasKey(key)) {
                return;
            }
            this.importKeyLocked(key);
            this.queueOnKeysAdded(Collections.singletonList(key));
        }
        finally {
            this.lock.unlock();
        }
    }

    public ECKey findKeyFromPubHash(byte[] pubKeyHash) {
        this.lock.lock();
        try {
            ECKey eCKey = this.hashToKeys.get(ByteString.copyFrom(pubKeyHash));
            return eCKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    public ECKey findKeyFromPubKey(byte[] pubKey) {
        this.lock.lock();
        try {
            ECKey eCKey = this.pubkeyToKeys.get(ByteString.copyFrom(pubKey));
            return eCKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean hasKey(ECKey key) {
        return this.findKeyFromPubKey(key.getPubKey()) != null;
    }

    @Override
    public int numKeys() {
        return this.pubkeyToKeys.size();
    }

    public State isWatching() {
        this.lock.lock();
        try {
            if (this.hashToKeys.isEmpty()) {
                State state = State.EMPTY;
                return state;
            }
            State state = this.isWatching ? State.WATCHING : State.REGULAR;
            return state;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removeKey(ECKey key) {
        this.lock.lock();
        try {
            boolean a = this.hashToKeys.remove(ByteString.copyFrom(key.getPubKeyHash())) != null;
            boolean b = this.pubkeyToKeys.remove(ByteString.copyFrom(key.getPubKey())) != null;
            Preconditions.checkState(a == b);
            boolean bl = a;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Instant earliestKeyCreationTime() {
        this.lock.lock();
        try {
            Instant instant = this.hashToKeys.values().stream().map(key -> key.getCreationTime().orElse(Instant.EPOCH)).min(Instant::compareTo).orElse(Instant.MAX);
            return instant;
        }
        finally {
            this.lock.unlock();
        }
    }

    public List<ListenerRegistration<KeyChainEventListener>> getListeners() {
        return new ArrayList<ListenerRegistration<KeyChainEventListener>>(this.listeners);
    }

    Map<ECKey, Protos.Key.Builder> serializeToEditableProtobufs() {
        return this.hashToKeys.values().stream().collect(Collectors.toMap(ecKey -> ecKey, ecKey -> BasicKeyChain.toProtoKeyBuilder(ecKey), (oldVal, newVal) -> newVal, LinkedHashMap::new));
    }

    private static Protos.Key.Builder toProtoKeyBuilder(ECKey ecKey) {
        Protos.Key.Builder protoKey = BasicKeyChain.serializeEncryptableItem(ecKey);
        protoKey.setPublicKey(ByteString.copyFrom(ecKey.getPubKey()));
        return protoKey;
    }

    @Override
    public List<Protos.Key> serializeToProtobuf() {
        return this.serializeToEditableProtobufs().values().stream().map(GeneratedMessageLite.Builder::build).collect(Collectors.toList());
    }

    static Protos.Key.Builder serializeEncryptableItem(EncryptableItem item) {
        Protos.Key.Builder proto = Protos.Key.newBuilder();
        item.getCreationTime().ifPresent(creationTime -> proto.setCreationTimestamp(creationTime.toEpochMilli()));
        if (item.isEncrypted() && item.getEncryptedData() != null) {
            EncryptedData data = item.getEncryptedData();
            proto.setEncryptedData(((Protos.EncryptedData.Builder)proto.getEncryptedData().toBuilder()).setEncryptedPrivateKey(ByteString.copyFrom(data.encryptedBytes)).setInitialisationVector(ByteString.copyFrom(data.initialisationVector)));
            Preconditions.checkState(item.getEncryptionType() == Protos.Wallet.EncryptionType.ENCRYPTED_SCRYPT_AES);
            proto.setType(Protos.Key.Type.ENCRYPTED);
        } else {
            byte[] secret = item.getSecretBytes();
            if (secret != null) {
                proto.setSecretBytes(ByteString.copyFrom(secret));
            }
            proto.setType(Protos.Key.Type.ORIGINAL);
        }
        return proto;
    }

    public static BasicKeyChain fromProtobufUnencrypted(List<Protos.Key> keys2) throws UnreadableWalletException {
        BasicKeyChain chain = new BasicKeyChain();
        chain.deserializeFromProtobuf(keys2);
        return chain;
    }

    public static BasicKeyChain fromProtobufEncrypted(List<Protos.Key> keys2, KeyCrypter crypter) throws UnreadableWalletException {
        BasicKeyChain chain = new BasicKeyChain(Objects.requireNonNull(crypter));
        chain.deserializeFromProtobuf(keys2);
        return chain;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deserializeFromProtobuf(List<Protos.Key> keys2) throws UnreadableWalletException {
        this.lock.lock();
        try {
            Preconditions.checkState(this.hashToKeys.isEmpty(), () -> "tried to deserialize into a non-empty chain");
            for (Protos.Key key : keys2) {
                ECKey ecKey;
                byte[] priv;
                if (key.getType() != Protos.Key.Type.ORIGINAL && key.getType() != Protos.Key.Type.ENCRYPTED) continue;
                boolean encrypted = key.getType() == Protos.Key.Type.ENCRYPTED;
                byte[] byArray = priv = key.hasSecretBytes() ? key.getSecretBytes().toByteArray() : null;
                if (!key.hasPublicKey()) {
                    throw new UnreadableWalletException("Public key missing");
                }
                byte[] pub = key.getPublicKey().toByteArray();
                if (encrypted) {
                    Preconditions.checkState(this.keyCrypter != null, () -> "this wallet is encrypted but encrypt() was not called prior to deserialization");
                    if (!key.hasEncryptedData()) {
                        throw new UnreadableWalletException("Encrypted private key data missing");
                    }
                    Protos.EncryptedData proto = key.getEncryptedData();
                    EncryptedData e = new EncryptedData(proto.getInitialisationVector().toByteArray(), proto.getEncryptedPrivateKey().toByteArray());
                    ecKey = ECKey.fromEncrypted(e, this.keyCrypter, pub);
                } else {
                    ecKey = priv != null ? ECKey.fromPrivateAndPrecalculatedPublic(priv, pub) : ECKey.fromPublicOnly(pub);
                }
                ecKey.setCreationTime(Instant.ofEpochMilli(key.getCreationTimestamp()));
                this.importKeyLocked(ecKey);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void addEventListener(KeyChainEventListener listener) {
        this.addEventListener(listener, Threading.USER_THREAD);
    }

    @Override
    public void addEventListener(KeyChainEventListener listener, Executor executor) {
        this.addEventListener(new ListenerRegistration<KeyChainEventListener>(listener, executor));
    }

    void addEventListener(ListenerRegistration<KeyChainEventListener> listener) {
        this.listeners.add(listener);
    }

    @Override
    public boolean removeEventListener(KeyChainEventListener listener) {
        return ListenerRegistration.removeFromList(listener, this.listeners);
    }

    private void queueOnKeysAdded(List<ECKey> keys2) {
        Preconditions.checkState(this.lock.isHeldByCurrentThread());
        for (ListenerRegistration<KeyChainEventListener> registration : this.listeners) {
            registration.executor.execute(() -> ((KeyChainEventListener)registration.listener).onKeysAdded(keys2));
        }
    }

    @Override
    public BasicKeyChain toEncrypted(CharSequence password) {
        Objects.requireNonNull(password);
        Preconditions.checkArgument(password.length() > 0);
        KeyCrypterScrypt scrypt = new KeyCrypterScrypt();
        AesKey derivedKey = scrypt.deriveKey(password);
        return this.toEncrypted(scrypt, derivedKey);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BasicKeyChain toEncrypted(KeyCrypter keyCrypter, AesKey aesKey) {
        this.lock.lock();
        try {
            Objects.requireNonNull(keyCrypter);
            Preconditions.checkState(this.keyCrypter == null, () -> "key chain is already encrypted");
            BasicKeyChain encrypted = new BasicKeyChain(keyCrypter);
            for (ECKey eCKey : this.hashToKeys.values()) {
                ECKey encryptedKey;
                if (!ECKey.encryptionIsReversible(eCKey, encryptedKey = eCKey.encrypt(keyCrypter, aesKey), keyCrypter, aesKey)) {
                    throw new KeyCrypterException("The key " + eCKey.toString() + " cannot be successfully decrypted after encryption so aborting wallet encryption.");
                }
                encrypted.importKeyLocked(encryptedKey);
            }
            for (ListenerRegistration listenerRegistration : this.listeners) {
                encrypted.addEventListener(listenerRegistration);
            }
            BasicKeyChain basicKeyChain = encrypted;
            return basicKeyChain;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public BasicKeyChain toDecrypted(CharSequence password) {
        Objects.requireNonNull(this.keyCrypter, "Wallet is already decrypted");
        AesKey aesKey = this.keyCrypter.deriveKey(password);
        return this.toDecrypted(aesKey);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BasicKeyChain toDecrypted(AesKey aesKey) {
        this.lock.lock();
        try {
            Preconditions.checkState(this.keyCrypter != null, () -> "wallet is already decrypted");
            if (this.numKeys() > 0 && !this.checkAESKey(aesKey)) {
                throw new KeyCrypterException("Password/key was incorrect.");
            }
            BasicKeyChain decrypted = new BasicKeyChain();
            for (ECKey eCKey : this.hashToKeys.values()) {
                decrypted.importKeyLocked(eCKey.decrypt(aesKey));
            }
            for (ListenerRegistration listenerRegistration : this.listeners) {
                decrypted.addEventListener(listenerRegistration);
            }
            BasicKeyChain basicKeyChain = decrypted;
            return basicKeyChain;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean checkPassword(CharSequence password) {
        Objects.requireNonNull(password);
        Preconditions.checkState(this.keyCrypter != null, () -> "key chain not encrypted");
        return this.checkAESKey(this.keyCrypter.deriveKey(password));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean checkAESKey(AesKey aesKey) {
        this.lock.lock();
        try {
            if (this.hashToKeys.isEmpty()) {
                boolean bl = false;
                return bl;
            }
            Preconditions.checkState(this.keyCrypter != null, () -> "key chain is not encrypted");
            ECKey first = null;
            for (ECKey key : this.hashToKeys.values()) {
                if (!key.isEncrypted()) continue;
                first = key;
                break;
            }
            Preconditions.checkState(first != null, () -> "no encrypted keys in the wallet");
            try {
                ECKey rebornKey = first.decrypt(aesKey);
                boolean bl = Arrays.equals(first.getPubKey(), rebornKey.getPubKey());
                return bl;
            }
            catch (KeyCrypterException e) {
                boolean bl = false;
                this.lock.unlock();
                return bl;
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BloomFilter getFilter(int size, double falsePositiveRate, int tweak) {
        this.lock.lock();
        try {
            BloomFilter filter2 = new BloomFilter(size, falsePositiveRate, tweak);
            for (ECKey key : this.hashToKeys.values()) {
                filter2.insert(key);
            }
            BloomFilter bloomFilter = filter2;
            return bloomFilter;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public int numBloomFilterEntries() {
        return this.numKeys() * 2;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<ECKey> findOldestKeyAfter(Instant time) {
        this.lock.lock();
        try {
            ECKey oldest = null;
            for (ECKey key : this.hashToKeys.values()) {
                Instant keyTime = key.getCreationTime().orElse(Instant.EPOCH);
                if (!keyTime.isAfter(time) || oldest != null && !oldest.getCreationTime().orElse(Instant.EPOCH).isAfter(keyTime)) continue;
                oldest = key;
            }
            Optional<Object> optional = Optional.ofNullable(oldest);
            return optional;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Nullable
    @Deprecated
    public ECKey findOldestKeyAfter(long timeSecs) {
        return this.findOldestKeyAfter(Instant.ofEpochSecond(timeSecs)).orElse(null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<ECKey> findKeysBefore(Instant time) {
        this.lock.lock();
        try {
            LinkedList<ECKey> results = new LinkedList<ECKey>();
            for (ECKey key : this.hashToKeys.values()) {
                Instant keyTime = key.getCreationTime().orElse(Instant.EPOCH);
                if (!keyTime.isBefore(time)) continue;
                results.add(key);
            }
            LinkedList<ECKey> linkedList = results;
            return linkedList;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Deprecated
    public List<ECKey> findKeysBefore(long timeSecs) {
        return this.findKeysBefore(Instant.ofEpochSecond(timeSecs));
    }

    public String toString(boolean includePrivateKeys, @Nullable AesKey aesKey, Network network) {
        StringBuilder builder = new StringBuilder();
        List<ECKey> keys2 = this.getKeys();
        Collections.sort(keys2, ECKey.AGE_COMPARATOR);
        for (ECKey key : keys2) {
            key.formatKeyWithAddress(includePrivateKeys, aesKey, builder, network, null, "imported");
        }
        return builder.toString();
    }

    @Deprecated
    public String toString(boolean includePrivateKeys, @Nullable AesKey aesKey, NetworkParameters params) {
        return this.toString(includePrivateKeys, aesKey, params.network());
    }

    public static enum State {
        EMPTY,
        WATCHING,
        REGULAR;

    }
}

