/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.wallet;

import com.google.common.base.MoreObjects;
import com.google.protobuf.ByteString;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.bitcoinj.base.Network;
import org.bitcoinj.base.ScriptType;
import org.bitcoinj.base.internal.ByteUtils;
import org.bitcoinj.base.internal.InternalUtils;
import org.bitcoinj.base.internal.Preconditions;
import org.bitcoinj.base.internal.Stopwatch;
import org.bitcoinj.base.internal.StreamUtils;
import org.bitcoinj.base.internal.TimeUtils;
import org.bitcoinj.core.BloomFilter;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.crypto.AesKey;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.DeterministicHierarchy;
import org.bitcoinj.crypto.DeterministicKey;
import org.bitcoinj.crypto.ECKey;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.HDKeyDerivation;
import org.bitcoinj.crypto.HDPath;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.KeyCrypterScrypt;
import org.bitcoinj.crypto.LazyECPoint;
import org.bitcoinj.protobuf.wallet.Protos;
import org.bitcoinj.script.Script;
import org.bitcoinj.utils.ListenerRegistration;
import org.bitcoinj.utils.Threading;
import org.bitcoinj.wallet.BasicKeyChain;
import org.bitcoinj.wallet.DefaultKeyChainFactory;
import org.bitcoinj.wallet.DeterministicSeed;
import org.bitcoinj.wallet.EncryptableKeyChain;
import org.bitcoinj.wallet.KeyChain;
import org.bitcoinj.wallet.KeyChainFactory;
import org.bitcoinj.wallet.RedeemData;
import org.bitcoinj.wallet.UnreadableWalletException;
import org.bitcoinj.wallet.listeners.KeyChainEventListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeterministicKeyChain
implements EncryptableKeyChain {
    private static final Logger log = LoggerFactory.getLogger(DeterministicKeyChain.class);
    protected final ReentrantLock lock = Threading.lock(DeterministicKeyChain.class);
    public static final String DEFAULT_PASSPHRASE_FOR_MNEMONIC = "";
    private DeterministicHierarchy hierarchy;
    @Nullable
    private DeterministicKey rootKey;
    @Nullable
    private final DeterministicSeed seed;
    private final ScriptType outputScriptType;
    private final HDPath accountPath;
    public static final HDPath ACCOUNT_ZERO_PATH = HDPath.M(ChildNumber.ZERO_HARDENED);
    public static final HDPath ACCOUNT_ONE_PATH = HDPath.M(ChildNumber.ONE_HARDENED);
    public static final HDPath BIP44_ACCOUNT_ZERO_PATH = HDPath.M(new ChildNumber(44, true)).extend(ChildNumber.ZERO_HARDENED, ChildNumber.ZERO_HARDENED);
    public static final HDPath EXTERNAL_SUBPATH = HDPath.M(ChildNumber.ZERO);
    public static final HDPath INTERNAL_SUBPATH = HDPath.M(ChildNumber.ONE);
    private static final int LAZY_CALCULATE_LOOKAHEAD = -1;
    protected int lookaheadSize = 100;
    protected int lookaheadThreshold = this.calcDefaultLookaheadThreshold();
    private DeterministicKey externalParentKey;
    private DeterministicKey internalParentKey;
    private int issuedExternalKeys;
    private int issuedInternalKeys;
    private int keyLookaheadEpoch;
    private final BasicKeyChain basicKeyChain;
    private boolean isFollowing;
    private int sigsRequiredToSpend = 1;

    private int calcDefaultLookaheadThreshold() {
        return this.lookaheadSize / 3;
    }

    public static Builder<?> builder() {
        return new Builder();
    }

    public DeterministicKeyChain(DeterministicKey key, boolean isFollowing, boolean isWatching, ScriptType outputScriptType) {
        if (isWatching) {
            Preconditions.checkArgument(key.isPubKeyOnly(), () -> "private subtrees not currently supported for watching keys: if you got this key from DKC.getWatchingKey() then use .dropPrivate().dropParent() on it first");
        } else {
            Preconditions.checkArgument(key.hasPrivKey(), () -> "private subtrees are required");
        }
        Preconditions.checkArgument(isWatching || !isFollowing, () -> "can only follow a key that is watched");
        this.basicKeyChain = new BasicKeyChain();
        this.seed = null;
        this.rootKey = null;
        this.basicKeyChain.importKey(key);
        this.hierarchy = new DeterministicHierarchy(key);
        this.accountPath = key.getPath();
        this.outputScriptType = outputScriptType;
        this.initializeHierarchyUnencrypted();
        this.isFollowing = isFollowing;
    }

    protected DeterministicKeyChain(DeterministicSeed seed, @Nullable KeyCrypter crypter, ScriptType outputScriptType, List<ChildNumber> accountPath) {
        Preconditions.checkArgument(outputScriptType == null || outputScriptType == ScriptType.P2PKH || outputScriptType == ScriptType.P2WPKH, () -> "only P2PKH or P2WPKH allowed");
        this.outputScriptType = outputScriptType != null ? outputScriptType : ScriptType.P2PKH;
        this.accountPath = HDPath.M(accountPath);
        this.seed = seed;
        this.basicKeyChain = new BasicKeyChain(crypter);
        if (!seed.isEncrypted()) {
            this.rootKey = HDKeyDerivation.createMasterPrivateKey(Objects.requireNonNull(seed.getSeedBytes()));
            Optional<Instant> creationTime = seed.getCreationTime();
            if (creationTime.isPresent()) {
                this.rootKey.setCreationTime(creationTime.get());
            } else {
                this.rootKey.clearCreationTime();
            }
            this.basicKeyChain.importKey(this.rootKey);
            this.hierarchy = new DeterministicHierarchy(this.rootKey);
            for (HDPath path : this.getAccountPath().ancestors(true)) {
                this.basicKeyChain.importKey(this.hierarchy.get(path, false, true));
            }
            this.initializeHierarchyUnencrypted();
        }
    }

    protected DeterministicKeyChain(KeyCrypter crypter, AesKey aesKey, DeterministicKeyChain chain) {
        Objects.requireNonNull(chain.rootKey);
        Objects.requireNonNull(chain.seed);
        Preconditions.checkArgument(!chain.rootKey.isEncrypted(), () -> "chain already encrypted");
        this.accountPath = chain.getAccountPath();
        this.outputScriptType = chain.outputScriptType;
        this.issuedExternalKeys = chain.issuedExternalKeys;
        this.issuedInternalKeys = chain.issuedInternalKeys;
        this.lookaheadSize = chain.lookaheadSize;
        this.lookaheadThreshold = chain.lookaheadThreshold;
        this.seed = chain.seed.encrypt(crypter, aesKey);
        this.basicKeyChain = new BasicKeyChain(crypter);
        this.rootKey = chain.rootKey.encrypt(crypter, aesKey, null);
        this.hierarchy = new DeterministicHierarchy(this.rootKey);
        this.basicKeyChain.importKey(this.rootKey);
        for (HDPath path : this.getAccountPath().ancestors()) {
            this.encryptNonLeaf(aesKey, chain, this.rootKey, path);
        }
        DeterministicKey account = this.encryptNonLeaf(aesKey, chain, this.rootKey, this.getAccountPath());
        this.externalParentKey = this.encryptNonLeaf(aesKey, chain, account, this.getAccountPath().extend(EXTERNAL_SUBPATH));
        this.internalParentKey = this.encryptNonLeaf(aesKey, chain, account, this.getAccountPath().extend(INTERNAL_SUBPATH));
        for (DeterministicKey deterministicKey : chain.getLeafKeys()) {
            this.putKey(DeterministicKeyChain.cloneKey(this.hierarchy, deterministicKey));
        }
        for (ListenerRegistration listenerRegistration : chain.basicKeyChain.getListeners()) {
            this.basicKeyChain.addEventListener(listenerRegistration);
        }
    }

    public HDPath getAccountPath() {
        return this.accountPath;
    }

    public ScriptType getOutputScriptType() {
        return this.outputScriptType;
    }

    private DeterministicKey encryptNonLeaf(AesKey aesKey, DeterministicKeyChain chain, DeterministicKey parent, List<ChildNumber> path) {
        DeterministicKey key = chain.hierarchy.get(path, false, false);
        key = key.encrypt(Objects.requireNonNull(this.basicKeyChain.getKeyCrypter()), aesKey, parent);
        this.putKey(key);
        return key;
    }

    private void initializeHierarchyUnencrypted() {
        this.externalParentKey = this.hierarchy.deriveChild(this.getAccountPath(), false, false, ChildNumber.ZERO);
        this.internalParentKey = this.hierarchy.deriveChild(this.getAccountPath(), false, false, ChildNumber.ONE);
        this.basicKeyChain.importKey(this.externalParentKey);
        this.basicKeyChain.importKey(this.internalParentKey);
    }

    @Override
    public DeterministicKey getKey(KeyChain.KeyPurpose purpose) {
        return this.getKeys(purpose, 1).get(0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeterministicKey> getKeys(KeyChain.KeyPurpose purpose, int numberOfKeys) {
        Preconditions.checkArgument(numberOfKeys > 0);
        this.lock.lock();
        try {
            DeterministicKey parentKey;
            int index;
            switch (purpose) {
                case RECEIVE_FUNDS: 
                case REFUND: {
                    this.issuedExternalKeys += numberOfKeys;
                    index = this.issuedExternalKeys;
                    parentKey = this.externalParentKey;
                    break;
                }
                case AUTHENTICATION: 
                case CHANGE: {
                    this.issuedInternalKeys += numberOfKeys;
                    index = this.issuedInternalKeys;
                    parentKey = this.internalParentKey;
                    break;
                }
                default: {
                    throw new UnsupportedOperationException();
                }
            }
            List<DeterministicKey> lookahead = this.maybeLookAhead(parentKey, index, 0, 0);
            this.putKeys(lookahead);
            ArrayList<DeterministicKey> keys2 = new ArrayList<DeterministicKey>(numberOfKeys);
            for (int i2 = 0; i2 < numberOfKeys; ++i2) {
                HDPath path = parentKey.getPath().extend(new ChildNumber(index - numberOfKeys + i2, false), new ChildNumber[0]);
                DeterministicKey k = this.hierarchy.get(path, false, false);
                this.checkForBitFlip(k);
                keys2.add(k);
            }
            ArrayList<DeterministicKey> arrayList = keys2;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void putKey(DeterministicKey key) {
        this.hierarchy.putKey(key);
        this.basicKeyChain.importKey(key);
    }

    private void putKeys(List<DeterministicKey> keys2) {
        this.hierarchy.putKeys(keys2);
        this.basicKeyChain.importKeys(keys2);
    }

    private static DeterministicKey cloneKey(DeterministicHierarchy hierarchy, DeterministicKey key) {
        DeterministicKey parent = hierarchy.get(Objects.requireNonNull(key.getParent()).getPath(), false, false);
        return new DeterministicKey(key.dropPrivateBytes(), parent);
    }

    private void checkForBitFlip(DeterministicKey k) {
        DeterministicKey parent = Objects.requireNonNull(k.getParent());
        byte[] rederived = HDKeyDerivation.deriveChildKeyBytesFromPublic((DeterministicKey)parent, (ChildNumber)k.getChildNumber(), (HDKeyDerivation.PublicDeriveMode)HDKeyDerivation.PublicDeriveMode.WITH_INVERSION).keyBytes;
        byte[] actual = k.getPubKey();
        if (!Arrays.equals(rederived, actual)) {
            throw new IllegalStateException(String.format(Locale.US, "Bit-flip check failed: %s vs %s", Arrays.toString(rederived), Arrays.toString(actual)));
        }
    }

    public DeterministicKey markKeyAsUsed(DeterministicKey k) {
        int numChildren = k.getChildNumber().i() + 1;
        if (k.getParent() == this.internalParentKey) {
            if (this.issuedInternalKeys < numChildren) {
                this.issuedInternalKeys = numChildren;
                this.maybeLookAhead();
            }
        } else if (k.getParent() == this.externalParentKey && this.issuedExternalKeys < numChildren) {
            this.issuedExternalKeys = numChildren;
            this.maybeLookAhead();
        }
        return k;
    }

    public DeterministicKey findKeyFromPubHash(byte[] pubkeyHash) {
        this.lock.lock();
        try {
            DeterministicKey deterministicKey = (DeterministicKey)this.basicKeyChain.findKeyFromPubHash(pubkeyHash);
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    public DeterministicKey findKeyFromPubKey(byte[] pubkey) {
        this.lock.lock();
        try {
            DeterministicKey deterministicKey = (DeterministicKey)this.basicKeyChain.findKeyFromPubKey(pubkey);
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public DeterministicKey markPubHashAsUsed(byte[] pubkeyHash) {
        this.lock.lock();
        try {
            DeterministicKey k = (DeterministicKey)this.basicKeyChain.findKeyFromPubHash(pubkeyHash);
            if (k != null) {
                this.markKeyAsUsed(k);
            }
            DeterministicKey deterministicKey = k;
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public DeterministicKey markPubKeyAsUsed(byte[] pubkey) {
        this.lock.lock();
        try {
            DeterministicKey k = (DeterministicKey)this.basicKeyChain.findKeyFromPubKey(pubkey);
            if (k != null) {
                this.markKeyAsUsed(k);
            }
            DeterministicKey deterministicKey = k;
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean hasKey(ECKey key) {
        this.lock.lock();
        try {
            boolean bl = this.basicKeyChain.hasKey(key);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    protected DeterministicKey getKeyByPath(ChildNumber ... path) {
        return this.getKeyByPath(HDPath.M(Arrays.asList(path)));
    }

    protected DeterministicKey getKeyByPath(List<ChildNumber> path) {
        return this.getKeyByPath(path, false);
    }

    public DeterministicKey getKeyByPath(List<ChildNumber> path, boolean create) {
        return this.hierarchy.get(path, false, create);
    }

    @Nullable
    public DeterministicKey getRootKey() {
        return this.rootKey;
    }

    public DeterministicKey getWatchingKey() {
        return this.getKeyByPath(this.getAccountPath());
    }

    public boolean isWatching() {
        return this.getWatchingKey().isWatching();
    }

    @Override
    public int numKeys() {
        this.lock.lock();
        try {
            this.maybeLookAhead();
            int n = this.basicKeyChain.numKeys();
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int numLeafKeysIssued() {
        this.lock.lock();
        try {
            int n = this.issuedExternalKeys + this.issuedInternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Instant earliestKeyCreationTime() {
        return (this.seed != null ? this.seed.getCreationTime() : this.getWatchingKey().getCreationTime()).orElse(Instant.EPOCH);
    }

    @Override
    public void addEventListener(KeyChainEventListener listener) {
        this.basicKeyChain.addEventListener(listener);
    }

    @Override
    public void addEventListener(KeyChainEventListener listener, Executor executor) {
        this.basicKeyChain.addEventListener(listener, executor);
    }

    @Override
    public boolean removeEventListener(KeyChainEventListener listener) {
        return this.basicKeyChain.removeEventListener(listener);
    }

    @Nullable
    public List<String> getMnemonicCode() {
        if (this.seed == null) {
            return null;
        }
        this.lock.lock();
        try {
            List<String> list = this.seed.getMnemonicCode();
            return list;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isFollowing() {
        return this.isFollowing;
    }

    @Override
    public List<Protos.Key> serializeToProtobuf() {
        this.lock.lock();
        try {
            List<Protos.Key> list = this.serializeMyselfToProtobuf();
            return list;
        }
        finally {
            this.lock.unlock();
        }
    }

    protected List<Protos.Key> serializeMyselfToProtobuf() {
        LinkedList<Protos.Key> entries = new LinkedList<Protos.Key>();
        if (this.seed != null) {
            Protos.Key.Builder mnemonicEntry = BasicKeyChain.serializeEncryptableItem(this.seed);
            mnemonicEntry.setType(Protos.Key.Type.DETERMINISTIC_MNEMONIC);
            DeterministicKeyChain.serializeSeedEncryptableItem(this.seed, mnemonicEntry);
            for (ChildNumber childNumber : this.getAccountPath()) {
                mnemonicEntry.addAccountPath(childNumber.i());
            }
            entries.add((Protos.Key)mnemonicEntry.build());
        }
        Map<ECKey, Protos.Key.Builder> keys2 = this.basicKeyChain.serializeToEditableProtobufs();
        for (Map.Entry<ECKey, Protos.Key.Builder> entry : keys2.entrySet()) {
            DeterministicKey key = (DeterministicKey)entry.getKey();
            Protos.Key.Builder proto = entry.getValue();
            proto.setType(Protos.Key.Type.DETERMINISTIC_KEY);
            Protos.DeterministicKey.Builder detKey = (Protos.DeterministicKey.Builder)proto.getDeterministicKey().toBuilder();
            detKey.setChainCode(ByteString.copyFrom(key.getChainCode()));
            for (ChildNumber num : key.getPath()) {
                detKey.addPath(num.i());
            }
            if (key.equals(this.externalParentKey)) {
                detKey.setIssuedSubkeys(this.issuedExternalKeys);
                detKey.setLookaheadSize(this.lookaheadSize);
                detKey.setSigsRequiredToSpend(this.getSigsRequiredToSpend());
            } else if (key.equals(this.internalParentKey)) {
                detKey.setIssuedSubkeys(this.issuedInternalKeys);
                detKey.setLookaheadSize(this.lookaheadSize);
                detKey.setSigsRequiredToSpend(this.getSigsRequiredToSpend());
            }
            if (entries.isEmpty() && this.isFollowing()) {
                detKey.setIsFollowing(true);
            }
            proto.setDeterministicKey(detKey);
            if (key.getParent() != null) {
                proto.clearCreationTimestamp();
            } else {
                proto.setOutputScriptType(Protos.Key.OutputScriptType.valueOf(this.outputScriptType.name()));
            }
            entries.add((Protos.Key)proto.build());
        }
        return entries;
    }

    static List<DeterministicKeyChain> fromProtobuf(List<Protos.Key> keys2, @Nullable KeyCrypter crypter) throws UnreadableWalletException {
        return DeterministicKeyChain.fromProtobuf(keys2, crypter, new DefaultKeyChainFactory());
    }

    public static List<DeterministicKeyChain> fromProtobuf(List<Protos.Key> keys2, @Nullable KeyCrypter crypter, KeyChainFactory factory2) throws UnreadableWalletException {
        LinkedList<DeterministicKeyChain> chains = new LinkedList<DeterministicKeyChain>();
        DeterministicSeed seed = null;
        DeterministicKeyChain chain = null;
        int lookaheadSize = -1;
        int sigsRequiredToSpend = 1;
        HDPath accountPath = HDPath.M();
        ScriptType outputScriptType = ScriptType.P2PKH;
        for (Protos.Key key : keys2) {
            DeterministicKey detkey;
            Protos.Key.Type t = key.getType();
            if (t == Protos.Key.Type.DETERMINISTIC_MNEMONIC) {
                accountPath = DeterministicKeyChain.deserializeAccountPath(key.getAccountPathList());
                if (chain != null) {
                    DeterministicKeyChain.addChain(chains, chain, lookaheadSize, sigsRequiredToSpend);
                    chain = null;
                }
                Instant seedCreationTime = Instant.ofEpochMilli(key.getCreationTimestamp());
                String passphrase = DEFAULT_PASSPHRASE_FOR_MNEMONIC;
                if (key.hasSecretBytes()) {
                    if (key.hasEncryptedDeterministicSeed()) {
                        throw new UnreadableWalletException("Malformed key proto: " + key);
                    }
                    byte[] seedBytes = null;
                    if (key.hasDeterministicSeed()) {
                        seedBytes = key.getDeterministicSeed().toByteArray();
                    }
                    seed = new DeterministicSeed(key.getSecretBytes().toStringUtf8(), seedBytes, passphrase, seedCreationTime);
                } else if (key.hasEncryptedData()) {
                    if (key.hasDeterministicSeed()) {
                        throw new UnreadableWalletException("Malformed key proto: " + key);
                    }
                    EncryptedData data = new EncryptedData(key.getEncryptedData().getInitialisationVector().toByteArray(), key.getEncryptedData().getEncryptedPrivateKey().toByteArray());
                    EncryptedData encryptedSeedBytes = null;
                    if (key.hasEncryptedDeterministicSeed()) {
                        Protos.EncryptedData encryptedSeed = key.getEncryptedDeterministicSeed();
                        encryptedSeedBytes = new EncryptedData(encryptedSeed.getInitialisationVector().toByteArray(), encryptedSeed.getEncryptedPrivateKey().toByteArray());
                    }
                    seed = new DeterministicSeed(data, encryptedSeedBytes, seedCreationTime);
                } else {
                    throw new UnreadableWalletException("Malformed key proto: " + key);
                }
                if (!log.isDebugEnabled()) continue;
                log.debug("Deserializing: DETERMINISTIC_MNEMONIC: {}", (Object)seed);
                continue;
            }
            if (t != Protos.Key.Type.DETERMINISTIC_KEY) continue;
            if (!key.hasDeterministicKey()) {
                throw new UnreadableWalletException("Deterministic key missing extra data: " + key);
            }
            byte[] chainCode = key.getDeterministicKey().getChainCode().toByteArray();
            LazyECPoint pubkey = new LazyECPoint(key.getPublicKey().toByteArray());
            HDPath path = HDPath.deserialize(key.getDeterministicKey().getPathList());
            if (key.hasOutputScriptType()) {
                outputScriptType = ScriptType.valueOf(key.getOutputScriptType().name());
            }
            boolean isWatchingAccountKey = false;
            boolean isFollowingKey = false;
            boolean isSpendingKey = false;
            if (key.getDeterministicKey().getIsFollowing()) {
                if (chain != null) {
                    DeterministicKeyChain.addChain(chains, chain, lookaheadSize, sigsRequiredToSpend);
                    chain = null;
                    seed = null;
                }
                isFollowingKey = true;
            }
            if (chain == null) {
                DeterministicKey accountKey;
                if (seed == null && key.hasSecretBytes()) {
                    accountKey = new DeterministicKey(path, chainCode, pubkey, ByteUtils.bytesToBigInteger(key.getSecretBytes().toByteArray()), null);
                    accountKey.setCreationTime(Instant.ofEpochMilli(key.getCreationTimestamp()));
                    chain = factory2.makeSpendingKeyChain(accountKey, outputScriptType);
                    isSpendingKey = true;
                } else if (seed == null) {
                    accountKey = new DeterministicKey(path, chainCode, pubkey, null, null);
                    accountKey.setCreationTime(Instant.ofEpochMilli(key.getCreationTimestamp()));
                    chain = factory2.makeWatchingKeyChain(accountKey, outputScriptType);
                    isWatchingAccountKey = true;
                } else {
                    chain = factory2.makeKeyChain(seed, crypter, outputScriptType, accountPath);
                    chain.lookaheadSize = -1;
                }
            }
            DeterministicKey parent = null;
            if (!(path.isEmpty() || isWatchingAccountKey || isSpendingKey)) {
                parent = chain.hierarchy.get(path.parent(), false, false);
            }
            if (key.hasSecretBytes()) {
                BigInteger priv = ByteUtils.bytesToBigInteger(key.getSecretBytes().toByteArray());
                detkey = new DeterministicKey(path, chainCode, pubkey, priv, parent);
            } else if (key.hasEncryptedData()) {
                Protos.EncryptedData proto = key.getEncryptedData();
                EncryptedData data = new EncryptedData(proto.getInitialisationVector().toByteArray(), proto.getEncryptedPrivateKey().toByteArray());
                Objects.requireNonNull(crypter, "Encountered an encrypted key but no key crypter provided");
                detkey = new DeterministicKey((List<ChildNumber>)path, chainCode, crypter, pubkey, data, parent);
            } else {
                detkey = new DeterministicKey(path, chainCode, pubkey, null, parent);
            }
            if (key.hasCreationTimestamp()) {
                detkey.setCreationTime(Instant.ofEpochMilli(key.getCreationTimestamp()));
            }
            if (log.isDebugEnabled()) {
                log.debug("Deserializing: DETERMINISTIC_KEY: {}", (Object)detkey);
            }
            if (!isWatchingAccountKey) {
                if (path.isEmpty()) {
                    if (chain.rootKey == null) {
                        chain.rootKey = detkey;
                        chain.hierarchy = new DeterministicHierarchy(detkey);
                    }
                } else if (path.size() == chain.getAccountPath().size() + 1 || isSpendingKey) {
                    if (detkey.getChildNumber().num() == 0) {
                        chain.externalParentKey = detkey;
                        chain.issuedExternalKeys = key.getDeterministicKey().getIssuedSubkeys();
                        lookaheadSize = Math.max(lookaheadSize, key.getDeterministicKey().getLookaheadSize());
                        sigsRequiredToSpend = key.getDeterministicKey().getSigsRequiredToSpend();
                    } else if (detkey.getChildNumber().num() == 1) {
                        chain.internalParentKey = detkey;
                        chain.issuedInternalKeys = key.getDeterministicKey().getIssuedSubkeys();
                    }
                }
            }
            chain.putKey(detkey);
        }
        if (chain != null) {
            DeterministicKeyChain.addChain(chains, chain, lookaheadSize, sigsRequiredToSpend);
        }
        return chains;
    }

    private static void addChain(List<DeterministicKeyChain> chains, DeterministicKeyChain chain, int lookaheadSize, int sigsRequiredToSpend) {
        Preconditions.checkState(lookaheadSize >= 0);
        chain.setLookaheadSize(lookaheadSize);
        chain.setSigsRequiredToSpend(sigsRequiredToSpend);
        chain.maybeLookAhead();
        chains.add(chain);
    }

    private static HDPath deserializeAccountPath(List<Integer> integerList) {
        HDPath path = HDPath.deserialize(integerList);
        return path.isEmpty() ? ACCOUNT_ZERO_PATH : path;
    }

    @Override
    public DeterministicKeyChain toEncrypted(CharSequence password) {
        Objects.requireNonNull(password);
        Preconditions.checkArgument(password.length() > 0);
        Preconditions.checkState(this.seed != null, () -> "attempt to encrypt a watching chain");
        Preconditions.checkState(!this.seed.isEncrypted());
        KeyCrypterScrypt scrypt = new KeyCrypterScrypt();
        AesKey derivedKey = scrypt.deriveKey(password);
        return this.toEncrypted(scrypt, derivedKey);
    }

    @Override
    public DeterministicKeyChain toEncrypted(KeyCrypter keyCrypter, AesKey aesKey) {
        return new DeterministicKeyChain(keyCrypter, aesKey, this);
    }

    @Override
    public DeterministicKeyChain toDecrypted(CharSequence password) {
        Objects.requireNonNull(password);
        Preconditions.checkArgument(password.length() > 0);
        KeyCrypter crypter = this.getKeyCrypter();
        Preconditions.checkState(crypter != null, () -> "chain not encrypted");
        AesKey derivedKey = crypter.deriveKey(password);
        return this.toDecrypted(derivedKey);
    }

    @Override
    public DeterministicKeyChain toDecrypted(AesKey aesKey) {
        Preconditions.checkState(this.getKeyCrypter() != null, () -> "key chain not encrypted");
        Preconditions.checkState(this.seed != null, () -> "can't decrypt a watching chain");
        Preconditions.checkState(this.seed.isEncrypted());
        String passphrase = DEFAULT_PASSPHRASE_FOR_MNEMONIC;
        DeterministicSeed decSeed = this.seed.decrypt(this.getKeyCrypter(), passphrase, aesKey);
        DeterministicKeyChain chain = this.makeKeyChainFromSeed(decSeed, this.getAccountPath(), this.outputScriptType);
        if (!chain.getWatchingKey().getPubKeyPoint().equals(this.getWatchingKey().getPubKeyPoint())) {
            throw new KeyCrypterException.PublicPrivateMismatch("Provided AES key is wrong");
        }
        chain.lookaheadSize = this.lookaheadSize;
        for (DeterministicKey deterministicKey : this.getLeafKeys()) {
            Preconditions.checkState(deterministicKey.isEncrypted());
            chain.putKey(DeterministicKeyChain.cloneKey(chain.hierarchy, deterministicKey));
        }
        chain.issuedExternalKeys = this.issuedExternalKeys;
        chain.issuedInternalKeys = this.issuedInternalKeys;
        for (ListenerRegistration listenerRegistration : this.basicKeyChain.getListeners()) {
            chain.basicKeyChain.addEventListener(listenerRegistration);
        }
        return chain;
    }

    protected DeterministicKeyChain makeKeyChainFromSeed(DeterministicSeed seed, List<ChildNumber> accountPath, ScriptType outputScriptType) {
        return new DeterministicKeyChain(seed, null, outputScriptType, accountPath);
    }

    @Override
    public boolean checkPassword(CharSequence password) {
        Objects.requireNonNull(password);
        Preconditions.checkState(this.getKeyCrypter() != null, () -> "key chain not encrypted");
        return this.checkAESKey(this.getKeyCrypter().deriveKey(password));
    }

    @Override
    public boolean checkAESKey(AesKey aesKey) {
        Preconditions.checkState(this.rootKey != null, () -> "can't check password for a watching chain");
        Objects.requireNonNull(aesKey);
        Preconditions.checkState(this.getKeyCrypter() != null, () -> "key chain not encrypted");
        try {
            return this.rootKey.decrypt(aesKey).getPubKeyPoint().equals(this.rootKey.getPubKeyPoint());
        }
        catch (KeyCrypterException e) {
            return false;
        }
    }

    @Override
    @Nullable
    public KeyCrypter getKeyCrypter() {
        return this.basicKeyChain.getKeyCrypter();
    }

    @Override
    public int numBloomFilterEntries() {
        return this.numKeys() * 2;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BloomFilter getFilter(int size, double falsePositiveRate, int tweak) {
        this.lock.lock();
        try {
            Preconditions.checkArgument(size >= this.numBloomFilterEntries());
            this.maybeLookAhead();
            BloomFilter bloomFilter = this.basicKeyChain.getFilter(size, falsePositiveRate, tweak);
            return bloomFilter;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getLookaheadSize() {
        this.lock.lock();
        try {
            int n = this.lookaheadSize;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setLookaheadSize(int lookaheadSize) {
        this.lock.lock();
        try {
            boolean readjustThreshold = this.lookaheadThreshold == this.calcDefaultLookaheadThreshold();
            this.lookaheadSize = lookaheadSize;
            if (readjustThreshold) {
                this.lookaheadThreshold = this.calcDefaultLookaheadThreshold();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setLookaheadThreshold(int num) {
        this.lock.lock();
        try {
            if (num >= this.lookaheadSize) {
                throw new IllegalArgumentException("Threshold larger or equal to the lookaheadSize");
            }
            this.lookaheadThreshold = num;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getLookaheadThreshold() {
        this.lock.lock();
        try {
            if (this.lookaheadThreshold >= this.lookaheadSize) {
                int n = 0;
                return n;
            }
            int n = this.lookaheadThreshold;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void maybeLookAhead() {
        this.lock.lock();
        try {
            List<DeterministicKey> keys2 = this.concatLists(this.maybeLookAhead(this.externalParentKey, this.issuedExternalKeys), this.maybeLookAhead(this.internalParentKey, this.issuedInternalKeys));
            if (!keys2.isEmpty()) {
                ++this.keyLookaheadEpoch;
                this.putKeys(keys2);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private <T> List<T> concatLists(List<T> list1, List<T> list2) {
        return Stream.concat(list1.stream(), list2.stream()).collect(StreamUtils.toUnmodifiableList());
    }

    private List<DeterministicKey> maybeLookAhead(DeterministicKey parent, int issued) {
        Preconditions.checkState(this.lock.isHeldByCurrentThread());
        return this.maybeLookAhead(parent, issued, this.getLookaheadSize(), this.getLookaheadThreshold());
    }

    private List<DeterministicKey> maybeLookAhead(DeterministicKey parent, int issued, int lookaheadSize, int lookaheadThreshold) {
        Preconditions.checkState(this.lock.isHeldByCurrentThread());
        int numChildren = this.hierarchy.getNumChildren(parent.getPath());
        int needed = issued + lookaheadSize + lookaheadThreshold - numChildren;
        int limit = needed > lookaheadThreshold ? needed : 0;
        log.info("{} keys needed for {} = {} issued + {} lookahead size + {} lookahead threshold - {} num children", limit, parent.getPathAsString(), issued, lookaheadSize, lookaheadThreshold, numChildren);
        Stopwatch watch = Stopwatch.start();
        List<DeterministicKey> result = HDKeyDerivation.generate(parent, numChildren).limit(limit).map(DeterministicKey::dropPrivateBytes).collect(StreamUtils.toUnmodifiableList());
        log.info("Took {}", (Object)watch);
        return result;
    }

    public void maybeLookAheadScripts() {
    }

    public int getIssuedExternalKeys() {
        this.lock.lock();
        try {
            int n = this.issuedExternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getIssuedInternalKeys() {
        this.lock.lock();
        try {
            int n = this.issuedInternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Nullable
    public DeterministicSeed getSeed() {
        this.lock.lock();
        try {
            DeterministicSeed deterministicSeed = this.seed;
            return deterministicSeed;
        }
        finally {
            this.lock.unlock();
        }
    }

    List<DeterministicKey> getKeys(boolean includeLookahead, boolean includeParents) {
        return this.getKeys(this.filterKeys(includeLookahead, includeParents));
    }

    private Predicate<DeterministicKey> filterKeys(boolean includeLookahead, boolean includeParents) {
        Predicate<DeterministicKey> keyFilter;
        if (!includeLookahead) {
            int treeSize = this.internalParentKey.getPath().size();
            keyFilter = key -> {
                DeterministicKey parent = key.getParent();
                return !(!includeParents && parent == null || !includeParents && key.getPath().size() <= treeSize || this.internalParentKey.equals(parent) && key.getChildNumber().i() >= this.issuedInternalKeys || this.externalParentKey.equals(parent) && key.getChildNumber().i() >= this.issuedExternalKeys);
            };
        } else {
            keyFilter = key -> true;
        }
        return keyFilter;
    }

    private List<DeterministicKey> getKeys(Predicate<DeterministicKey> keyFilter) {
        return this.basicKeyChain.getKeys().stream().map(key -> (DeterministicKey)key).filter(keyFilter).collect(StreamUtils.toUnmodifiableList());
    }

    public List<DeterministicKey> getIssuedReceiveKeys() {
        return this.getKeys(this.filterKeys(false, false).and(key -> this.externalParentKey.equals(key.getParent())));
    }

    public List<DeterministicKey> getLeafKeys() {
        return this.getKeys(key -> key.getPath().size() == this.getAccountPath().size() + 2);
    }

    static void serializeSeedEncryptableItem(DeterministicSeed seed, Protos.Key.Builder proto) {
        if (seed.isEncrypted() && seed.getEncryptedSeedData() != null) {
            EncryptedData data = seed.getEncryptedSeedData();
            proto.setEncryptedDeterministicSeed(((Protos.EncryptedData.Builder)proto.getEncryptedDeterministicSeed().toBuilder()).setEncryptedPrivateKey(ByteString.copyFrom(data.encryptedBytes)).setInitialisationVector(ByteString.copyFrom(data.initialisationVector)));
            Preconditions.checkState(seed.getEncryptionType() == Protos.Wallet.EncryptionType.ENCRYPTED_SCRYPT_AES);
        } else {
            byte[] secret = seed.getSeedBytes();
            if (secret != null) {
                proto.setDeterministicSeed(ByteString.copyFrom(secret));
            }
        }
    }

    public int getKeyLookaheadEpoch() {
        this.lock.lock();
        try {
            int n = this.keyLookaheadEpoch;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public RedeemData getRedeemData(DeterministicKey followedKey) {
        throw new UnsupportedOperationException();
    }

    public Script freshOutputScript(KeyChain.KeyPurpose purpose) {
        throw new UnsupportedOperationException();
    }

    public String toString() {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this).omitNullValues();
        helper.addValue((Object)this.outputScriptType);
        helper.add("accountPath", (Object)this.accountPath);
        helper.add("lookaheadSize", this.lookaheadSize);
        helper.add("lookaheadThreshold", this.lookaheadThreshold);
        if (this.isFollowing) {
            helper.addValue((Object)"following");
        }
        return helper.toString();
    }

    public String toString(boolean includeLookahead, boolean includePrivateKeys, @Nullable AesKey aesKey, Network network) {
        DeterministicKey watchingKey = this.getWatchingKey();
        StringBuilder builder = new StringBuilder();
        if (this.seed != null) {
            if (includePrivateKeys) {
                DeterministicSeed decryptedSeed = this.seed.isEncrypted() ? this.seed.decrypt(this.getKeyCrypter(), DEFAULT_PASSPHRASE_FOR_MNEMONIC, aesKey) : this.seed;
                List<String> words = decryptedSeed.getMnemonicCode();
                builder.append("Seed as words:     ").append(InternalUtils.SPACE_JOINER.join(words)).append('\n');
                builder.append("Seed as hex:       ").append(decryptedSeed.toHexString()).append('\n');
            } else if (this.seed.isEncrypted()) {
                builder.append("Seed is encrypted\n");
            }
            builder.append("Seed birthday:     ");
            Optional<Instant> seedCreationTime = this.seed.getCreationTime();
            if (seedCreationTime.isPresent()) {
                builder.append(seedCreationTime.get().getEpochSecond()).append("  [").append(TimeUtils.dateTimeFormat(seedCreationTime.get())).append("]");
            } else {
                builder.append("unknown");
            }
            builder.append("\n");
        } else {
            builder.append("Key birthday:      ");
            Optional<Instant> watchingKeyCreationTime = watchingKey.getCreationTime();
            if (watchingKeyCreationTime.isPresent()) {
                builder.append(watchingKeyCreationTime.get().getEpochSecond()).append("  [").append(TimeUtils.dateTimeFormat(watchingKeyCreationTime.get())).append("]");
            } else {
                builder.append("unknown");
            }
            builder.append("\n");
        }
        builder.append("Ouput script type: ").append((Object)this.outputScriptType).append('\n');
        builder.append("Key to watch:      ").append(watchingKey.serializePubB58(network, this.outputScriptType)).append('\n');
        builder.append("Lookahead siz/thr: ").append(this.lookaheadSize).append('/').append(this.lookaheadThreshold).append('\n');
        this.formatAddresses(includeLookahead, includePrivateKeys, aesKey, network, builder);
        return builder.toString();
    }

    @Deprecated
    public String toString(boolean includeLookahead, boolean includePrivateKeys, @Nullable AesKey aesKey, NetworkParameters params) {
        return this.toString(includeLookahead, includePrivateKeys, aesKey, params.network());
    }

    protected void formatAddresses(boolean includeLookahead, boolean includePrivateKeys, @Nullable AesKey aesKey, Network network, StringBuilder builder) {
        for (DeterministicKey key : this.getKeys(includeLookahead, true)) {
            String comment = null;
            if (key.equals(this.getRootKey())) {
                comment = "root";
            } else if (key.equals(this.getWatchingKey())) {
                comment = "account";
            } else if (key.equals(this.internalParentKey)) {
                comment = "internal";
            } else if (key.equals(this.externalParentKey)) {
                comment = "external";
            } else if (this.internalParentKey.equals(key.getParent()) && key.getChildNumber().i() >= this.issuedInternalKeys) {
                comment = "*";
            } else if (this.externalParentKey.equals(key.getParent()) && key.getChildNumber().i() >= this.issuedExternalKeys) {
                comment = "*";
            }
            key.formatKeyWithAddress(includePrivateKeys, aesKey, builder, network, this.outputScriptType, comment);
        }
    }

    public void setSigsRequiredToSpend(int sigsRequiredToSpend) {
        this.sigsRequiredToSpend = sigsRequiredToSpend;
    }

    public int getSigsRequiredToSpend() {
        return this.sigsRequiredToSpend;
    }

    @Nullable
    public RedeemData findRedeemDataByScriptHash(ByteString bytes) {
        return null;
    }

    public static class Builder<T extends Builder<T>> {
        protected SecureRandom random;
        protected int bits = 128;
        protected String passphrase;
        @Nullable
        protected Instant creationTime = null;
        protected byte[] entropy;
        protected DeterministicSeed seed;
        protected ScriptType outputScriptType = ScriptType.P2PKH;
        protected DeterministicKey watchingKey = null;
        protected DeterministicKey spendingKey = null;
        protected HDPath accountPath = null;

        protected Builder() {
        }

        protected T self() {
            return (T)this;
        }

        public T entropy(byte[] entropy, Instant creationTime) {
            this.entropy = entropy;
            this.creationTime = Objects.requireNonNull(creationTime);
            return this.self();
        }

        @Deprecated
        public T entropy(byte[] entropy, long creationTimeSecs) {
            Preconditions.checkArgument(creationTimeSecs > 0L);
            return this.entropy(entropy, Instant.ofEpochSecond(creationTimeSecs));
        }

        public T seed(DeterministicSeed seed) {
            this.seed = seed;
            return this.self();
        }

        public T random(SecureRandom random, int bits) {
            this.random = random;
            this.bits = bits;
            return this.self();
        }

        public T random(SecureRandom random) {
            this.random = random;
            return this.self();
        }

        public T watch(DeterministicKey accountKey) {
            Preconditions.checkState(this.accountPath == null, () -> "either watch or accountPath");
            this.watchingKey = accountKey;
            return this.self();
        }

        public T spend(DeterministicKey accountKey) {
            Preconditions.checkState(this.accountPath == null, () -> "either spend or accountPath");
            this.spendingKey = accountKey;
            return this.self();
        }

        public T outputScriptType(ScriptType outputScriptType) {
            this.outputScriptType = outputScriptType;
            return this.self();
        }

        public T passphrase(String passphrase) {
            this.passphrase = passphrase;
            return this.self();
        }

        public T accountPath(List<ChildNumber> accountPath) {
            Preconditions.checkState(this.watchingKey == null, () -> "either watch or accountPath");
            this.accountPath = HDPath.M(Objects.requireNonNull(accountPath));
            return this.self();
        }

        public DeterministicKeyChain build() {
            Preconditions.checkState(this.passphrase == null || this.seed == null, () -> "passphrase must not be specified with seed");
            if (this.accountPath == null) {
                this.accountPath = ACCOUNT_ZERO_PATH;
            }
            if (this.random != null) {
                return new DeterministicKeyChain(DeterministicSeed.ofRandom(this.random, this.bits, this.getPassphrase()), null, this.outputScriptType, this.accountPath);
            }
            if (this.entropy != null) {
                return new DeterministicKeyChain(DeterministicSeed.ofEntropy(this.entropy, this.getPassphrase(), this.creationTime), null, this.outputScriptType, this.accountPath);
            }
            if (this.seed != null) {
                return new DeterministicKeyChain(this.seed, null, this.outputScriptType, this.accountPath);
            }
            if (this.watchingKey != null) {
                return new DeterministicKeyChain(this.watchingKey, false, true, this.outputScriptType);
            }
            if (this.spendingKey != null) {
                return new DeterministicKeyChain(this.spendingKey, false, false, this.outputScriptType);
            }
            throw new IllegalStateException();
        }

        protected String getPassphrase() {
            return this.passphrase != null ? this.passphrase : DeterministicKeyChain.DEFAULT_PASSPHRASE_FOR_MNEMONIC;
        }
    }
}

