/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.aot.generate;

import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.stream.Stream;
import org.springframework.aot.generate.ValueCodeGenerator;
import org.springframework.core.ResolvableType;
import org.springframework.javapoet.CodeBlock;
import org.springframework.lang.Nullable;
import org.springframework.util.ClassUtils;
import org.springframework.util.ObjectUtils;

public abstract class ValueCodeGeneratorDelegates {
    public static final List<ValueCodeGenerator.Delegate> INSTANCES = List.of(new PrimitiveDelegate(), new StringDelegate(), new CharsetDelegate(), new EnumDelegate(), new ClassDelegate(), new ResolvableTypeDelegate(), new ArrayDelegate(), new ListDelegate(), new SetDelegate(), new MapDelegate());

    private static class PrimitiveDelegate
    implements ValueCodeGenerator.Delegate {
        private static final Map<Character, String> CHAR_ESCAPES = Map.of(Character.valueOf('\b'), "\\b", Character.valueOf('\t'), "\\t", Character.valueOf('\n'), "\\n", Character.valueOf('\f'), "\\f", Character.valueOf('\r'), "\\r", Character.valueOf('\"'), "\"", Character.valueOf('\''), "\\'", Character.valueOf('\\'), "\\\\");

        private PrimitiveDelegate() {
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator codeGenerator, Object value) {
            if (value instanceof Boolean || value instanceof Integer) {
                return CodeBlock.of("$L", value);
            }
            if (value instanceof Byte) {
                return CodeBlock.of("(byte) $L", value);
            }
            if (value instanceof Short) {
                return CodeBlock.of("(short) $L", value);
            }
            if (value instanceof Long) {
                return CodeBlock.of("$LL", value);
            }
            if (value instanceof Float) {
                return CodeBlock.of("$LF", value);
            }
            if (value instanceof Double) {
                return CodeBlock.of("(double) $L", value);
            }
            if (value instanceof Character) {
                Character character = (Character)value;
                return CodeBlock.of("'$L'", this.escape(character.charValue()));
            }
            return null;
        }

        private String escape(char ch) {
            String escaped = CHAR_ESCAPES.get(Character.valueOf(ch));
            if (escaped != null) {
                return escaped;
            }
            return !Character.isISOControl(ch) ? Character.toString(ch) : String.format("\\u%04x", ch);
        }
    }

    private static class StringDelegate
    implements ValueCodeGenerator.Delegate {
        private StringDelegate() {
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator codeGenerator, Object value) {
            if (value instanceof String) {
                return CodeBlock.of("$S", value);
            }
            return null;
        }
    }

    private static class CharsetDelegate
    implements ValueCodeGenerator.Delegate {
        private CharsetDelegate() {
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator codeGenerator, Object value) {
            if (value instanceof Charset) {
                Charset charset = (Charset)value;
                return CodeBlock.of("$T.forName($S)", Charset.class, charset.name());
            }
            return null;
        }
    }

    private static class EnumDelegate
    implements ValueCodeGenerator.Delegate {
        private EnumDelegate() {
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator codeGenerator, Object value) {
            if (value instanceof Enum) {
                Enum enumValue = (Enum)value;
                return CodeBlock.of("$T.$L", enumValue.getDeclaringClass(), enumValue.name());
            }
            return null;
        }
    }

    private static class ClassDelegate
    implements ValueCodeGenerator.Delegate {
        private ClassDelegate() {
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator codeGenerator, Object value) {
            if (value instanceof Class) {
                Class clazz = (Class)value;
                return CodeBlock.of("$T.class", ClassUtils.getUserClass(clazz));
            }
            return null;
        }
    }

    private static class ResolvableTypeDelegate
    implements ValueCodeGenerator.Delegate {
        private ResolvableTypeDelegate() {
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator codeGenerator, Object value) {
            if (value instanceof ResolvableType) {
                ResolvableType resolvableType = (ResolvableType)value;
                return ResolvableTypeDelegate.generateCode(resolvableType, false);
            }
            return null;
        }

        private static CodeBlock generateCode(ResolvableType resolvableType, boolean allowClassResult) {
            if (ResolvableType.NONE.equals(resolvableType)) {
                return CodeBlock.of("$T.NONE", ResolvableType.class);
            }
            Class<?> type = ClassUtils.getUserClass(resolvableType.toClass());
            if (resolvableType.hasGenerics() && resolvableType.hasResolvableGenerics()) {
                return ResolvableTypeDelegate.generateCodeWithGenerics(resolvableType, type);
            }
            if (allowClassResult) {
                return CodeBlock.of("$T.class", type);
            }
            return CodeBlock.of("$T.forClass($T.class)", ResolvableType.class, type);
        }

        private static CodeBlock generateCodeWithGenerics(ResolvableType target, Class<?> type) {
            ResolvableType[] generics = target.getGenerics();
            boolean hasNoNestedGenerics = Arrays.stream(generics).noneMatch(ResolvableType::hasGenerics);
            CodeBlock.Builder code = CodeBlock.builder();
            code.add("$T.forClassWithGenerics($T.class", ResolvableType.class, type);
            for (ResolvableType generic2 : generics) {
                code.add(", $L", ResolvableTypeDelegate.generateCode(generic2, hasNoNestedGenerics));
            }
            code.add(")", new Object[0]);
            return code.build();
        }
    }

    private static class ArrayDelegate
    implements ValueCodeGenerator.Delegate {
        private ArrayDelegate() {
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator codeGenerator, Object value) {
            if (value.getClass().isArray()) {
                Stream<CodeBlock> elements = Arrays.stream(ObjectUtils.toObjectArray(value)).map(codeGenerator::generateCode);
                CodeBlock.Builder code = CodeBlock.builder();
                code.add("new $T {", value.getClass());
                code.add(elements.collect(CodeBlock.joining(", ")));
                code.add("}", new Object[0]);
                return code.build();
            }
            return null;
        }
    }

    private static class ListDelegate
    extends CollectionDelegate<List<?>> {
        ListDelegate() {
            super(List.class, CodeBlock.of("$T.emptyList()", Collections.class));
        }
    }

    private static class SetDelegate
    extends CollectionDelegate<Set<?>> {
        SetDelegate() {
            super(Set.class, CodeBlock.of("$T.emptySet()", Collections.class));
        }

        @Override
        protected CodeBlock generateCollectionCode(ValueCodeGenerator valueCodeGenerator, Set<?> collection) {
            if (collection instanceof LinkedHashSet) {
                return CodeBlock.of("new $T($L)", LinkedHashSet.class, this.generateCollectionOf(valueCodeGenerator, collection, List.class));
            }
            return super.generateCollectionCode(valueCodeGenerator, this.orderForCodeConsistency(collection));
        }

        private Set<?> orderForCodeConsistency(Set<?> set) {
            try {
                return new TreeSet(set);
            }
            catch (ClassCastException ex) {
                return set;
            }
        }
    }

    public static class MapDelegate
    implements ValueCodeGenerator.Delegate {
        private static final CodeBlock EMPTY_RESULT = CodeBlock.of("$T.emptyMap()", Collections.class);

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator valueCodeGenerator, Object value) {
            if (value instanceof Map) {
                Map map = (Map)value;
                if (map.isEmpty()) {
                    return EMPTY_RESULT;
                }
                return this.generateMapCode(valueCodeGenerator, map);
            }
            return null;
        }

        @Nullable
        protected CodeBlock generateMapCode(ValueCodeGenerator valueCodeGenerator, Map<?, ?> map) {
            boolean useOfEntries = (map = this.orderForCodeConsistency(map)).size() > 10;
            CodeBlock.Builder code = CodeBlock.builder();
            code.add("$T" + (!useOfEntries ? ".of(" : ".ofEntries("), Map.class);
            Iterator<Map.Entry<?, ?>> iterator2 = map.entrySet().iterator();
            while (iterator2.hasNext()) {
                Map.Entry<?, ?> entry = iterator2.next();
                CodeBlock keyCode = valueCodeGenerator.generateCode(entry.getKey());
                CodeBlock valueCode = valueCodeGenerator.generateCode(entry.getValue());
                if (!useOfEntries) {
                    code.add("$L, $L", keyCode, valueCode);
                } else {
                    code.add("$T.entry($L,$L)", Map.class, keyCode, valueCode);
                }
                if (!iterator2.hasNext()) continue;
                code.add(", ", new Object[0]);
            }
            code.add(")", new Object[0]);
            return code.build();
        }

        private <K, V> Map<K, V> orderForCodeConsistency(Map<K, V> map) {
            try {
                return new TreeMap<K, V>(map);
            }
            catch (ClassCastException ex) {
                return map;
            }
        }
    }

    public static abstract class CollectionDelegate<T extends Collection<?>>
    implements ValueCodeGenerator.Delegate {
        private final Class<?> collectionType;
        private final CodeBlock emptyResult;

        protected CollectionDelegate(Class<?> collectionType, CodeBlock emptyResult) {
            this.collectionType = collectionType;
            this.emptyResult = emptyResult;
        }

        @Override
        @Nullable
        public CodeBlock generateCode(ValueCodeGenerator valueCodeGenerator, Object value) {
            if (this.collectionType.isInstance(value)) {
                Collection collection = (Collection)value;
                if (collection.isEmpty()) {
                    return this.emptyResult;
                }
                return this.generateCollectionCode(valueCodeGenerator, collection);
            }
            return null;
        }

        protected CodeBlock generateCollectionCode(ValueCodeGenerator valueCodeGenerator, T collection) {
            return this.generateCollectionOf(valueCodeGenerator, (Collection<?>)collection, this.collectionType);
        }

        protected final CodeBlock generateCollectionOf(ValueCodeGenerator valueCodeGenerator, Collection<?> collection, Class<?> collectionType) {
            CodeBlock.Builder code = CodeBlock.builder();
            code.add("$T.of(", collectionType);
            Iterator<?> iterator2 = collection.iterator();
            while (iterator2.hasNext()) {
                Object element = iterator2.next();
                code.add("$L", valueCodeGenerator.generateCode(element));
                if (!iterator2.hasNext()) continue;
                code.add(", ", new Object[0]);
            }
            code.add(")", new Object[0]);
            return code.build();
        }
    }
}

