/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.bips.bip32;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.DeterministicKey;
import org.bitcoinj.crypto.HDKeyDerivation;
import org.bitcoinj.crypto.MnemonicCode;
import org.bitcoinj.crypto.MnemonicException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.bips.util.CryptoUtils;
import xyz.tcheeric.bips.util.HexUtils;

public final class Bip32 {
    private static final Logger LOGGER = LoggerFactory.getLogger(Bip32.class);

    private Bip32() {
        throw new UnsupportedOperationException("Utility class cannot be instantiated");
    }

    @Nonnull
    public static byte[] mnemonicToSeed(@Nonnull List<String> mnemonic, @Nullable String passphrase) {
        try {
            MnemonicCode.INSTANCE.check(mnemonic);
            byte[] seed = MnemonicCode.toSeed(mnemonic, passphrase != null ? passphrase : "");
            LOGGER.debug("Bip32 mnemonic_to_seed_succeeded word_count={} impact=seed_generated", (Object)mnemonic.size());
            return seed;
        }
        catch (MnemonicException.MnemonicLengthException e) {
            LOGGER.warn("Bip32 mnemonic_to_seed_failed reason=invalid_length impact=seed_not_generated");
            throw new IllegalArgumentException("Invalid mnemonic length: " + e.getMessage(), e);
        }
        catch (MnemonicException.MnemonicWordException e) {
            LOGGER.warn("Bip32 mnemonic_to_seed_failed reason=invalid_words impact=seed_not_generated");
            throw new IllegalArgumentException("Invalid mnemonic words: " + e.getMessage(), e);
        }
        catch (MnemonicException.MnemonicChecksumException e) {
            LOGGER.warn("Bip32 mnemonic_to_seed_failed reason=checksum_failure impact=seed_not_generated");
            throw new IllegalArgumentException("Invalid mnemonic checksum: " + e.getMessage(), e);
        }
        catch (MnemonicException e) {
            LOGGER.warn("Bip32 mnemonic_to_seed_failed reason=mnemonic_error impact=seed_not_generated");
            throw new IllegalArgumentException("Invalid mnemonic: " + e.getMessage(), e);
        }
    }

    @Nonnull
    public static byte[] mnemonicToSeed(@Nonnull String mnemonicPhrase, @Nullable String passphrase) {
        List<String> words = Arrays.asList(mnemonicPhrase.trim().split("\\s+"));
        return Bip32.mnemonicToSeed(words, passphrase);
    }

    @Nonnull
    public static DeterministicKey deriveMasterKey(@Nonnull byte[] seed) {
        if (seed == null || seed.length < 16) {
            LOGGER.warn("Bip32 derive_master_key_failed reason=seed_too_short seed_length={} impact=derivation_aborted", (Object)(seed == null ? 0 : seed.length));
            throw new IllegalArgumentException("Seed must be at least 16 bytes");
        }
        DeterministicKey masterKey = HDKeyDerivation.createMasterPrivateKey(seed);
        LOGGER.debug("Bip32 derive_master_key_succeeded seed_length={} impact=master_key_ready", (Object)seed.length);
        return masterKey;
    }

    @Nonnull
    public static DeterministicKey deriveKey(@Nonnull DeterministicKey masterKey, @Nonnull String path) {
        if (path == null || path.isEmpty()) {
            LOGGER.warn("Bip32 derive_key_failed reason=blank_path impact=derivation_aborted");
            throw new IllegalArgumentException("Path cannot be null or empty");
        }
        List<ChildNumber> childNumbers = Bip32.parsePath(path);
        LOGGER.debug("Bip32 derive_key_started path={} depth={} impact=derivation_in_progress", (Object)path, (Object)childNumbers.size());
        DeterministicKey currentKey = masterKey;
        for (ChildNumber childNumber : childNumbers) {
            currentKey = HDKeyDerivation.deriveChildKey(currentKey, childNumber);
        }
        LOGGER.debug("Bip32 derive_key_succeeded path={} depth={} impact=child_key_ready", (Object)path, (Object)childNumbers.size());
        return currentKey;
    }

    public static DeterministicKey deriveKey(DeterministicKey masterKey, ChildNumber ... childNumbers) {
        DeterministicKey currentKey = masterKey;
        LOGGER.debug("Bip32 derive_key_started child_count={} impact=derivation_in_progress", (Object)childNumbers.length);
        for (ChildNumber childNumber : childNumbers) {
            currentKey = HDKeyDerivation.deriveChildKey(currentKey, childNumber);
        }
        LOGGER.debug("Bip32 derive_key_succeeded child_count={} impact=child_key_ready", (Object)childNumbers.length);
        return currentKey;
    }

    public static DeterministicKey deriveKeyFromMnemonic(String mnemonicPhrase, String passphrase, String path) {
        byte[] seed = Bip32.mnemonicToSeed(mnemonicPhrase, passphrase);
        DeterministicKey masterKey = Bip32.deriveMasterKey(seed);
        return Bip32.deriveKey(masterKey, path);
    }

    public static byte[] getPrivateKeyBytes(DeterministicKey key) {
        if (!key.hasPrivKey()) {
            LOGGER.warn("Bip32 get_private_key_bytes_failed reason=missing_private_key impact=request_denied");
            throw new IllegalArgumentException("Key does not have private key component");
        }
        return key.getPrivKeyBytes();
    }

    public static byte[] getPublicKeyBytes(DeterministicKey key, boolean compressed) {
        return key.getPubKey();
    }

    public static byte[] getChainCode(DeterministicKey key) {
        return key.getChainCode();
    }

    public static List<ChildNumber> parsePath(String path) {
        if (path == null || path.isEmpty()) {
            LOGGER.warn("Bip32 parse_path_failed reason=blank_path impact=derivation_aborted");
            throw new IllegalArgumentException("Path cannot be null or empty");
        }
        String normalizedPath = path.trim();
        if (normalizedPath.toLowerCase().startsWith("m/")) {
            normalizedPath = normalizedPath.substring(2);
        }
        String[] components = normalizedPath.split("/");
        return Arrays.stream(components).filter(s -> !s.isEmpty()).map(Bip32::parseChildNumber).collect(Collectors.toList());
    }

    private static ChildNumber parseChildNumber(String component) {
        if (component == null || component.isEmpty()) {
            LOGGER.warn("Bip32 parse_child_number_failed reason=blank_component impact=derivation_aborted");
            throw new IllegalArgumentException("Child number component cannot be empty");
        }
        boolean hardened = component.endsWith("'") || component.endsWith("h") || component.endsWith("H");
        String numberStr = component;
        if (hardened) {
            numberStr = component.substring(0, component.length() - 1);
        }
        try {
            int index = Integer.parseUnsignedInt(numberStr);
            return new ChildNumber(index, hardened);
        }
        catch (NumberFormatException e) {
            LOGGER.warn("Bip32 parse_child_number_failed reason=invalid_number component={} impact=derivation_aborted", (Object)component);
            throw new IllegalArgumentException("Invalid child number: " + component, e);
        }
    }

    @Deprecated
    public static String bytesToHex(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        return HexUtils.toHex(bytes);
    }

    @Deprecated
    public static byte[] hexToBytes(String hex) {
        return HexUtils.fromHex(hex);
    }

    @Deprecated
    public static byte[] sha256(byte[] data) {
        return CryptoUtils.sha256(data);
    }
}

