/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.bips.bip39;

import java.util.ArrayList;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.bips.bip39.WordList;
import xyz.tcheeric.bips.util.Bip39Constants;
import xyz.tcheeric.bips.util.CryptoUtils;
import xyz.tcheeric.bips.util.MnemonicUtils;

public class MnemonicValidator {
    private static final Logger LOGGER = LoggerFactory.getLogger(MnemonicValidator.class);
    private final WordList wordList;

    public MnemonicValidator(WordList wordList) {
        this.wordList = wordList;
    }

    public MnemonicValidator() {
        this(WordList.english());
    }

    public ValidationResult validate(String mnemonic) {
        if (mnemonic == null || mnemonic.trim().isEmpty()) {
            LOGGER.warn("MnemonicValidator validation_failed reason=blank_input impact=mnemonic_rejected");
            return ValidationResult.failure("Mnemonic cannot be null or empty");
        }
        String[] words = MnemonicUtils.splitMnemonic(mnemonic);
        LOGGER.debug("MnemonicValidator validation_started word_count={} impact=validation_in_progress", (Object)words.length);
        if (!Bip39Constants.isValidWordCount(words.length)) {
            LOGGER.warn("MnemonicValidator validation_failed reason=invalid_word_count word_count={} impact=mnemonic_rejected", (Object)words.length);
            return ValidationResult.failure("Invalid word count: " + words.length + " (must be 12, 15, 18, 21, or 24)");
        }
        ArrayList<String> invalidWords = new ArrayList<String>();
        int[] wordIndexes = new int[words.length];
        for (int i2 = 0; i2 < words.length; ++i2) {
            int index = this.wordList.getIndex(words[i2]);
            if (index == -1) {
                invalidWords.add(words[i2]);
            }
            wordIndexes[i2] = index;
        }
        if (!invalidWords.isEmpty()) {
            LOGGER.warn("MnemonicValidator validation_failed reason=unknown_words invalid_word_count={} impact=mnemonic_rejected", (Object)invalidWords.size());
            return ValidationResult.failure("Invalid words not in " + String.valueOf((Object)this.wordList.getLanguage()) + " wordlist: " + String.valueOf(invalidWords));
        }
        if (!this.validateChecksum(wordIndexes)) {
            LOGGER.warn("MnemonicValidator validation_failed reason=checksum_mismatch word_count={} impact=mnemonic_rejected", (Object)words.length);
            return ValidationResult.failure("Invalid checksum");
        }
        LOGGER.debug("MnemonicValidator validation_succeeded word_count={} impact=mnemonic_accepted", (Object)words.length);
        return ValidationResult.success();
    }

    public void validateOrThrow(String mnemonic) throws InvalidMnemonicException {
        ValidationResult result = this.validate(mnemonic);
        if (!result.isValid()) {
            throw new InvalidMnemonicException(result.getErrorMessage());
        }
    }

    public boolean isValid(String mnemonic) {
        return this.validate(mnemonic).isValid();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean validateChecksum(int[] wordIndexes) {
        int wordCount = wordIndexes.length;
        int entropyBits = Bip39Constants.wordCountToEntropyBits(wordCount);
        int checksumBits = entropyBits / 32;
        int totalBits = entropyBits + checksumBits;
        byte[] entropyWithChecksum = this.wordIndexesToBytes(wordIndexes, totalBits);
        byte[] entropy = this.extractEntropy(entropyWithChecksum, entropyBits);
        try {
            byte expectedChecksum = CryptoUtils.sha256FirstByte(entropy);
            byte actualChecksum = this.extractChecksum(entropyWithChecksum, entropyBits, checksumBits);
            boolean bl = this.checksumsMatch(expectedChecksum, actualChecksum, checksumBits);
            return bl;
        }
        finally {
            Arrays.fill(entropy, (byte)0);
            Arrays.fill(entropyWithChecksum, (byte)0);
        }
    }

    private byte[] wordIndexesToBytes(int[] wordIndexes, int totalBits) {
        byte[] entropyWithChecksum = new byte[(totalBits + 7) / 8];
        for (int i2 = 0; i2 < wordIndexes.length; ++i2) {
            int bitIndex = i2 * 11;
            this.write11Bits(entropyWithChecksum, bitIndex, wordIndexes[i2]);
        }
        return entropyWithChecksum;
    }

    private byte[] extractEntropy(byte[] entropyWithChecksum, int entropyBits) {
        int entropyBytes = entropyBits / 8;
        return Arrays.copyOf(entropyWithChecksum, entropyBytes);
    }

    private byte extractChecksum(byte[] entropyWithChecksum, int entropyBits, int checksumBits) {
        int entropyBytes;
        if (checksumBits == 0) {
            return 0;
        }
        int checksumByteIndex = entropyBytes = entropyBits / 8;
        int checksumBitOffset = entropyBits % 8;
        if (checksumBitOffset == 0) {
            return entropyWithChecksum[checksumByteIndex];
        }
        int shift = 8 - checksumBitOffset;
        return (byte)(entropyWithChecksum[checksumByteIndex] & 255 << shift);
    }

    private boolean checksumsMatch(byte expectedChecksum, byte actualChecksum, int checksumBits) {
        if (checksumBits == 0) {
            return true;
        }
        int mask = 255 << 8 - checksumBits & 0xFF;
        return (expectedChecksum & mask) == (actualChecksum & mask);
    }

    private void write11Bits(byte[] data, int bitIndex, int value) {
        int byteIndex = bitIndex / 8;
        int bitOffset = bitIndex % 8;
        int shifted = value << 13 - bitOffset;
        if (byteIndex < data.length) {
            int n = byteIndex;
            data[n] = (byte)(data[n] | (byte)(shifted >> 16 & 0xFF));
        }
        if (byteIndex + 1 < data.length) {
            int n = byteIndex + 1;
            data[n] = (byte)(data[n] | (byte)(shifted >> 8 & 0xFF));
        }
        if (byteIndex + 2 < data.length) {
            int n = byteIndex + 2;
            data[n] = (byte)(data[n] | (byte)(shifted & 0xFF));
        }
    }

    public WordList getWordList() {
        return this.wordList;
    }

    public static class ValidationResult {
        private final boolean valid;
        private final String errorMessage;

        private ValidationResult(boolean valid, String errorMessage) {
            this.valid = valid;
            this.errorMessage = errorMessage;
        }

        public boolean isValid() {
            return this.valid;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public static ValidationResult success() {
            return new ValidationResult(true, null);
        }

        public static ValidationResult failure(String errorMessage) {
            return new ValidationResult(false, errorMessage);
        }
    }

    public static class InvalidMnemonicException
    extends Exception {
        public InvalidMnemonicException(String message) {
            super(message);
        }
    }
}

