/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.cashu.voucher.app;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.cashu.voucher.app.dto.RedeemVoucherRequest;
import xyz.tcheeric.cashu.voucher.app.dto.RedeemVoucherResponse;
import xyz.tcheeric.cashu.voucher.app.ports.VoucherLedgerPort;
import xyz.tcheeric.cashu.voucher.domain.SignedVoucher;
import xyz.tcheeric.cashu.voucher.domain.VoucherStatus;
import xyz.tcheeric.cashu.voucher.domain.VoucherValidator;

public class MerchantVerificationService {
    private static final Logger log = LoggerFactory.getLogger(MerchantVerificationService.class);
    private final VoucherLedgerPort ledgerPort;

    public MerchantVerificationService(@NonNull VoucherLedgerPort ledgerPort) {
        if (ledgerPort == null) {
            throw new NullPointerException("ledgerPort is marked non-null but is null");
        }
        this.ledgerPort = ledgerPort;
        log.info("MerchantVerificationService initialized");
    }

    public VerificationResult verifyOffline(@NonNull SignedVoucher voucher, @NonNull String expectedIssuerId) {
        VoucherValidator.ValidationResult domainResult;
        if (voucher == null) {
            throw new NullPointerException("voucher is marked non-null but is null");
        }
        if (expectedIssuerId == null) {
            throw new NullPointerException("expectedIssuerId is marked non-null but is null");
        }
        if (expectedIssuerId.isBlank()) {
            throw new IllegalArgumentException("Expected issuer ID cannot be blank");
        }
        log.debug("Performing offline verification: voucherId={}, expectedIssuer={}", (Object)voucher.getSecret().getVoucherId(), (Object)expectedIssuerId);
        ArrayList<String> errors = new ArrayList<String>();
        String actualIssuerId = voucher.getSecret().getIssuerId();
        if (!actualIssuerId.equals(expectedIssuerId)) {
            String error = String.format("Voucher issued by '%s' but expected issuer is '%s' (Model B: vouchers only redeemable at issuing merchant)", actualIssuerId, expectedIssuerId);
            errors.add(error);
            log.warn("Issuer mismatch: {}", (Object)error);
        }
        if (!(domainResult = VoucherValidator.validate(voucher)).isValid()) {
            errors.addAll(domainResult.getErrors());
            log.warn("Domain validation failed: {}", (Object)domainResult.getErrorMessage());
        }
        if (errors.isEmpty()) {
            log.debug("Offline verification passed: voucherId={}", (Object)voucher.getSecret().getVoucherId());
            return VerificationResult.success();
        }
        log.debug("Offline verification failed: voucherId={}, errors={}", (Object)voucher.getSecret().getVoucherId(), (Object)errors.size());
        return VerificationResult.failure(errors);
    }

    public VerificationResult verifyOnline(@NonNull SignedVoucher voucher, @NonNull String expectedIssuerId) {
        if (voucher == null) {
            throw new NullPointerException("voucher is marked non-null but is null");
        }
        if (expectedIssuerId == null) {
            throw new NullPointerException("expectedIssuerId is marked non-null but is null");
        }
        if (expectedIssuerId.isBlank()) {
            throw new IllegalArgumentException("Expected issuer ID cannot be blank");
        }
        String voucherId = voucher.getSecret().getVoucherId();
        log.info("Performing online verification: voucherId={}, expectedIssuer={}", (Object)voucherId, (Object)expectedIssuerId);
        VerificationResult offlineResult = this.verifyOffline(voucher, expectedIssuerId);
        if (!offlineResult.isValid()) {
            log.warn("Online verification failed at offline stage: voucherId={}", (Object)voucherId);
            return offlineResult;
        }
        try {
            Optional<VoucherStatus> statusOpt = this.ledgerPort.queryStatus(voucherId);
            if (statusOpt.isEmpty()) {
                String error = "Voucher not found in public ledger";
                log.warn("Online verification failed: voucherId={}, {}", (Object)voucherId, (Object)error);
                return VerificationResult.failure(error);
            }
            VoucherStatus status = statusOpt.get();
            log.debug("Voucher ledger status: voucherId={}, status={}", (Object)voucherId, (Object)status);
            if (status == VoucherStatus.REDEEMED) {
                String error = "Voucher already redeemed (double-spend attempt detected)";
                log.warn("Double-spend detected: voucherId={}", (Object)voucherId);
                return VerificationResult.failure(error);
            }
            if (status == VoucherStatus.REVOKED) {
                String error = "Voucher has been revoked by issuer";
                log.warn("Revoked voucher: voucherId={}", (Object)voucherId);
                return VerificationResult.failure(error);
            }
            if (status == VoucherStatus.EXPIRED) {
                String error = "Voucher has expired";
                log.warn("Expired voucher (ledger status): voucherId={}", (Object)voucherId);
                return VerificationResult.failure(error);
            }
            if (status == VoucherStatus.ISSUED) {
                log.info("Online verification passed: voucherId={}, status=ISSUED", (Object)voucherId);
                return VerificationResult.success();
            }
            String error = "Unknown voucher status: " + String.valueOf((Object)status);
            log.error("Unknown status in ledger: voucherId={}, status={}", (Object)voucherId, (Object)status);
            return VerificationResult.failure(error);
        }
        catch (Exception e) {
            String error = "Failed to query voucher status from ledger: " + e.getMessage();
            log.error("Ledger query failed: voucherId={}", (Object)voucherId, (Object)e);
            return VerificationResult.failure(error);
        }
    }

    public void markRedeemed(@NonNull String voucherId) {
        if (voucherId == null) {
            throw new NullPointerException("voucherId is marked non-null but is null");
        }
        if (voucherId.isBlank()) {
            throw new IllegalArgumentException("Voucher ID cannot be blank");
        }
        log.info("Marking voucher as redeemed: voucherId={}", (Object)voucherId);
        try {
            this.ledgerPort.updateStatus(voucherId, VoucherStatus.REDEEMED);
            log.info("Voucher marked as redeemed successfully: voucherId={}", (Object)voucherId);
        }
        catch (Exception e) {
            log.error("Failed to mark voucher as redeemed: voucherId={}", (Object)voucherId, (Object)e);
            throw new RuntimeException("Failed to mark voucher as redeemed", e);
        }
    }

    public RedeemVoucherResponse redeem(@NonNull RedeemVoucherRequest request, @NonNull SignedVoucher voucher) {
        VerificationResult verifyResult;
        if (request == null) {
            throw new NullPointerException("request is marked non-null but is null");
        }
        if (voucher == null) {
            throw new NullPointerException("voucher is marked non-null but is null");
        }
        String voucherId = voucher.getSecret().getVoucherId();
        log.info("Processing redemption request: voucherId={}, merchantId={}", (Object)voucherId, (Object)request.getMerchantId());
        if (Boolean.TRUE.equals(request.getVerifyOnline())) {
            verifyResult = this.verifyOnline(voucher, request.getMerchantId());
        } else {
            log.warn("Offline verification requested: voucherId={} - double-spend not prevented!", (Object)voucherId);
            verifyResult = this.verifyOffline(voucher, request.getMerchantId());
        }
        if (!verifyResult.isValid()) {
            log.warn("Redemption rejected: voucherId={}, errors={}", (Object)voucherId, (Object)verifyResult.getErrorMessage());
            return RedeemVoucherResponse.failure(verifyResult.getErrorMessage());
        }
        try {
            this.markRedeemed(voucherId);
            log.info("Redemption successful: voucherId={}, amount={}", (Object)voucherId, (Object)voucher.getSecret().getFaceValue());
            return RedeemVoucherResponse.success(voucher);
        }
        catch (Exception e) {
            String error = "Verification passed but failed to mark as redeemed: " + e.getMessage();
            log.error("Redemption failed at marking stage: voucherId={}", (Object)voucherId, (Object)e);
            return RedeemVoucherResponse.failure(error);
        }
    }

    public static class VerificationResult {
        private final boolean valid;
        private final List<String> errors;

        public static VerificationResult success() {
            return new VerificationResult(true, Collections.emptyList());
        }

        public static VerificationResult failure(@NonNull String error) {
            if (error == null) {
                throw new NullPointerException("error is marked non-null but is null");
            }
            ArrayList<String> errors = new ArrayList<String>();
            errors.add(error);
            return new VerificationResult(false, errors);
        }

        public static VerificationResult failure(@NonNull List<String> errors) {
            if (errors == null) {
                throw new NullPointerException("errors is marked non-null but is null");
            }
            return new VerificationResult(false, new ArrayList<String>(errors));
        }

        public List<String> getErrors() {
            return Collections.unmodifiableList(this.errors);
        }

        public String getErrorMessage() {
            return String.join((CharSequence)"; ", this.errors);
        }

        public boolean isValid() {
            return this.valid;
        }

        public VerificationResult(boolean valid, List<String> errors) {
            this.valid = valid;
            this.errors = errors;
        }
    }
}

