/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.cashu.voucher.app;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.cashu.voucher.app.dto.IssueVoucherRequest;
import xyz.tcheeric.cashu.voucher.app.dto.IssueVoucherResponse;
import xyz.tcheeric.cashu.voucher.app.ports.VoucherBackupPort;
import xyz.tcheeric.cashu.voucher.app.ports.VoucherLedgerPort;
import xyz.tcheeric.cashu.voucher.domain.SignedVoucher;
import xyz.tcheeric.cashu.voucher.domain.VoucherSecret;
import xyz.tcheeric.cashu.voucher.domain.VoucherSignatureService;
import xyz.tcheeric.cashu.voucher.domain.VoucherStatus;

public class VoucherService {
    private static final Logger log = LoggerFactory.getLogger(VoucherService.class);
    private final VoucherLedgerPort ledgerPort;
    private final VoucherBackupPort backupPort;
    private final String mintIssuerPrivateKey;
    private final String mintIssuerPublicKey;

    public VoucherService(@NonNull VoucherLedgerPort ledgerPort, @NonNull VoucherBackupPort backupPort, @NonNull String mintIssuerPrivateKey, @NonNull String mintIssuerPublicKey) {
        if (ledgerPort == null) {
            throw new NullPointerException("ledgerPort is marked non-null but is null");
        }
        if (backupPort == null) {
            throw new NullPointerException("backupPort is marked non-null but is null");
        }
        if (mintIssuerPrivateKey == null) {
            throw new NullPointerException("mintIssuerPrivateKey is marked non-null but is null");
        }
        if (mintIssuerPublicKey == null) {
            throw new NullPointerException("mintIssuerPublicKey is marked non-null but is null");
        }
        if (mintIssuerPrivateKey.isBlank()) {
            throw new IllegalArgumentException("Mint issuer private key cannot be blank");
        }
        if (mintIssuerPublicKey.isBlank()) {
            throw new IllegalArgumentException("Mint issuer public key cannot be blank");
        }
        this.ledgerPort = ledgerPort;
        this.backupPort = backupPort;
        this.mintIssuerPrivateKey = mintIssuerPrivateKey;
        this.mintIssuerPublicKey = mintIssuerPublicKey;
        log.info("VoucherService initialized with issuer public key: {}...", (Object)mintIssuerPublicKey.substring(0, Math.min(8, mintIssuerPublicKey.length())));
    }

    public IssueVoucherResponse issue(@NonNull IssueVoucherRequest request) {
        VoucherSecret secret;
        if (request == null) {
            throw new NullPointerException("request is marked non-null but is null");
        }
        log.info("Issuing voucher: issuerId={}, unit={}, amount={}", request.getIssuerId(), request.getUnit(), request.getAmount());
        this.validateIssueRequest(request);
        Long expiresAt = null;
        if (request.getExpiresInDays() != null) {
            expiresAt = Instant.now().plus((long)request.getExpiresInDays().intValue(), ChronoUnit.DAYS).getEpochSecond();
            log.debug("Voucher will expire at: {} (in {} days)", (Object)Instant.ofEpochSecond(expiresAt), (Object)request.getExpiresInDays());
        }
        if (request.getVoucherId() != null && !request.getVoucherId().isBlank()) {
            secret = VoucherSecret.create(request.getVoucherId(), request.getIssuerId(), request.getUnit(), request.getAmount(), expiresAt, request.getMemo());
            log.debug("Created voucher with custom ID: {}", (Object)request.getVoucherId());
        } else {
            secret = VoucherSecret.create(request.getIssuerId(), request.getUnit(), request.getAmount(), expiresAt, request.getMemo());
            log.debug("Created voucher with auto-generated ID: {}", (Object)secret.getVoucherId());
        }
        SignedVoucher signedVoucher = VoucherSignatureService.createSigned(secret, this.mintIssuerPrivateKey, this.mintIssuerPublicKey);
        log.debug("Voucher signed successfully");
        try {
            this.ledgerPort.publish(signedVoucher, VoucherStatus.ISSUED);
            log.info("Voucher published to ledger: voucherId={}, status=ISSUED", (Object)secret.getVoucherId());
        }
        catch (Exception e) {
            log.error("Failed to publish voucher to ledger: voucherId={}", (Object)secret.getVoucherId(), (Object)e);
            throw new RuntimeException("Failed to publish voucher to ledger", e);
        }
        String token = this.serializeToToken(signedVoucher);
        return IssueVoucherResponse.builder().voucher(signedVoucher).token(token).build();
    }

    public Optional<VoucherStatus> queryStatus(@NonNull String voucherId) {
        if (voucherId == null) {
            throw new NullPointerException("voucherId is marked non-null but is null");
        }
        if (voucherId.isBlank()) {
            throw new IllegalArgumentException("Voucher ID cannot be blank");
        }
        log.debug("Querying voucher status: voucherId={}", (Object)voucherId);
        try {
            Optional<VoucherStatus> status = this.ledgerPort.queryStatus(voucherId);
            if (status.isPresent()) {
                log.debug("Voucher status found: voucherId={}, status={}", (Object)voucherId, (Object)status.get());
            } else {
                log.debug("Voucher not found in ledger: voucherId={}", (Object)voucherId);
            }
            return status;
        }
        catch (Exception e) {
            log.error("Failed to query voucher status: voucherId={}", (Object)voucherId, (Object)e);
            throw new RuntimeException("Failed to query voucher status", e);
        }
    }

    public void updateStatus(@NonNull String voucherId, @NonNull VoucherStatus newStatus) {
        if (voucherId == null) {
            throw new NullPointerException("voucherId is marked non-null but is null");
        }
        if (newStatus == null) {
            throw new NullPointerException("newStatus is marked non-null but is null");
        }
        if (voucherId.isBlank()) {
            throw new IllegalArgumentException("Voucher ID cannot be blank");
        }
        log.info("Updating voucher status: voucherId={}, newStatus={}", (Object)voucherId, (Object)newStatus);
        try {
            this.ledgerPort.updateStatus(voucherId, newStatus);
            log.info("Voucher status updated successfully: voucherId={}, status={}", (Object)voucherId, (Object)newStatus);
        }
        catch (Exception e) {
            log.error("Failed to update voucher status: voucherId={}, status={}", new Object[]{voucherId, newStatus, e});
            throw new RuntimeException("Failed to update voucher status", e);
        }
    }

    public void backup(@NonNull List<SignedVoucher> vouchers, @NonNull String userPrivateKey) {
        if (vouchers == null) {
            throw new NullPointerException("vouchers is marked non-null but is null");
        }
        if (userPrivateKey == null) {
            throw new NullPointerException("userPrivateKey is marked non-null but is null");
        }
        if (userPrivateKey.isBlank()) {
            throw new IllegalArgumentException("User private key cannot be blank");
        }
        log.info("Backing up {} voucher(s)", (Object)vouchers.size());
        if (vouchers.isEmpty()) {
            log.debug("No vouchers to backup, skipping");
            return;
        }
        try {
            this.backupPort.backup(vouchers, userPrivateKey);
            log.info("Successfully backed up {} voucher(s)", (Object)vouchers.size());
        }
        catch (Exception e) {
            log.error("Failed to backup {} voucher(s)", (Object)vouchers.size(), (Object)e);
            throw new RuntimeException("Failed to backup vouchers", e);
        }
    }

    public List<SignedVoucher> restore(@NonNull String userPrivateKey) {
        if (userPrivateKey == null) {
            throw new NullPointerException("userPrivateKey is marked non-null but is null");
        }
        if (userPrivateKey.isBlank()) {
            throw new IllegalArgumentException("User private key cannot be blank");
        }
        log.info("Restoring vouchers from backup");
        try {
            List<SignedVoucher> restored = this.backupPort.restore(userPrivateKey);
            log.info("Successfully restored {} voucher(s)", (Object)restored.size());
            return restored;
        }
        catch (Exception e) {
            log.error("Failed to restore vouchers", e);
            throw new RuntimeException("Failed to restore vouchers", e);
        }
    }

    public boolean exists(@NonNull String voucherId) {
        if (voucherId == null) {
            throw new NullPointerException("voucherId is marked non-null but is null");
        }
        if (voucherId.isBlank()) {
            throw new IllegalArgumentException("Voucher ID cannot be blank");
        }
        return this.ledgerPort.exists(voucherId);
    }

    private void validateIssueRequest(IssueVoucherRequest request) {
        if (request.getIssuerId() == null || request.getIssuerId().isBlank()) {
            throw new IllegalArgumentException("Issuer ID is required");
        }
        if (request.getUnit() == null || request.getUnit().isBlank()) {
            throw new IllegalArgumentException("Unit is required");
        }
        if (request.getAmount() == null || request.getAmount() <= 0L) {
            throw new IllegalArgumentException("Amount must be positive");
        }
        if (request.getExpiresInDays() != null && request.getExpiresInDays() <= 0) {
            throw new IllegalArgumentException("Expiry days must be positive if specified");
        }
    }

    private String serializeToToken(SignedVoucher voucher) {
        log.warn("Using placeholder token serialization - full implementation pending");
        return "cashuA" + voucher.getSecret().getVoucherId().replace("-", "");
    }
}

