/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.cashu.voucher.nostr;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import lombok.NonNull;
import nostr.event.impl.GenericEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.cashu.voucher.nostr.VoucherNostrException;

public class NostrClientAdapter {
    private static final Logger log = LoggerFactory.getLogger(NostrClientAdapter.class);
    private final List<String> relayUrls;
    private final long connectionTimeoutMs;
    private final int maxRetries;
    private final Map<String, RelayConnection> connections = new ConcurrentHashMap<String, RelayConnection>();
    private volatile boolean connected = false;

    public NostrClientAdapter(@NonNull List<String> relayUrls, long connectionTimeoutMs, int maxRetries) {
        if (relayUrls == null) {
            throw new NullPointerException("relayUrls is marked non-null but is null");
        }
        if (relayUrls.isEmpty()) {
            throw new IllegalArgumentException("At least one relay URL is required");
        }
        if (connectionTimeoutMs <= 0L) {
            throw new IllegalArgumentException("Connection timeout must be positive");
        }
        if (maxRetries < 0) {
            throw new IllegalArgumentException("Max retries cannot be negative");
        }
        this.relayUrls = new ArrayList<String>(relayUrls);
        this.connectionTimeoutMs = connectionTimeoutMs;
        this.maxRetries = maxRetries;
        log.info("NostrClientAdapter initialized: relays={}, timeout={}ms, maxRetries={}", relayUrls.size(), connectionTimeoutMs, maxRetries);
    }

    public void connect() {
        if (this.connected) {
            log.debug("Already connected to relays");
            return;
        }
        log.info("Connecting to {} relay(s)...", (Object)this.relayUrls.size());
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        for (String url : this.relayUrls) {
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                try {
                    this.connectToRelay(url);
                }
                catch (Exception e) {
                    log.error("Failed to connect to relay: {}", (Object)url, (Object)e);
                }
            });
            futures.add(future);
        }
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).orTimeout(this.connectionTimeoutMs, TimeUnit.MILLISECONDS).join();
        if (this.connections.isEmpty()) {
            throw new VoucherNostrException("Failed to connect to any relay");
        }
        this.connected = true;
        log.info("Connected to {}/{} relay(s)", (Object)this.connections.size(), (Object)this.relayUrls.size());
    }

    public void disconnect() {
        if (!this.connected) {
            log.debug("Already disconnected");
            return;
        }
        log.info("Disconnecting from {} relay(s)...", (Object)this.connections.size());
        for (Map.Entry<String, RelayConnection> entry : this.connections.entrySet()) {
            try {
                entry.getValue().close();
                log.debug("Disconnected from relay: {}", (Object)entry.getKey());
            }
            catch (Exception e) {
                log.error("Error disconnecting from relay: {}", (Object)entry.getKey(), (Object)e);
            }
        }
        this.connections.clear();
        this.connected = false;
        log.info("Disconnected from all relays");
    }

    public boolean publishEvent(@NonNull GenericEvent event, long timeoutMs) {
        if (event == null) {
            throw new NullPointerException("event is marked non-null but is null");
        }
        this.ensureConnected();
        log.debug("Publishing event: id={}, kind={}", (Object)event.getId(), (Object)event.getKind());
        ArrayList<CompletableFuture<Boolean>> futures = new ArrayList<CompletableFuture<Boolean>>();
        for (Map.Entry<String, RelayConnection> entry : this.connections.entrySet()) {
            String relayUrl = entry.getKey();
            RelayConnection conn = entry.getValue();
            CompletableFuture<Boolean> future = CompletableFuture.supplyAsync(() -> {
                try {
                    return conn.publish(event);
                }
                catch (Exception e) {
                    log.error("Failed to publish to relay: {}", (Object)relayUrl, (Object)e);
                    return false;
                }
            });
            futures.add(future);
        }
        try {
            CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).orTimeout(timeoutMs, TimeUnit.MILLISECONDS).join();
        }
        catch (Exception e) {
            log.warn("Some publish operations timed out or failed", e);
        }
        long successCount = futures.stream().map(f -> f.getNow(false)).filter(Boolean::booleanValue).count();
        boolean success = successCount > 0L;
        log.debug("Event published: id={}, success={}/{} relays", event.getId(), successCount, this.connections.size());
        return success;
    }

    public List<GenericEvent> queryEvents(@NonNull String subscriptionId, long timeoutMs) {
        if (subscriptionId == null) {
            throw new NullPointerException("subscriptionId is marked non-null but is null");
        }
        this.ensureConnected();
        log.debug("Querying events: subscriptionId={}", (Object)subscriptionId);
        ConcurrentHashMap eventMap = new ConcurrentHashMap();
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        for (Map.Entry<String, RelayConnection> entry : this.connections.entrySet()) {
            String relayUrl = entry.getKey();
            RelayConnection conn = entry.getValue();
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                try {
                    List<GenericEvent> events = conn.query(subscriptionId);
                    for (GenericEvent event : events) {
                        eventMap.putIfAbsent(event.getId(), event);
                    }
                    log.debug("Received {} event(s) from relay: {}", (Object)events.size(), (Object)relayUrl);
                }
                catch (Exception e) {
                    log.error("Failed to query relay: {}", (Object)relayUrl, (Object)e);
                }
            });
            futures.add(future);
        }
        try {
            CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).orTimeout(timeoutMs, TimeUnit.MILLISECONDS).join();
        }
        catch (Exception e) {
            log.warn("Some query operations timed out or failed", e);
        }
        ArrayList<GenericEvent> results = new ArrayList<GenericEvent>(eventMap.values());
        log.debug("Query complete: {} unique event(s) found", (Object)results.size());
        return results;
    }

    public boolean isConnected() {
        return this.connected && !this.connections.isEmpty();
    }

    public int getConnectedRelayCount() {
        return this.connections.size();
    }

    public List<String> getConnectedRelays() {
        return Collections.unmodifiableList(new ArrayList<String>(this.connections.keySet()));
    }

    private void connectToRelay(String url) {
        log.debug("Connecting to relay: {}", (Object)url);
        int attempt = 0;
        Exception lastException = null;
        while (attempt <= this.maxRetries) {
            try {
                RelayConnection conn = new RelayConnection(url);
                conn.connect();
                this.connections.put(url, conn);
                log.info("Connected to relay: {}", (Object)url);
                return;
            }
            catch (Exception e) {
                lastException = e;
                if (++attempt > this.maxRetries) continue;
                log.warn("Connection attempt {}/{} failed for relay: {}", attempt, this.maxRetries + 1, url);
                try {
                    Thread.sleep(1000 * attempt);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new VoucherNostrException("Connection interrupted", ie);
                }
            }
        }
        log.error("Failed to connect to relay after {} attempts: {}", (Object)(this.maxRetries + 1), (Object)url);
        throw new VoucherNostrException("Failed to connect to relay: " + url, lastException);
    }

    private void ensureConnected() {
        if (!this.connected || this.connections.isEmpty()) {
            throw new VoucherNostrException("Not connected to any relay. Call connect() first.");
        }
    }

    private String generateSubscriptionId() {
        return "sub_" + UUID.randomUUID().toString().replace("-", "").substring(0, 16);
    }

    private static class RelayConnection {
        private final String url;
        private volatile boolean connected = false;

        RelayConnection(String url) {
            this.url = url;
        }

        void connect() {
            log.debug("Simulating connection to: {}", (Object)this.url);
            this.connected = true;
        }

        boolean publish(GenericEvent event) {
            if (!this.connected) {
                throw new VoucherNostrException("Relay not connected");
            }
            log.debug("Simulating publish to: {}", (Object)this.url);
            return true;
        }

        List<GenericEvent> query(String subscriptionId) {
            if (!this.connected) {
                throw new VoucherNostrException("Relay not connected");
            }
            log.debug("Simulating query to: {} with subscription: {}", (Object)this.url, (Object)subscriptionId);
            return Collections.emptyList();
        }

        void close() {
            log.debug("Simulating close for: {}", (Object)this.url);
            this.connected = false;
        }
    }
}

