/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.identity.application.usecases.bunker;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.identity.api.exceptions.IdentityException;
import xyz.tcheeric.identity.api.exceptions.IdentityNotFoundException;
import xyz.tcheeric.identity.api.ports.BunkerIdentityRepository;
import xyz.tcheeric.identity.api.ports.SigningPort;
import xyz.tcheeric.identity.domain.BunkerIdentity;

public class EncryptMessageUseCase {
    private static final Logger LOGGER = LoggerFactory.getLogger(EncryptMessageUseCase.class);
    private static final Pattern HEX_PUBKEY_PATTERN = Pattern.compile("^[0-9a-fA-F]{64}$");
    private static final Pattern NPUB_PATTERN = Pattern.compile("^npub1[qpzry9x8gf2tvdw0s3jn54khce6mua7l]{58}$");
    private final SigningPort signingPort;
    private final BunkerIdentityRepository identityRepository;

    public EncryptMessageUseCase(SigningPort signingPort, BunkerIdentityRepository identityRepository) {
        this.signingPort = Objects.requireNonNull(signingPort, "Signing port cannot be null");
        this.identityRepository = Objects.requireNonNull(identityRepository, "Identity repository cannot be null");
    }

    public CompletableFuture<String> encryptNip44(String keyName, String recipientPubkey, String plaintext) {
        return this.encrypt(keyName, recipientPubkey, plaintext, EncryptionType.NIP44);
    }

    public CompletableFuture<String> encryptNip04(String keyName, String recipientPubkey, String plaintext) {
        return this.encrypt(keyName, recipientPubkey, plaintext, EncryptionType.NIP04);
    }

    private CompletableFuture<String> encrypt(String keyName, String recipientPubkey, String plaintext, EncryptionType type) {
        Objects.requireNonNull(keyName, "Key name cannot be null");
        Objects.requireNonNull(recipientPubkey, "Recipient public key cannot be null");
        Objects.requireNonNull(plaintext, "Plaintext cannot be null");
        String normalizedPubkey = this.normalizePublicKey(recipientPubkey);
        if (normalizedPubkey == null) {
            return CompletableFuture.failedFuture(new IdentityException("Invalid recipient public key format. Expected 64-character hex or npub1... format. Suggestion: Verify the public key and try again."));
        }
        LOGGER.debug("encrypt_message_started key_name={} type={} recipient={}", new Object[]{keyName, type, normalizedPubkey.substring(0, 16)});
        return ((CompletableFuture)this.identityRepository.findByKeyName(keyName).thenCompose(optIdentity -> {
            BunkerIdentity identity = (BunkerIdentity)optIdentity.orElseThrow(() -> new IdentityNotFoundException("Bunker identity with key name '" + keyName + "' not found. Suggestion: Check the key name or create/import the identity first."));
            if (identity.isLocked()) {
                LOGGER.warn("encrypt_message_rejected key_name={} reason=identity_locked", (Object)keyName);
                return CompletableFuture.failedFuture(new IdentityException("Identity '" + keyName + "' is locked. Suggestion: Unlock the identity with the passphrase before encrypting."));
            }
            return this.performEncryption(identity, normalizedPubkey, plaintext, type);
        })).whenComplete((ciphertext, ex) -> {
            if (ex != null) {
                LOGGER.error("encrypt_message_failed key_name={} type={} error={}", new Object[]{keyName, type, ex.getMessage()});
            } else if (ciphertext != null) {
                LOGGER.info("encrypt_message_complete key_name={} type={} ciphertext_length={}", new Object[]{keyName, type, ciphertext.length()});
            }
        });
    }

    private CompletableFuture<String> performEncryption(BunkerIdentity identity, String recipientPubkey, String plaintext, EncryptionType type) {
        LOGGER.debug("bunker_encrypt identity={} type={}", (Object)identity.getKeyName(), (Object)type);
        CompletableFuture<String> encryptFuture = switch (type.ordinal()) {
            default -> throw new MatchException(null, null);
            case 0 -> this.signingPort.nip04Encrypt(identity, recipientPubkey, plaintext);
            case 1 -> this.signingPort.nip44Encrypt(identity, recipientPubkey, plaintext);
        };
        return ((CompletableFuture)encryptFuture.thenApply(ciphertext -> {
            if (ciphertext == null || ciphertext.isEmpty()) {
                throw new IdentityException("Encryption returned empty result for identity '" + identity.getKeyName() + "'. Suggestion: The bunker may not support " + String.valueOf((Object)type) + " encryption or returned an error.");
            }
            return ciphertext;
        })).exceptionally(ex -> {
            Throwable cause;
            Throwable throwable = cause = ex instanceof CompletionException ? ex.getCause() : ex;
            if (cause instanceof IdentityException) {
                throw (IdentityException)cause;
            }
            throw new IdentityException("Failed to encrypt message with identity '" + identity.getKeyName() + "': " + cause.getMessage() + ". Suggestion: Check bunker connectivity and verify the identity is unlocked.", cause);
        });
    }

    private String normalizePublicKey(String pubkey) {
        if (HEX_PUBKEY_PATTERN.matcher(pubkey).matches()) {
            return pubkey.toLowerCase();
        }
        if (NPUB_PATTERN.matcher(pubkey).matches()) {
            return null;
        }
        return null;
    }

    private static enum EncryptionType {
        NIP04,
        NIP44;

    }

    public record EncryptionResult(boolean success, String ciphertext, String type, String error) {
        public static EncryptionResult success(String ciphertext, String type) {
            return new EncryptionResult(true, ciphertext, type, null);
        }

        public static EncryptionResult failure(String error) {
            return new EncryptionResult(false, null, null, error);
        }
    }
}

