/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.nsecbunker.client.transport;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import nostr.crypto.bech32.Bech32;
import nostr.crypto.schnorr.Schnorr;
import nostr.event.impl.GenericEvent;
import nostr.id.Identity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.nsecbunker.connection.RelayConnection;
import xyz.tcheeric.nsecbunker.connection.RelayPool;
import xyz.tcheeric.nsecbunker.connection.RelayPoolListener;
import xyz.tcheeric.nsecbunker.protocol.crypto.Nip04Crypto;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Decoder;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Encoder;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Request;
import xyz.tcheeric.nsecbunker.protocol.nip46.Nip46Response;

public class RelayNip46Transport
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(RelayNip46Transport.class);
    private static final int NIP46_KIND = 24133;
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private final RelayPool relayPool;
    private final Identity clientIdentity;
    private final String bunkerPubkeyHex;
    private final Nip04Crypto crypto;
    private final Nip46Encoder encoder;
    private final Nip46Decoder decoder;
    private final Duration timeout;
    private final Map<String, CompletableFuture<Nip46Response>> pendingRequests;
    private volatile boolean connected = false;
    private volatile String subscriptionId;

    private RelayNip46Transport(List<String> relays, String clientPrivateKey, String bunkerPubkey, Duration timeout2, Duration connectTimeout) {
        Objects.requireNonNull(relays, "relays must not be null");
        Objects.requireNonNull(clientPrivateKey, "clientPrivateKey must not be null");
        Objects.requireNonNull(bunkerPubkey, "bunkerPubkey must not be null");
        this.timeout = timeout2 != null ? timeout2 : Duration.ofSeconds(30L);
        Duration connTimeout = connectTimeout != null ? connectTimeout : Duration.ofSeconds(10L);
        this.relayPool = RelayPool.builder().relays(relays).connectTimeout(connTimeout).minConnectedRelays(1).deduplicateEvents(true).build();
        this.clientIdentity = this.createIdentity(clientPrivateKey);
        this.bunkerPubkeyHex = this.resolvePubkeyToHex(bunkerPubkey);
        String clientPrivateKeyHex = this.resolvePrivateKeyToHex(clientPrivateKey);
        this.crypto = Nip04Crypto.create(clientPrivateKeyHex, this.bunkerPubkeyHex);
        this.encoder = new Nip46Encoder();
        this.decoder = new Nip46Decoder();
        this.pendingRequests = new ConcurrentHashMap<String, CompletableFuture<Nip46Response>>();
        this.relayPool.addListener(new ResponseListener());
    }

    public void connect() {
        if (this.connected) {
            return;
        }
        try {
            this.relayPool.connectAll();
            String clientPubkeyHex = this.clientIdentity.getPublicKey().toString();
            this.subscriptionId = "nip46-" + clientPubkeyHex.substring(0, 8);
            String filter2 = String.format("{\"kinds\":[%d],\"#p\":[\"%s\"],\"authors\":[\"%s\"]}", 24133, clientPubkeyHex, this.bunkerPubkeyHex);
            this.relayPool.broadcastReq(this.subscriptionId, filter2);
            this.connected = true;
            log.info("relay_transport_connected relays={} subscription={}", (Object)this.relayPool.getConnectedCount(), (Object)this.subscriptionId);
        }
        catch (Exception e) {
            log.error("relay_transport_connection_failed error={}", (Object)e.getMessage());
            throw new RuntimeException("Failed to connect relay transport", e);
        }
    }

    public Function<Nip46Request, Nip46Response> createRequestHandler() {
        return request -> {
            try {
                return this.sendRequestAsync((Nip46Request)request).get(this.timeout.toMillis(), TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                log.error("nip46_request_timeout request_id={} method={}", (Object)request.getId(), (Object)request.getMethod());
                throw new RuntimeException("NIP-46 request timed out: " + request.getId(), e);
            }
            catch (Exception e) {
                log.error("nip46_request_failed request_id={} method={} error={}", request.getId(), request.getMethod(), e.getMessage());
                throw new RuntimeException("NIP-46 request failed: " + e.getMessage(), e);
            }
        };
    }

    public CompletableFuture<Nip46Response> sendRequestAsync(Nip46Request request) {
        Objects.requireNonNull(request, "request must not be null");
        if (!this.connected) {
            return CompletableFuture.failedFuture(new IllegalStateException("Transport not connected"));
        }
        CompletableFuture<Nip46Response> future = new CompletableFuture<Nip46Response>();
        this.pendingRequests.put(request.getId(), future);
        try {
            String requestJson = this.encoder.encodeRequest(request);
            String encryptedContent = this.crypto.encrypt(requestJson);
            String eventJson = this.createSignedEventJson(encryptedContent);
            int sentCount = this.relayPool.broadcastEvent(eventJson);
            if (sentCount == 0) {
                this.pendingRequests.remove(request.getId());
                return CompletableFuture.failedFuture(new RuntimeException("No relays available to send request"));
            }
            log.debug("nip46_request_sent request_id={} method={} relays={}", request.getId(), request.getMethod(), sentCount);
            CompletableFuture.delayedExecutor(this.timeout.toMillis(), TimeUnit.MILLISECONDS).execute(() -> {
                CompletableFuture<Nip46Response> pending = this.pendingRequests.remove(request.getId());
                if (pending != null && !pending.isDone()) {
                    pending.completeExceptionally(new TimeoutException("Request timed out: " + request.getId()));
                }
            });
            return future;
        }
        catch (Exception e) {
            this.pendingRequests.remove(request.getId());
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public void close() {
        if (this.subscriptionId != null) {
            try {
                this.relayPool.broadcastClose(this.subscriptionId);
            }
            catch (Exception e) {
                log.debug("Error closing subscription: {}", (Object)e.getMessage());
            }
        }
        this.relayPool.close();
        this.pendingRequests.clear();
        this.connected = false;
        log.info("relay_transport_closed");
    }

    public boolean isConnected() {
        return this.connected && this.relayPool.hasMinimumConnections();
    }

    public int getConnectedRelayCount() {
        return this.relayPool.getConnectedCount();
    }

    private String createSignedEventJson(String content) {
        long createdAt = Instant.now().getEpochSecond();
        String clientPubkeyHex = this.clientIdentity.getPublicKey().toString();
        String tagsJson = String.format("[[\"p\",\"%s\"]]", this.bunkerPubkeyHex);
        String serialized = String.format("[0,\"%s\",%d,%d,%s,\"%s\"]", clientPubkeyHex, createdAt, 24133, tagsJson, this.escapeJson(content));
        String eventId = this.sha256Hex(serialized);
        String signature = this.signMessage(eventId);
        return String.format("{\"id\":\"%s\",\"pubkey\":\"%s\",\"created_at\":%d,\"kind\":%d,\"tags\":%s,\"content\":\"%s\",\"sig\":\"%s\"}", eventId, clientPubkeyHex, createdAt, 24133, tagsJson, this.escapeJson(content), signature);
    }

    private String sha256Hex(String input) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(input.getBytes(StandardCharsets.UTF_8));
            StringBuilder hex = new StringBuilder();
            for (byte b : hash) {
                hex.append(String.format("%02x", b));
            }
            return hex.toString();
        }
        catch (Exception e) {
            throw new RuntimeException("SHA-256 not available", e);
        }
    }

    private String signMessage(String messageHex) {
        try {
            byte[] messageBytes = RelayNip46Transport.hexToBytes(messageHex);
            byte[] privateKeyBytes = this.clientIdentity.getPrivateKey().getRawData();
            byte[] auxRand = new byte[32];
            new SecureRandom().nextBytes(auxRand);
            byte[] signature = Schnorr.sign(messageBytes, privateKeyBytes, auxRand);
            return RelayNip46Transport.bytesToHex(signature);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to sign message", e);
        }
    }

    private static byte[] hexToBytes(String hex) {
        byte[] bytes = new byte[hex.length() / 2];
        for (int i2 = 0; i2 < bytes.length; ++i2) {
            int index = i2 * 2;
            int value = Integer.parseInt(hex.substring(index, index + 2), 16);
            bytes[i2] = (byte)value;
        }
        return bytes;
    }

    private static String bytesToHex(byte[] bytes) {
        StringBuilder hex = new StringBuilder();
        for (byte b : bytes) {
            hex.append(String.format("%02x", b));
        }
        return hex.toString();
    }

    private String escapeJson(String s) {
        return s.replace("\\", "\\\\").replace("\"", "\\\"").replace("\n", "\\n").replace("\r", "\\r").replace("\t", "\\t");
    }

    private Identity createIdentity(String privateKey) {
        String hexKey = this.resolvePrivateKeyToHex(privateKey);
        return Identity.create(hexKey);
    }

    private String resolvePrivateKeyToHex(String privateKey) {
        if (privateKey.startsWith("nsec1")) {
            try {
                return Bech32.fromBech32(privateKey);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid nsec: " + privateKey, e);
            }
        }
        return privateKey;
    }

    private String resolvePubkeyToHex(String pubkey) {
        if (pubkey.startsWith("npub1")) {
            try {
                return Bech32.fromBech32(pubkey);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid npub: " + pubkey, e);
            }
        }
        return pubkey;
    }

    public static RelayNip46TransportBuilder builder() {
        return new RelayNip46TransportBuilder();
    }

    private class ResponseListener
    implements RelayPoolListener {
        private ResponseListener() {
        }

        @Override
        public void onEvent(RelayConnection relay, String subscriptionId, GenericEvent event) {
            if (event.getKind() != 24133) {
                return;
            }
            try {
                this.handleResponseEvent(event);
            }
            catch (Exception e) {
                log.warn("nip46_response_handling_failed event_id={} error={}", (Object)event.getId(), (Object)e.getMessage());
            }
        }

        private void handleResponseEvent(GenericEvent event) {
            String decrypted;
            String content = event.getContent();
            if (content == null || content.isEmpty()) {
                return;
            }
            try {
                decrypted = RelayNip46Transport.this.crypto.decrypt(content);
            }
            catch (Exception e) {
                log.debug("nip46_decrypt_failed, trying as plaintext: {}", (Object)e.getMessage());
                decrypted = content;
            }
            RelayNip46Transport.this.decoder.tryDecodeResponse(decrypted).ifPresent(response -> {
                log.debug("nip46_response_received request_id={} result={}", (Object)response.getId(), (Object)(response.getResult() != null ? "present" : "error"));
                CompletableFuture<Nip46Response> future = RelayNip46Transport.this.pendingRequests.remove(response.getId());
                if (future != null) {
                    future.complete((Nip46Response)response);
                } else {
                    log.debug("nip46_response_orphaned request_id={}", (Object)response.getId());
                }
            });
        }
    }

    public static class RelayNip46TransportBuilder {
        private List<String> relays;
        private String clientPrivateKey;
        private String bunkerPubkey;
        private Duration timeout;
        private Duration connectTimeout;

        RelayNip46TransportBuilder() {
        }

        public RelayNip46TransportBuilder relays(List<String> relays) {
            this.relays = relays;
            return this;
        }

        public RelayNip46TransportBuilder clientPrivateKey(String clientPrivateKey) {
            this.clientPrivateKey = clientPrivateKey;
            return this;
        }

        public RelayNip46TransportBuilder bunkerPubkey(String bunkerPubkey) {
            this.bunkerPubkey = bunkerPubkey;
            return this;
        }

        public RelayNip46TransportBuilder timeout(Duration timeout2) {
            this.timeout = timeout2;
            return this;
        }

        public RelayNip46TransportBuilder connectTimeout(Duration connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public RelayNip46Transport build() {
            return new RelayNip46Transport(this.relays, this.clientPrivateKey, this.bunkerPubkey, this.timeout, this.connectTimeout);
        }

        public String toString() {
            return "RelayNip46Transport.RelayNip46TransportBuilder(relays=" + String.valueOf(this.relays) + ", clientPrivateKey=" + this.clientPrivateKey + ", bunkerPubkey=" + this.bunkerPubkey + ", timeout=" + String.valueOf(this.timeout) + ", connectTimeout=" + String.valueOf(this.connectTimeout) + ")";
        }
    }
}

