/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.nsecbunker.connection;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import nostr.event.impl.GenericEvent;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.nsecbunker.connection.ConnectionHealth;
import xyz.tcheeric.nsecbunker.connection.ConnectionListener;
import xyz.tcheeric.nsecbunker.connection.ConnectionState;
import xyz.tcheeric.nsecbunker.connection.ReconnectionStrategy;
import xyz.tcheeric.nsecbunker.connection.RelayHealthMonitor;
import xyz.tcheeric.nsecbunker.connection.RelayListener;
import xyz.tcheeric.nsecbunker.core.exception.BunkerConnectionException;

public class RelayConnection {
    private static final Logger log = LoggerFactory.getLogger(RelayConnection.class);
    private static final int NORMAL_CLOSURE = 1000;
    private static final int GOING_AWAY = 1001;
    private final String url;
    private final OkHttpClient client;
    private final ObjectMapper objectMapper;
    private final Duration connectTimeout;
    private final AtomicReference<ConnectionState> state;
    private volatile WebSocket webSocket;
    private final List<RelayListener> listeners;
    private final List<ConnectionListener> connectionListeners;
    private volatile CountDownLatch connectLatch;
    private volatile Throwable connectError;
    private volatile ReconnectionStrategy reconnectionStrategy;
    private final AtomicInteger reconnectionAttempt;
    private final AtomicBoolean autoReconnectEnabled;
    private volatile ScheduledExecutorService reconnectExecutor;
    private volatile ScheduledFuture<?> reconnectFuture;
    private volatile boolean closedByUser;
    private volatile RelayHealthMonitor healthMonitor;

    public RelayConnection(String url) {
        this(url, null, null);
    }

    public RelayConnection(String url, OkHttpClient client) {
        this(url, client, null);
    }

    public RelayConnection(String url, OkHttpClient client, Duration connectTimeout) {
        Objects.requireNonNull(url, "URL must not be null");
        if (!url.startsWith("wss://") && !url.startsWith("ws://")) {
            throw new IllegalArgumentException("URL must start with wss:// or ws://");
        }
        this.url = url;
        this.connectTimeout = connectTimeout != null ? connectTimeout : Duration.ofSeconds(30L);
        this.client = client != null ? client : this.createDefaultClient();
        this.objectMapper = new ObjectMapper();
        this.state = new AtomicReference<ConnectionState>(ConnectionState.DISCONNECTED);
        this.listeners = new CopyOnWriteArrayList<RelayListener>();
        this.connectionListeners = new CopyOnWriteArrayList<ConnectionListener>();
        this.reconnectionStrategy = ReconnectionStrategy.none();
        this.reconnectionAttempt = new AtomicInteger(0);
        this.autoReconnectEnabled = new AtomicBoolean(false);
        this.closedByUser = false;
    }

    public ConnectionState getState() {
        return this.state.get();
    }

    public boolean isConnected() {
        return this.state.get() == ConnectionState.CONNECTED;
    }

    public void addListener(RelayListener listener) {
        if (listener != null) {
            this.listeners.add(listener);
        }
    }

    public void removeListener(RelayListener listener) {
        this.listeners.remove(listener);
    }

    public void addConnectionListener(ConnectionListener listener) {
        if (listener != null) {
            this.connectionListeners.add(listener);
        }
    }

    public void removeConnectionListener(ConnectionListener listener) {
        this.connectionListeners.remove(listener);
    }

    public RelayConnection setReconnectionStrategy(ReconnectionStrategy strategy) {
        this.reconnectionStrategy = strategy != null ? strategy : ReconnectionStrategy.none();
        this.autoReconnectEnabled.set(strategy != null && strategy.getMaxAttempts() != 0);
        return this;
    }

    public ReconnectionStrategy getReconnectionStrategy() {
        return this.reconnectionStrategy;
    }

    public RelayConnection enableAutoReconnect() {
        this.autoReconnectEnabled.set(true);
        return this;
    }

    public RelayConnection disableAutoReconnect() {
        this.autoReconnectEnabled.set(false);
        this.cancelReconnect();
        return this;
    }

    public boolean isAutoReconnectEnabled() {
        return this.autoReconnectEnabled.get();
    }

    public int getReconnectionAttempt() {
        return this.reconnectionAttempt.get();
    }

    public synchronized RelayHealthMonitor getHealthMonitor() {
        if (this.healthMonitor == null) {
            this.healthMonitor = new RelayHealthMonitor(this);
        }
        return this.healthMonitor;
    }

    public ConnectionHealth getHealth() {
        RelayHealthMonitor monitor = this.healthMonitor;
        if (monitor != null) {
            return monitor.getHealth();
        }
        return this.isConnected() ? ConnectionHealth.connected(this.url) : ConnectionHealth.disconnected(this.url);
    }

    public void connect() throws BunkerConnectionException {
        this.connect(this.connectTimeout);
    }

    public void connect(Duration timeout2) throws BunkerConnectionException {
        ConnectionState currentState = this.state.get();
        if (currentState == ConnectionState.CONNECTED) {
            log.debug("Already connected to {}", (Object)this.url);
            return;
        }
        if (currentState.isTerminal()) {
            throw new BunkerConnectionException("Connection is in terminal state: " + String.valueOf((Object)currentState), this.url);
        }
        if (!this.state.compareAndSet(currentState, ConnectionState.CONNECTING)) {
            throw new BunkerConnectionException("Connection state changed during connect attempt", this.url);
        }
        this.connectLatch = new CountDownLatch(1);
        this.connectError = null;
        Request request = new Request.Builder().url(this.url).build();
        this.webSocket = this.client.newWebSocket(request, new RelayWebSocketListener());
        try {
            boolean connected = this.connectLatch.await(timeout2.toMillis(), TimeUnit.MILLISECONDS);
            if (!connected) {
                this.state.set(ConnectionState.FAILED);
                this.close();
                throw new BunkerConnectionException("Connection timed out after " + timeout2.toMillis() + "ms", this.url);
            }
            if (this.connectError != null) {
                this.state.set(ConnectionState.FAILED);
                throw new BunkerConnectionException("Connection failed: " + this.connectError.getMessage(), this.url, this.connectError);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.state.set(ConnectionState.FAILED);
            throw new BunkerConnectionException("Connection interrupted", this.url, e);
        }
    }

    public void connectAsync() {
        ConnectionState currentState = this.state.get();
        if (currentState == ConnectionState.CONNECTED || currentState.isTerminal()) {
            return;
        }
        if (!this.state.compareAndSet(currentState, ConnectionState.CONNECTING)) {
            return;
        }
        Request request = new Request.Builder().url(this.url).build();
        this.webSocket = this.client.newWebSocket(request, new RelayWebSocketListener());
    }

    public boolean send(String message) {
        WebSocket ws = this.webSocket;
        if (ws == null || this.state.get() != ConnectionState.CONNECTED) {
            throw new IllegalStateException("Not connected to relay");
        }
        log.debug("Sending to {}: {}", (Object)this.url, (Object)message);
        boolean sent = ws.send(message);
        if (sent && this.healthMonitor != null) {
            this.healthMonitor.recordMessageSent();
        }
        return sent;
    }

    public boolean sendReq(String subscriptionId, String ... filters) {
        StringBuilder sb = new StringBuilder("[\"REQ\",\"").append(subscriptionId).append("\"");
        for (String filter2 : filters) {
            sb.append(",").append(filter2);
        }
        sb.append("]");
        return this.send(sb.toString());
    }

    public boolean sendClose(String subscriptionId) {
        return this.send("[\"CLOSE\",\"" + subscriptionId + "\"]");
    }

    public boolean sendEvent(String eventJson) {
        return this.send("[\"EVENT\"," + eventJson + "]");
    }

    public void close() {
        this.close(1000, "Client closing");
    }

    public void close(int code, String reason) {
        ConnectionState currentState = this.state.get();
        if (currentState.isTerminal()) {
            return;
        }
        this.closedByUser = true;
        this.cancelReconnect();
        RelayHealthMonitor monitor = this.healthMonitor;
        if (monitor != null && monitor.isRunning()) {
            monitor.stop();
        }
        this.state.set(ConnectionState.DISCONNECTING);
        WebSocket ws = this.webSocket;
        if (ws != null) {
            try {
                ws.close(code, reason);
            }
            catch (Exception e) {
                log.debug("Error closing WebSocket: {}", (Object)e.getMessage());
            }
        }
        this.state.set(ConnectionState.CLOSED);
        this.shutdownReconnectExecutor();
    }

    private OkHttpClient createDefaultClient() {
        return new OkHttpClient.Builder().connectTimeout(this.connectTimeout).readTimeout(Duration.ofMinutes(5L)).writeTimeout(Duration.ofSeconds(30L)).pingInterval(Duration.ofSeconds(30L)).build();
    }

    private void setState(ConnectionState newState) {
        ConnectionState oldState = this.state.getAndSet(newState);
        if (oldState != newState) {
            for (RelayListener relayListener : this.listeners) {
                try {
                    relayListener.onStateChange(this, oldState, newState);
                }
                catch (Exception e) {
                    log.warn("Error in listener onStateChange: {}", (Object)e.getMessage());
                }
            }
            for (ConnectionListener connectionListener : this.connectionListeners) {
                try {
                    connectionListener.onStateChanged(this.url, oldState, newState);
                }
                catch (Exception e) {
                    log.warn("Error in ConnectionListener.onStateChanged: {}", (Object)e.getMessage());
                }
            }
        }
    }

    private void notifyError(Throwable t) {
        for (RelayListener relayListener : this.listeners) {
            try {
                relayListener.onError(this, t);
            }
            catch (Exception e) {
                log.warn("Error in listener onError: {}", (Object)e.getMessage());
            }
        }
        for (ConnectionListener connectionListener : this.connectionListeners) {
            try {
                connectionListener.onError(this.url, t);
            }
            catch (Exception e) {
                log.warn("Error in ConnectionListener.onError: {}", (Object)e.getMessage());
            }
        }
    }

    private void notifyReconnecting(int attempt) {
        for (ConnectionListener listener : this.connectionListeners) {
            try {
                listener.onReconnecting(this.url, attempt);
            }
            catch (Exception e) {
                log.warn("Error in ConnectionListener.onReconnecting: {}", (Object)e.getMessage());
            }
        }
    }

    private void scheduleReconnect(Throwable cause) {
        if (!this.autoReconnectEnabled.get() || this.closedByUser) {
            return;
        }
        ReconnectionStrategy strategy = this.reconnectionStrategy;
        int attempt = this.reconnectionAttempt.incrementAndGet();
        if (!strategy.shouldReconnect(attempt)) {
            log.info("Max reconnection attempts ({}) reached for {}", (Object)(attempt - 1), (Object)this.url);
            this.reconnectionAttempt.set(0);
            return;
        }
        Optional<Duration> delayOpt = strategy.getDelay(attempt);
        if (delayOpt.isEmpty()) {
            return;
        }
        Duration delay = delayOpt.get();
        log.info("Scheduling reconnection attempt {} for {} in {}ms", attempt, this.url, delay.toMillis());
        strategy.onAttemptFailed(attempt, cause);
        this.notifyReconnecting(attempt);
        ScheduledExecutorService executor = this.getOrCreateReconnectExecutor();
        this.reconnectFuture = executor.schedule(() -> {
            try {
                if (!this.autoReconnectEnabled.get() || this.closedByUser) {
                    return;
                }
                ConnectionState currentState = this.state.get();
                if (currentState == ConnectionState.CONNECTED) {
                    this.reconnectionAttempt.set(0);
                    return;
                }
                if (currentState.isTerminal()) {
                    this.state.set(ConnectionState.DISCONNECTED);
                }
                this.setState(ConnectionState.RECONNECTING);
                log.info("Attempting reconnection {} to {}", (Object)attempt, (Object)this.url);
                Request request = new Request.Builder().url(this.url).build();
                this.webSocket = this.client.newWebSocket(request, new RelayWebSocketListener());
            }
            catch (Exception e) {
                log.error("Error during reconnection attempt {}: {}", (Object)attempt, (Object)e.getMessage());
                this.scheduleReconnect(e);
            }
        }, delay.toMillis(), TimeUnit.MILLISECONDS);
    }

    private synchronized ScheduledExecutorService getOrCreateReconnectExecutor() {
        if (this.reconnectExecutor == null || this.reconnectExecutor.isShutdown()) {
            ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(1);
            executor.setRemoveOnCancelPolicy(true);
            this.reconnectExecutor = executor;
        }
        return this.reconnectExecutor;
    }

    private void cancelReconnect() {
        ScheduledFuture<?> future = this.reconnectFuture;
        if (future != null) {
            future.cancel(false);
            this.reconnectFuture = null;
        }
    }

    private void shutdownReconnectExecutor() {
        ScheduledExecutorService executor = this.reconnectExecutor;
        if (executor != null) {
            executor.shutdownNow();
            this.reconnectExecutor = null;
        }
    }

    private void handleMessage(String text) {
        if (this.healthMonitor != null) {
            this.healthMonitor.recordMessageReceived();
        }
        for (RelayListener listener : this.listeners) {
            try {
                listener.onRawMessage(this, text);
            }
            catch (Exception e) {
                log.warn("Error in listener onRawMessage: {}", (Object)e.getMessage());
            }
        }
        try {
            String messageType;
            JsonNode root = this.objectMapper.readTree(text);
            if (!root.isArray() || root.isEmpty()) {
                log.warn("Invalid message format from {}: {}", (Object)this.url, (Object)text);
                return;
            }
            switch (messageType = root.get(0).asText()) {
                case "EVENT": {
                    this.handleEventMessage(root);
                    break;
                }
                case "OK": {
                    this.handleOkMessage(root);
                    break;
                }
                case "EOSE": {
                    this.handleEoseMessage(root);
                    break;
                }
                case "NOTICE": {
                    this.handleNoticeMessage(root);
                    break;
                }
                case "CLOSED": {
                    this.handleClosedMessage(root);
                    break;
                }
                case "AUTH": {
                    this.handleAuthMessage(root);
                    break;
                }
                default: {
                    log.debug("Unknown message type from {}: {}", (Object)this.url, (Object)messageType);
                    break;
                }
            }
        }
        catch (JsonProcessingException e) {
            log.warn("Failed to parse message from {}: {}", (Object)this.url, (Object)e.getMessage());
        }
    }

    private void handleEventMessage(JsonNode root) {
        if (root.size() < 3) {
            return;
        }
        String subscriptionId = root.get(1).asText();
        JsonNode eventNode = root.get(2);
        try {
            GenericEvent event = this.objectMapper.treeToValue((TreeNode)eventNode, GenericEvent.class);
            for (RelayListener listener : this.listeners) {
                try {
                    listener.onEvent(this, subscriptionId, event);
                }
                catch (Exception e) {
                    log.warn("Error in listener onEvent: {}", (Object)e.getMessage());
                }
            }
        }
        catch (JsonProcessingException e) {
            log.warn("Failed to parse event from {}: {}", (Object)this.url, (Object)e.getMessage());
        }
    }

    private void handleOkMessage(JsonNode root) {
        if (root.size() < 3) {
            return;
        }
        String eventId = root.get(1).asText();
        boolean success = root.get(2).asBoolean();
        String message = root.size() > 3 ? root.get(3).asText() : null;
        for (RelayListener listener : this.listeners) {
            try {
                listener.onOk(this, eventId, success, message);
            }
            catch (Exception e) {
                log.warn("Error in listener onOk: {}", (Object)e.getMessage());
            }
        }
    }

    private void handleEoseMessage(JsonNode root) {
        if (root.size() < 2) {
            return;
        }
        String subscriptionId = root.get(1).asText();
        for (RelayListener listener : this.listeners) {
            try {
                listener.onEndOfStoredEvents(this, subscriptionId);
            }
            catch (Exception e) {
                log.warn("Error in listener onEndOfStoredEvents: {}", (Object)e.getMessage());
            }
        }
    }

    private void handleNoticeMessage(JsonNode root) {
        if (root.size() < 2) {
            return;
        }
        String message = root.get(1).asText();
        for (RelayListener listener : this.listeners) {
            try {
                listener.onNotice(this, message);
            }
            catch (Exception e) {
                log.warn("Error in listener onNotice: {}", (Object)e.getMessage());
            }
        }
    }

    private void handleClosedMessage(JsonNode root) {
        if (root.size() < 3) {
            return;
        }
        String subscriptionId = root.get(1).asText();
        String message = root.get(2).asText();
        for (RelayListener listener : this.listeners) {
            try {
                listener.onClosed(this, subscriptionId, message);
            }
            catch (Exception e) {
                log.warn("Error in listener onClosed: {}", (Object)e.getMessage());
            }
        }
    }

    private void handleAuthMessage(JsonNode root) {
        if (root.size() < 2) {
            return;
        }
        String challenge = root.get(1).asText();
        for (RelayListener listener : this.listeners) {
            try {
                listener.onAuth(this, challenge);
            }
            catch (Exception e) {
                log.warn("Error in listener onAuth: {}", (Object)e.getMessage());
            }
        }
    }

    public String toString() {
        return "RelayConnection{url='" + this.url + "', state=" + String.valueOf((Object)this.state.get()) + "}";
    }

    public String getUrl() {
        return this.url;
    }

    public Duration getConnectTimeout() {
        return this.connectTimeout;
    }

    private class RelayWebSocketListener
    extends WebSocketListener {
        private RelayWebSocketListener() {
        }

        @Override
        public void onOpen(@NotNull WebSocket webSocket, @NotNull Response response) {
            log.info("Connected to relay: {}", (Object)RelayConnection.this.url);
            RelayConnection.this.reconnectionAttempt.set(0);
            RelayConnection.this.closedByUser = false;
            if (RelayConnection.this.reconnectionStrategy != null) {
                RelayConnection.this.reconnectionStrategy.reset();
            }
            RelayConnection.this.setState(ConnectionState.CONNECTED);
            for (RelayListener relayListener : RelayConnection.this.listeners) {
                try {
                    relayListener.onConnect(RelayConnection.this);
                }
                catch (Exception e) {
                    log.warn("Error in listener onConnect: {}", (Object)e.getMessage());
                }
            }
            for (ConnectionListener connectionListener : RelayConnection.this.connectionListeners) {
                try {
                    connectionListener.onConnected(RelayConnection.this.url);
                }
                catch (Exception e) {
                    log.warn("Error in ConnectionListener.onConnected: {}", (Object)e.getMessage());
                }
            }
            CountDownLatch latch = RelayConnection.this.connectLatch;
            if (latch != null) {
                latch.countDown();
            }
        }

        @Override
        public void onMessage(@NotNull WebSocket webSocket, @NotNull String text) {
            log.debug("Received from {}: {}", (Object)RelayConnection.this.url, (Object)text);
            RelayConnection.this.handleMessage(text);
        }

        @Override
        public void onClosing(@NotNull WebSocket webSocket, int code, @NotNull String reason) {
            log.debug("Relay closing: {} - {} {}", RelayConnection.this.url, code, reason);
            webSocket.close(code, reason);
        }

        @Override
        public void onClosed(@NotNull WebSocket webSocket, int code, @NotNull String reason) {
            log.info("Disconnected from relay: {} - {} {}", RelayConnection.this.url, code, reason);
            ConnectionState previousState = RelayConnection.this.state.get();
            if (!previousState.isTerminal()) {
                RelayConnection.this.setState(ConnectionState.DISCONNECTED);
            }
            for (RelayListener relayListener : RelayConnection.this.listeners) {
                try {
                    relayListener.onDisconnect(RelayConnection.this, code, reason);
                }
                catch (Exception e) {
                    log.warn("Error in listener onDisconnect: {}", (Object)e.getMessage());
                }
            }
            for (ConnectionListener connectionListener : RelayConnection.this.connectionListeners) {
                try {
                    connectionListener.onDisconnected(RelayConnection.this.url, code, reason);
                }
                catch (Exception e) {
                    log.warn("Error in ConnectionListener.onDisconnected: {}", (Object)e.getMessage());
                }
            }
            CountDownLatch latch = RelayConnection.this.connectLatch;
            if (latch != null) {
                latch.countDown();
            }
            if (!RelayConnection.this.closedByUser && code != 1000 && code != 1001) {
                RelayConnection.this.scheduleReconnect(new RuntimeException("Connection closed: " + code + " " + reason));
            }
        }

        @Override
        public void onFailure(@NotNull WebSocket webSocket, @NotNull Throwable t, @Nullable Response response) {
            log.error("Connection failure to {}: {}", (Object)RelayConnection.this.url, (Object)t.getMessage());
            RelayConnection.this.connectError = t;
            RelayConnection.this.setState(ConnectionState.FAILED);
            RelayConnection.this.notifyError(t);
            CountDownLatch latch = RelayConnection.this.connectLatch;
            if (latch != null) {
                latch.countDown();
            }
            RelayConnection.this.scheduleReconnect(t);
        }
    }
}

