/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.nsecbunker.core.connection;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;
import xyz.tcheeric.nsecbunker.core.model.BunkerConnection;

public final class BunkerConnectionString {
    public static final String SCHEME = "bunker";
    public static final String PARAM_RELAY = "relay";
    public static final String PARAM_SECRET = "secret";
    private static final Pattern HEX_PUBKEY_PATTERN = Pattern.compile("^[0-9a-fA-F]{64}$");
    private static final Pattern NPUB_PATTERN = Pattern.compile("^npub1[a-z0-9]{58}$");
    private static final Pattern RELAY_URL_PATTERN = Pattern.compile("^wss?://[^\\s]+$");

    private BunkerConnectionString() {
    }

    public static BunkerConnection parse(String connectionString) {
        URI uri;
        Objects.requireNonNull(connectionString, "Connection string must not be null");
        String trimmed = connectionString.trim();
        if (trimmed.isEmpty()) {
            throw new IllegalArgumentException("Connection string must not be empty");
        }
        try {
            uri = new URI(trimmed);
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid connection string URI: " + e.getMessage(), e);
        }
        String scheme = uri.getScheme();
        if (scheme == null || !scheme.equalsIgnoreCase(SCHEME)) {
            throw new IllegalArgumentException("Invalid scheme: expected 'bunker' but got '" + scheme + "'");
        }
        String pubkey = BunkerConnectionString.extractPubkey(uri);
        if (pubkey == null || pubkey.isEmpty()) {
            throw new IllegalArgumentException("Missing public key in connection string");
        }
        if (!BunkerConnectionString.isValidPubkey(pubkey)) {
            throw new IllegalArgumentException("Invalid public key format: must be 64-char hex or npub1...");
        }
        Map<String, List<String>> params = BunkerConnectionString.parseQueryParams(uri.getRawQuery());
        List relays = params.getOrDefault(PARAM_RELAY, Collections.emptyList());
        ArrayList<String> validatedRelays = new ArrayList<String>();
        for (String relay : relays) {
            String decoded = BunkerConnectionString.urlDecode(relay);
            if (!BunkerConnectionString.isValidRelayUrl(decoded)) {
                throw new IllegalArgumentException("Invalid relay URL: " + decoded);
            }
            validatedRelays.add(decoded);
        }
        List secrets = params.getOrDefault(PARAM_SECRET, Collections.emptyList());
        String secret = secrets.isEmpty() ? null : BunkerConnectionString.urlDecode((String)secrets.get(0));
        return BunkerConnection.builder().remotePubkey(pubkey).relays(validatedRelays).secret(secret).build();
    }

    public static BunkerConnection tryParse(String connectionString) {
        try {
            return BunkerConnectionString.parse(connectionString);
        }
        catch (Exception e) {
            return null;
        }
    }

    public static boolean isValid(String connectionString) {
        return BunkerConnectionString.tryParse(connectionString) != null;
    }

    public static String build(String pubkey, List<String> relays, String secret) {
        Objects.requireNonNull(pubkey, "Public key must not be null");
        if (!BunkerConnectionString.isValidPubkey(pubkey)) {
            throw new IllegalArgumentException("Invalid public key format");
        }
        StringBuilder sb = new StringBuilder();
        sb.append(SCHEME).append("://").append(pubkey);
        boolean hasParams = false;
        if (relays != null && !relays.isEmpty()) {
            for (String relay : relays) {
                if (!BunkerConnectionString.isValidRelayUrl(relay)) {
                    throw new IllegalArgumentException("Invalid relay URL: " + relay);
                }
                sb.append(hasParams ? "&" : "?");
                sb.append(PARAM_RELAY).append("=").append(BunkerConnectionString.urlEncode(relay));
                hasParams = true;
            }
        }
        if (secret != null && !secret.isEmpty()) {
            sb.append(hasParams ? "&" : "?");
            sb.append(PARAM_SECRET).append("=").append(BunkerConnectionString.urlEncode(secret));
        }
        return sb.toString();
    }

    public static String build(BunkerConnection connection) {
        Objects.requireNonNull(connection, "Connection must not be null");
        return BunkerConnectionString.build(connection.getRemotePubkey(), connection.getRelays(), connection.getSecret());
    }

    public static boolean isValidPubkey(String pubkey) {
        if (pubkey == null || pubkey.isEmpty()) {
            return false;
        }
        return HEX_PUBKEY_PATTERN.matcher(pubkey).matches() || NPUB_PATTERN.matcher(pubkey).matches();
    }

    public static boolean isValidRelayUrl(String url) {
        if (url == null || url.isEmpty()) {
            return false;
        }
        return RELAY_URL_PATTERN.matcher(url).matches();
    }

    private static String extractPubkey(URI uri) {
        String host = uri.getHost();
        if (host != null && !host.isEmpty()) {
            return host;
        }
        String authority = uri.getAuthority();
        if (authority != null && !authority.isEmpty()) {
            int colonIndex;
            int atIndex = authority.indexOf(64);
            if (atIndex >= 0) {
                authority = authority.substring(atIndex + 1);
            }
            if ((colonIndex = authority.indexOf(58)) >= 0) {
                authority = authority.substring(0, colonIndex);
            }
            return authority;
        }
        String ssp = uri.getSchemeSpecificPart();
        if (ssp != null && ssp.startsWith("//")) {
            String remaining = ssp.substring(2);
            int queryIndex = remaining.indexOf(63);
            if (queryIndex >= 0) {
                remaining = remaining.substring(0, queryIndex);
            }
            return remaining;
        }
        return null;
    }

    private static Map<String, List<String>> parseQueryParams(String rawQuery) {
        String[] pairs;
        LinkedHashMap<String, List<String>> params = new LinkedHashMap<String, List<String>>();
        if (rawQuery == null || rawQuery.isEmpty()) {
            return params;
        }
        for (String pair : pairs = rawQuery.split("&")) {
            String value;
            String key;
            int eqIndex = pair.indexOf(61);
            if (eqIndex >= 0) {
                key = pair.substring(0, eqIndex);
                value = pair.substring(eqIndex + 1);
            } else {
                key = pair;
                value = "";
            }
            params.computeIfAbsent(key, k -> new ArrayList()).add(value);
        }
        return params;
    }

    private static String urlDecode(String value) {
        try {
            return URLDecoder.decode(value, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            return value;
        }
    }

    private static String urlEncode(String value) {
        try {
            return URLEncoder.encode(value, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            return value;
        }
    }
}

