/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.cli.commands;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;
import xyz.tcheeric.bips.bip39.Bip39;
import xyz.tcheeric.bips.bip39.MnemonicValidator;
import xyz.tcheeric.cashu.common.DeterministicSecret;
import xyz.tcheeric.cashu.common.KeySet;
import xyz.tcheeric.cashu.common.KeysetId;
import xyz.tcheeric.cashu.common.Proof;
import xyz.tcheeric.cashu.common.Signature;
import xyz.tcheeric.cashu.entities.rest.GetActiveKeySetsResponse;
import xyz.tcheeric.cashu.entities.rest.GetKeySetsResponse;
import xyz.tcheeric.cashu.wallet.client.service.WalletRecoveryService;
import xyz.tcheeric.wallet.cli.WalletMain;
import xyz.tcheeric.wallet.cli.mnemonic.MnemonicStorageService;
import xyz.tcheeric.wallet.cli.recovery.WalletRecoveryServiceFactory;
import xyz.tcheeric.wallet.core.BalanceService;
import xyz.tcheeric.wallet.core.H2WalletService;
import xyz.tcheeric.wallet.core.H2WalletStorageAdapter;
import xyz.tcheeric.wallet.core.ReceiveService;
import xyz.tcheeric.wallet.core.WalletConfig;
import xyz.tcheeric.wallet.core.api.adapter.CashuWalletMintApi;
import xyz.tcheeric.wallet.core.api.adapter.RestTemplateFactory;
import xyz.tcheeric.wallet.core.exception.WalletOperationException;
import xyz.tcheeric.wallet.core.ports.WalletStorage;
import xyz.tcheeric.wallet.core.proof.NewProof;
import xyz.tcheeric.wallet.core.state.WalletState;
import xyz.tcheeric.wallet.storage.file.FileWalletStorage;

@CommandLine.Command(name="recover", description={"Recover wallet from BIP39 mnemonic phrase (NUT-13)"}, mixinStandardHelpOptions=true)
public class RecoverWalletCmd
implements Runnable {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(RecoverWalletCmd.class);
    private static final String DEFAULT_WALLET_ID = "default";
    @CommandLine.ParentCommand
    WalletMain parent;
    @CommandLine.Option(names={"-m", "--mnemonic"}, description={"BIP39 mnemonic phrase (12-24 words)"}, required=true, interactive=true)
    private String mnemonic;
    @CommandLine.Option(names={"-p", "--passphrase"}, description={"Optional BIP39 passphrase (empty if none)"}, interactive=true)
    private String passphrase = "";
    @CommandLine.Option(names={"--mint"}, description={"Mint URL to recover from (default: from config)"})
    private String mintUrl;
    @CommandLine.Option(names={"-k", "--keysets"}, description={"Comma-separated keyset IDs to recover from (default: all active keysets)"}, split=",")
    private String[] keysetIds;
    @CommandLine.Option(names={"--batch-size"}, description={"Number of secrets to derive per batch. Default: ${DEFAULT-VALUE}"}, defaultValue="100")
    private int batchSize;
    @CommandLine.Option(names={"--store-mnemonic"}, description={"Store mnemonic encrypted in wallet state for future use"})
    boolean storeMnemonic;
    @CommandLine.Option(names={"--store-passphrase"}, description={"Passphrase used to encrypt mnemonic when storing"}, interactive=true, arity="0..1")
    String storePassphrase;
    private final WalletRecoveryServiceFactory recoveryServiceFactory;
    private final ReceiveService receiveService;
    private final MnemonicStorageService mnemonicStorageService;
    private final WalletStorage overrideWalletStorage;

    public RecoverWalletCmd() {
        this(new WalletRecoveryServiceFactory(), null, new MnemonicStorageService(), null);
    }

    public RecoverWalletCmd(WalletRecoveryServiceFactory recoveryServiceFactory, BalanceService balanceService) {
        this(recoveryServiceFactory, balanceService instanceof ReceiveService ? (ReceiveService)balanceService : null, new MnemonicStorageService(), null);
    }

    RecoverWalletCmd(WalletRecoveryServiceFactory recoveryServiceFactory, ReceiveService receiveService, MnemonicStorageService mnemonicStorageService, WalletStorage overrideWalletStorage) {
        this.recoveryServiceFactory = recoveryServiceFactory;
        this.receiveService = receiveService;
        this.mnemonicStorageService = mnemonicStorageService;
        this.overrideWalletStorage = overrideWalletStorage;
    }

    @Override
    public void run() {
        try {
            if (!this.validateMnemonic()) {
                return;
            }
            WalletConfig config = WalletConfig.load();
            String resolvedMintUrl = this.mintUrl != null ? this.mintUrl : config.defaultMintUrl();
            log.info("wallet_recovery_started mint_url={}", (Object)resolvedMintUrl);
            List<KeysetId> targetKeysetIds = this.resolveKeysetIds(resolvedMintUrl);
            if (targetKeysetIds.isEmpty()) {
                System.err.println("\n\u274c No keysets to recover from.");
                System.err.println("Suggestion: Use --keysets to specify keyset IDs, or ensure the mint has active keysets.");
                return;
            }
            List<KeySet> keySets = this.fetchKeySets(resolvedMintUrl, targetKeysetIds);
            if (keySets.isEmpty()) {
                System.err.println("\n\u274c Failed to fetch keyset public keys from mint.");
                System.err.println("Suggestion: Verify the mint URL is correct and the mint is running.");
                return;
            }
            this.displayRecoveryInfo(resolvedMintUrl, targetKeysetIds);
            WalletRecoveryService recoveryService = this.recoveryServiceFactory.create(resolvedMintUrl);
            List<Proof<DeterministicSecret>> recoveredProofs = recoveryService.recover(this.mnemonic, this.passphrase, targetKeysetIds, keySets);
            this.importRecoveredProofs(config, resolvedMintUrl, keySets, recoveredProofs);
            this.displayRecoveryResults(recoveredProofs);
            this.storeMnemonicIfRequested();
        }
        catch (Exception e) {
            this.handleError(e);
        }
    }

    private boolean validateMnemonic() {
        this.mnemonic = this.mnemonic.trim().replaceAll("\\s+", " ");
        MnemonicValidator.ValidationResult validation = Bip39.validateMnemonic(this.mnemonic);
        if (!validation.isValid()) {
            System.err.println("\n\u274c Invalid mnemonic phrase: " + validation.getErrorMessage());
            System.err.println("Suggestion: Verify the mnemonic was copied correctly and contains valid BIP39 words.");
            log.error("wallet_recovery_failed reason=invalid_mnemonic error={}", (Object)validation.getErrorMessage());
            return false;
        }
        int wordCount = this.mnemonic.split("\\s+").length;
        if (wordCount != 12 && wordCount != 15 && wordCount != 18 && wordCount != 21 && wordCount != 24) {
            System.err.println("\n\u274c Invalid mnemonic length: " + wordCount + " words");
            System.err.println("Valid lengths: 12, 15, 18, 21, or 24 words");
            log.error("wallet_recovery_failed reason=invalid_word_count word_count={}", (Object)wordCount);
            return false;
        }
        log.info("mnemonic_validated word_count={}", (Object)wordCount);
        return true;
    }

    private List<KeysetId> resolveKeysetIds(String mintUrl) {
        if (this.keysetIds != null && this.keysetIds.length > 0) {
            return Arrays.stream(this.keysetIds).map(KeysetId::fromString).collect(Collectors.toList());
        }
        log.info("fetching_active_keysets mint_url={}", (Object)mintUrl);
        CashuWalletMintApi mintApi = new CashuWalletMintApi(RestTemplateFactory.create());
        try {
            GetActiveKeySetsResponse response = mintApi.activeKeysets(mintUrl);
            if (response == null || response.getActiveKeySets() == null || response.getActiveKeySets().isEmpty()) {
                System.err.println("\n\u26a0\ufe0f  No active keysets found on mint: " + mintUrl);
                return List.of();
            }
            return response.getActiveKeySets().stream().map(ks -> KeysetId.fromString(ks.getId())).collect(Collectors.toList());
        }
        catch (Exception e) {
            System.err.println("\n\u26a0\ufe0f  Failed to fetch active keysets: " + e.getMessage());
            log.error("fetch_active_keysets_failed mint_url={} error={}", mintUrl, e.getMessage(), e);
            return List.of();
        }
    }

    private List<KeySet> fetchKeySets(String mintUrl, List<KeysetId> keysetIds) {
        CashuWalletMintApi mintApi = new CashuWalletMintApi(RestTemplateFactory.create());
        ArrayList<KeySet> keySets = new ArrayList<KeySet>();
        for (KeysetId keysetId : keysetIds) {
            try {
                log.debug("fetching_keyset mint_url={} keyset_id={}", (Object)mintUrl, (Object)keysetId);
                GetKeySetsResponse response = mintApi.keyset(mintUrl, keysetId.toString());
                if (response != null && response.getKeySets() != null && !response.getKeySets().isEmpty()) {
                    keySets.add(response.getKeySets().get(0));
                    continue;
                }
                System.err.println("\u26a0\ufe0f  Keyset not found: " + String.valueOf(keysetId));
                log.warn("keyset_not_found mint_url={} keyset_id={}", (Object)mintUrl, (Object)keysetId);
            }
            catch (Exception e) {
                System.err.println("\u26a0\ufe0f  Failed to fetch keyset " + String.valueOf(keysetId) + ": " + e.getMessage());
                log.error("fetch_keyset_failed mint_url={} keyset_id={} error={}", mintUrl, keysetId, e.getMessage(), e);
            }
        }
        return keySets;
    }

    private void displayRecoveryInfo(String mintUrl, List<KeysetId> keysetIds) {
        if (this.parent != null && this.parent.jsonOutput) {
            System.out.println("{\"status\":\"starting\",\"mint\":\"" + mintUrl + "\",\"keysets\":" + keysetIds.size() + ",\"batchSize\":" + this.batchSize + "}");
        } else {
            System.out.println("\n\ud83d\udd04 Starting wallet recovery...");
            System.out.println("\u2501".repeat(60));
            System.out.println("Mint URL: " + mintUrl);
            System.out.println("Keysets to recover: " + keysetIds.size());
            for (KeysetId keysetId : keysetIds) {
                System.out.println("  \u2022 " + String.valueOf(keysetId));
            }
            System.out.println("Batch size: " + this.batchSize + " secrets per request");
            System.out.println("\u2501".repeat(60));
            System.out.println();
        }
    }

    private void displayRecoveryResults(List<Proof<DeterministicSecret>> recoveredProofs) {
        long totalAmount = recoveredProofs.stream().mapToLong(Proof::getAmount).sum();
        if (this.parent != null && this.parent.jsonOutput) {
            System.out.println("{\"status\":\"success\",\"proofs\":" + recoveredProofs.size() + ",\"amount\":" + totalAmount + "}");
        } else {
            System.out.println("\n\u2705 Recovery complete!");
            System.out.println("\u2501".repeat(60));
            System.out.println("   Proofs recovered: " + recoveredProofs.size());
            System.out.println("   Total amount: " + totalAmount + " sats");
            System.out.println("\u2501".repeat(60));
            System.out.println();
            if (recoveredProofs.isEmpty()) {
                System.out.println("\u26a0\ufe0f  No proofs were recovered.");
                System.out.println("This could mean:");
                System.out.println("  \u2022 The wallet has no balance");
                System.out.println("  \u2022 The mnemonic was never used with this mint");
                System.out.println("  \u2022 The specified keysets are incorrect");
                System.out.println();
            }
        }
        log.info("wallet_recovery_completed proofs_recovered={} total_amount={}", (Object)recoveredProofs.size(), (Object)totalAmount);
    }

    void importRecoveredProofs(WalletConfig config, String mintUrl, List<KeySet> keySets, List<Proof<DeterministicSecret>> recoveredProofs) {
        if (recoveredProofs.isEmpty()) {
            log.info("wallet_recovery_import_skipped reason=no_proofs");
            return;
        }
        if (this.receiveService == null) {
            this.emitUserError("Failed to persist recovered proofs. Receive service unavailable.", "Run the CLI with the default H2 backend or configure wallet storage before recovering.");
            log.warn("wallet_recovery_import_skipped reason=no_receive_service");
            return;
        }
        Map<String, KeySet> keysetById = keySets.stream().collect(Collectors.toMap(KeySet::getId, ks -> ks));
        LinkedHashMap<String, List> proofsByUnit = new LinkedHashMap<String, List>();
        for (Proof<DeterministicSecret> proof : recoveredProofs) {
            String keysetId = proof.getKeySetId();
            KeySet keySet = keysetById.get(keysetId);
            if (keySet == null) {
                log.warn("wallet_recovery_import_skipped_proof reason=unknown_keyset keyset_id={}", (Object)keysetId);
                continue;
            }
            String unit = keySet.getUnit();
            if (unit == null || unit.isBlank()) {
                log.warn("wallet_recovery_import_skipped_proof reason=missing_unit keyset_id={}", (Object)keysetId);
                continue;
            }
            Signature signature = proof.getUnblindedSignature();
            if (signature == null) {
                log.warn("wallet_recovery_import_skipped_proof reason=missing_signature keyset_id={}", (Object)keysetId);
                continue;
            }
            DeterministicSecret secret = proof.getSecret();
            byte[] secretBytes = secret != null ? secret.getData() : new byte[]{};
            proofsByUnit.computeIfAbsent(unit, u -> new ArrayList()).add(new NewProof(proof.getAmount(), signature.toString(), secretBytes, keysetId));
        }
        if (proofsByUnit.isEmpty()) {
            log.warn("wallet_recovery_import_skipped reason=no_valid_proofs");
            return;
        }
        this.receiveService.init(config);
        for (Map.Entry entry : proofsByUnit.entrySet()) {
            String unit = (String)entry.getKey();
            List proofs = (List)entry.getValue();
            try {
                ReceiveService.ImportResult result = this.receiveService.importProofs(new ReceiveService.ImportRequest(mintUrl, unit, proofs));
                log.info("wallet_recovery_import_complete mint={} unit={} imported={} duplicates={}", mintUrl, unit, result.importedCount(), result.duplicateCount());
                if (this.parent != null && this.parent.jsonOutput) continue;
                System.out.printf("Imported %d proofs totaling %d %s (duplicates: %d)%n", result.importedCount(), result.importedAmount(), unit, result.duplicateCount());
            }
            catch (WalletOperationException e) {
                this.emitUserError("Failed to import recovered proofs for unit " + unit + ". " + e.getUserMessage(), "Resolve the mint or storage issue and rerun the recover command.");
                log.error("wallet_recovery_import_failed mint={} unit={} error={}", mintUrl, unit, e.getMessage(), e);
            }
        }
    }

    void storeMnemonicIfRequested() {
        if (!this.storeMnemonic) {
            return;
        }
        if (this.storePassphrase == null || this.storePassphrase.isEmpty()) {
            this.emitUserError("Passphrase required to store mnemonic.", "Provide --store-passphrase=<passphrase> when running the command.");
            return;
        }
        try {
            WalletStorage storage = this.resolveWalletStorage();
            WalletState state = storage.load(DEFAULT_WALLET_ID).orElseThrow(() -> new IllegalStateException("Wallet state not initialized. Suggestion: Run `wallet init` before storing a mnemonic."));
            WalletState updated = this.mnemonicStorageService.storeMnemonic(state, this.mnemonic, this.storePassphrase);
            storage.save(DEFAULT_WALLET_ID, updated);
            if (this.parent != null && this.parent.jsonOutput) {
                System.out.println("{\"status\":\"mnemonic_stored\"}");
            } else {
                System.out.println("Mnemonic encrypted and stored in wallet state.");
            }
            log.info("wallet_recovery_mnemonic_stored");
        }
        catch (Exception e) {
            this.emitUserError("Failed to store mnemonic: " + e.getMessage(), "Verify the wallet is initialized and the passphrase is correct, then retry.");
            log.warn("wallet_recovery_store_mnemonic_failed error={}", (Object)e.getMessage(), (Object)e);
        }
    }

    private WalletStorage resolveWalletStorage() {
        if (this.overrideWalletStorage != null) {
            return this.overrideWalletStorage;
        }
        ReceiveService receiveService = this.receiveService;
        if (receiveService instanceof H2WalletService) {
            H2WalletService h2 = (H2WalletService)receiveService;
            return new H2WalletStorageAdapter(h2.getDataSource());
        }
        return new FileWalletStorage();
    }

    private void emitUserError(String whatHappened, String suggestion) {
        String message = whatHappened + " Suggestion: " + suggestion;
        if (this.parent != null && this.parent.jsonOutput) {
            System.out.println("{\"status\":\"error\",\"message\":\"" + message.replace("\"", "\\\"") + "\"}");
        } else {
            System.err.println("\u274c " + message);
        }
    }

    private void handleError(Exception e) {
        if (this.parent != null && this.parent.jsonOutput) {
            System.out.println("{\"status\":\"error\",\"message\":\"" + e.getMessage().replace("\"", "\\\"") + "\"}");
        } else {
            System.err.println("\n\u274c Recovery failed: " + e.getMessage());
            System.err.println("Suggestion: Check the error message above and verify your inputs.");
        }
        log.error("wallet_recovery_failed error={}", (Object)e.getMessage(), (Object)e);
    }
}

