/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.cli.commands;

import jakarta.validation.ConstraintViolation;
import jakarta.validation.ConstraintViolationException;
import java.io.PrintWriter;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import picocli.CommandLine;
import xyz.tcheeric.wallet.cli.CliUtils;
import xyz.tcheeric.wallet.cli.NostrRelayUtils;
import xyz.tcheeric.wallet.core.WalletConfig;
import xyz.tcheeric.wallet.core.application.SendUseCase;
import xyz.tcheeric.wallet.core.exception.QuoteExpiredException;
import xyz.tcheeric.wallet.core.exception.WalletOperationException;
import xyz.tcheeric.wallet.core.token.TokenDecodingException;

@CommandLine.Command(name="send", description={"Pay a Lightning invoice or deliver proofs via Nostr P2PK, falling back to token links when needed."})
public class SendCmd
implements Runnable {
    private final SendUseCase sendUseCase;
    private final boolean requireH2Backend;
    @CommandLine.Spec
    CommandLine.Model.CommandSpec spec;
    @CommandLine.Option(names={"--mint"}, description={"Mint base URL"})
    String mint;
    @CommandLine.Option(names={"--unit"}, description={"Unit (defaults to config)"})
    String unit;
    @CommandLine.Option(names={"--amount"}, required=true, description={"Amount to send"})
    long amount;
    @CommandLine.Option(names={"--invoice"}, description={"BOLT11 invoice to pay"})
    String invoice;
    @CommandLine.Option(names={"--recipient"}, description={"Recipient Nostr pubkey for P2PK send"})
    String recipient;
    @CommandLine.Option(names={"--relays"}, split=",", description={"Preferred relay list for Nostr delivery"})
    List<String> relays;
    @CommandLine.Option(names={"--no-token-fallback"}, description={"Disable token-link fallback when Nostr delivery fails"})
    boolean disableTokenFallback;

    public SendCmd(SendUseCase sendUseCase, boolean requireH2Backend) {
        this.sendUseCase = Objects.requireNonNull(sendUseCase, "sendUseCase");
        this.requireH2Backend = requireH2Backend;
    }

    @Override
    public void run() {
        boolean hasRecipient;
        WalletConfig config = WalletConfig.load();
        String mintUrl = this.resolveOrDefault(this.mint, config.defaultMintUrl());
        String resolvedUnit = this.resolveOrDefault(this.unit, config.defaultUnit());
        WalletConfig flowConfig = new WalletConfig(mintUrl, resolvedUnit);
        boolean hasInvoice = this.invoice != null && !this.invoice.isBlank();
        boolean bl = hasRecipient = this.recipient != null && !this.recipient.isBlank();
        if (hasInvoice == hasRecipient) {
            throw new CommandLine.ParameterException(this.spec.commandLine(), "Specify exactly one of --invoice or --recipient");
        }
        if (this.requireH2Backend) {
            CliUtils.ensureH2Backend();
        }
        if (hasInvoice) {
            this.handleLightning(flowConfig);
        } else {
            try {
                this.handleP2pk(flowConfig, NostrRelayUtils.normalize(this.relays));
            }
            catch (TokenDecodingException e) {
                throw new CommandLine.ExecutionException(this.commandLine(), "Failed to decode token. " + e.getMessage() + ". Suggestion: Ensure the token string is complete and unmodified, then retry.", e);
            }
        }
    }

    private void handleLightning(WalletConfig flowConfig) {
        try {
            SendUseCase.PayLightningResult result = this.sendUseCase.payLightning(new SendUseCase.PayLightningCommand(flowConfig, this.amount, this.invoice));
            PrintWriter out = this.output();
            if (result.isSuccess()) {
                out.printf("Paid invoice %s using %d %s%n", this.invoice, this.amount, flowConfig.defaultUnit());
            } else {
                out.printf("Lightning payment finished with status %s: %s%n", new Object[]{result.status(), result.message() == null ? "no additional details" : result.message()});
            }
            out.flush();
        }
        catch (ConstraintViolationException e) {
            throw new CommandLine.ParameterException(this.spec.commandLine(), this.violationMessage(e), e);
        }
        catch (QuoteExpiredException e) {
            PrintWriter err = this.error();
            err.printf("Quote expired while paying %d %s. Request a new quote and retry.%n", this.amount, flowConfig.defaultUnit());
            err.flush();
            throw new CommandLine.ExecutionException(this.commandLine(), e.getUserMessage(), e);
        }
        catch (WalletOperationException e) {
            PrintWriter err = this.error();
            err.printf("Failed to pay invoice: %s (code=%s)%n", e.getUserMessage(), e.getErrorCode());
            if (e.isRetryable()) {
                err.println("The operation may succeed if you retry after addressing the issue.");
            }
            err.flush();
            throw new CommandLine.ExecutionException(this.commandLine(), e.getUserMessage(), e);
        }
    }

    private void handleP2pk(WalletConfig flowConfig, List<String> normalizedRelays) throws TokenDecodingException {
        try {
            SendUseCase.DeliverP2pkResult result = this.sendUseCase.deliverP2pk(new SendUseCase.DeliverP2pkCommand(flowConfig, this.amount, this.recipient, normalizedRelays, !this.disableTokenFallback));
            PrintWriter out = this.output();
            if (result.nostrDelivered()) {
                String relaySummary = result.attempts().stream().filter(SendUseCase.RelayAttempt::isSuccess).map(attempt -> this.formatRelayList(attempt.relays())).findFirst().orElse("(default relays)");
                out.printf("Sent %d %s to %s via relays: %s%n", result.totalAmount(), result.unit(), result.recipient(), relaySummary);
            } else if (result.tokenGenerated()) {
                String failedSummary = result.attempts().stream().filter(attempt -> attempt.status() == SendUseCase.AttemptStatus.FAILED).map(attempt -> this.formatRelayList(attempt.relays())).collect(Collectors.joining(", "));
                if (failedSummary.isBlank()) {
                    failedSummary = "(none)";
                }
                out.printf("Nostr delivery failed on relays %s. Generated token link: %s%n", failedSummary, result.token());
            } else {
                out.printf("Send operation completed without Nostr delivery or token generation.%n", new Object[0]);
            }
            out.flush();
        }
        catch (ConstraintViolationException e) {
            throw new CommandLine.ParameterException(this.spec.commandLine(), this.violationMessage(e), e);
        }
        catch (WalletOperationException e) {
            PrintWriter err = this.error();
            err.printf("Failed to deliver proofs: %s (code=%s)%n", e.getUserMessage(), e.getErrorCode());
            if (e.isRetryable()) {
                err.println("The operation may succeed if you retry after addressing the issue.");
            }
            err.flush();
            throw new CommandLine.ExecutionException(this.commandLine(), e.getUserMessage(), e);
        }
    }

    private String formatRelayList(List<String> relays) {
        if (relays == null || relays.isEmpty()) {
            return "(default relays)";
        }
        return String.join((CharSequence)", ", relays);
    }

    private String violationMessage(ConstraintViolationException exception) {
        return exception.getConstraintViolations().stream().map(this::formatViolation).sorted().collect(Collectors.joining("; "));
    }

    private String formatViolation(ConstraintViolation<?> violation) {
        String path = violation.getPropertyPath() == null ? "value" : violation.getPropertyPath().toString();
        return path + " " + violation.getMessage();
    }

    private PrintWriter output() {
        if (this.spec != null && this.spec.commandLine() != null) {
            return this.spec.commandLine().getOut();
        }
        return new PrintWriter(System.out, true);
    }

    private PrintWriter error() {
        if (this.spec != null && this.spec.commandLine() != null) {
            return this.spec.commandLine().getErr();
        }
        return new PrintWriter(System.err, true);
    }

    private CommandLine commandLine() {
        if (this.spec != null && this.spec.commandLine() != null) {
            return this.spec.commandLine();
        }
        return new CommandLine(this);
    }

    private String resolveOrDefault(String value, String defaultValue) {
        return value != null && !value.isBlank() ? value : defaultValue;
    }
}

