/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core;

import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Currency;
import java.util.HexFormat;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.Generated;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.params.Ed25519PrivateKeyParameters;
import org.bouncycastle.crypto.params.Ed25519PublicKeyParameters;
import org.bouncycastle.crypto.signers.Ed25519Signer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.cashu.voucher.domain.VoucherStatus;
import xyz.tcheeric.wallet.core.Balance;
import xyz.tcheeric.wallet.core.SendService;
import xyz.tcheeric.wallet.core.VoucherBackupService;
import xyz.tcheeric.wallet.core.VoucherService;
import xyz.tcheeric.wallet.core.WalletConfig;
import xyz.tcheeric.wallet.core.exception.VoucherImportException;
import xyz.tcheeric.wallet.core.exception.VoucherRedemptionException;
import xyz.tcheeric.wallet.core.exception.WalletOperationException;
import xyz.tcheeric.wallet.core.nostr.NostrGatewayService;
import xyz.tcheeric.wallet.core.nostr.voucher.VoucherLedgerAdapter;
import xyz.tcheeric.wallet.core.ports.WalletStorage;
import xyz.tcheeric.wallet.core.proof.ProofRecord;
import xyz.tcheeric.wallet.core.security.EncryptionService;
import xyz.tcheeric.wallet.core.security.IdentityKey;
import xyz.tcheeric.wallet.core.security.IdentityKeyService;
import xyz.tcheeric.wallet.core.state.SchemaVersion;
import xyz.tcheeric.wallet.core.state.StoredVoucher;
import xyz.tcheeric.wallet.core.state.VoucherBackupState;
import xyz.tcheeric.wallet.core.state.WalletSchemaMetadata;
import xyz.tcheeric.wallet.core.state.WalletState;
import xyz.tcheeric.wallet.core.token.TokenCodec;
import xyz.tcheeric.wallet.core.token.TokenDecodingException;
import xyz.tcheeric.wallet.core.token.TokenEnvelope;

public class VoucherServiceImpl
implements VoucherService {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(VoucherServiceImpl.class);
    private static final String DEFAULT_WALLET_ID = "default";
    private WalletConfig config;
    private final WalletStorage walletStorage;
    private final EncryptionService encryptionService;
    private final VoucherBackupService backupService;
    private final IdentityKeyService identityKeyService;
    private final NostrGatewayService nostrGateway;
    private final SendService sendService;
    private final TokenCodec tokenCodec;
    private VoucherLedgerAdapter ledgerAdapter;

    public VoucherServiceImpl(WalletStorage walletStorage, EncryptionService encryptionService, VoucherBackupService backupService, IdentityKeyService identityKeyService, SendService sendService, TokenCodec tokenCodec) {
        this(walletStorage, encryptionService, backupService, identityKeyService, null, sendService, tokenCodec);
    }

    public VoucherServiceImpl(WalletStorage walletStorage, EncryptionService encryptionService, VoucherBackupService backupService, IdentityKeyService identityKeyService, NostrGatewayService nostrGateway, SendService sendService, TokenCodec tokenCodec) {
        this.walletStorage = Objects.requireNonNull(walletStorage, "walletStorage");
        this.encryptionService = Objects.requireNonNull(encryptionService, "encryptionService");
        this.backupService = Objects.requireNonNull(backupService, "backupService");
        this.identityKeyService = Objects.requireNonNull(identityKeyService, "identityKeyService");
        this.nostrGateway = nostrGateway;
        this.sendService = Objects.requireNonNull(sendService, "sendService");
        this.tokenCodec = Objects.requireNonNull(tokenCodec, "tokenCodec");
    }

    @Override
    public void init(WalletConfig config) {
        this.config = Objects.requireNonNull(config, "config");
        this.sendService.init(config);
        if (!this.walletStorage.exists(DEFAULT_WALLET_ID)) {
            this.initializeEmptyState();
        }
        log.info("voucher_service initialized mint={} send_service_initialized=true", (Object)config.defaultMintUrl());
    }

    private void initializeEmptyState() {
        SchemaVersion v1_0 = SchemaVersion.V1_0;
        WalletSchemaMetadata schema = new WalletSchemaMetadata(v1_0, v1_0, v1_0, List.of(v1_0), List.of(v1_0));
        WalletState initialState = new WalletState(schema, Instant.now(), List.of(), List.of(), List.of());
        this.walletStorage.save(DEFAULT_WALLET_ID, initialState);
        log.debug("voucher_service initialized_empty_state wallet_id={}", (Object)DEFAULT_WALLET_ID);
    }

    @Override
    public Balance balance() {
        return new Balance(Map.of(this.config != null && this.config.defaultUnit() != null ? this.config.defaultUnit() : "sat", 0L));
    }

    @Override
    public VoucherService.IssueVoucherResult issueAndBackup(VoucherService.IssueVoucherRequest request) throws WalletOperationException {
        this.ensureInitialized();
        Objects.requireNonNull(request, "request");
        log.info("voucher_service issuing_voucher amount={} unit={} mint={}", new Object[]{request.amount(), request.unit(), request.mintUrl()});
        SendService.PreparedP2pkSend preparedSend = null;
        try {
            if (!this.isValidUnit(request.unit())) {
                throw new IllegalArgumentException("Invalid unit: " + request.unit() + ". Must be 'sat', 'msat', 'btc', or a valid 3-letter ISO currency code.");
            }
            Balance balance = this.sendService.balance();
            Long availableBalance = balance.totalsByUnit().get(request.unit());
            if (availableBalance == null || availableBalance < request.amount()) {
                long available = availableBalance != null ? availableBalance : 0L;
                throw new IllegalStateException("Insufficient balance for voucher issuance. Required: " + request.amount() + " " + request.unit() + ", Available: " + available + " " + request.unit());
            }
            IdentityKey identity = this.identityKeyService.loadOrCreate();
            String issuerPubkeyHex = identity.publicKeyHex();
            String issuerPrivkeyHex = HexFormat.of().formatHex(identity.privateKeyEncoded());
            String voucherId = this.generateVoucherId();
            Instant issuedAt = Instant.now();
            Long expiresAt = null;
            if (request.expiresInDays() != null) {
                expiresAt = issuedAt.plusSeconds((long)request.expiresInDays().intValue() * 24L * 3600L).getEpochSecond();
            }
            log.debug("voucher_service selecting_proofs amount={} unit={} mint={}", new Object[]{request.amount(), request.unit(), request.mintUrl()});
            preparedSend = this.sendService.prepareP2pkSend(new SendService.P2pkSendRequest(request.amount(), request.unit(), request.mintUrl()));
            log.info("voucher_service proofs_selected count={} total_amount={} voucher_id={}", new Object[]{preparedSend.proofs().size(), preparedSend.totalAmount(), voucherId});
            String metadataPayload = String.format("%s:%s:%d:%s:%d", voucherId, request.unit(), request.amount(), request.mintUrl(), issuedAt.getEpochSecond());
            String issuerSignature = this.signVoucherMetadata(metadataPayload, issuerPrivkeyHex);
            StoredVoucher voucher = new StoredVoucher(voucherId, issuerPubkeyHex, request.unit(), request.amount(), expiresAt, request.memo(), issuerSignature, issuerPubkeyHex, issuedAt, "issued");
            String token = this.createVoucherTokenWithProofs(voucher, preparedSend.proofs(), request.mintUrl(), request.unit());
            WalletState state = this.loadWalletState();
            ArrayList<StoredVoucher> updatedVouchers = new ArrayList<StoredVoucher>(state.vouchers());
            updatedVouchers.add(voucher);
            WalletState updatedState = state.withVouchers(updatedVouchers);
            this.walletStorage.save(DEFAULT_WALLET_ID, updatedState);
            preparedSend.commit();
            log.info("voucher_service voucher_created voucher_id={} amount={} proofs_spent={}", new Object[]{voucherId, request.amount(), preparedSend.proofs().size()});
            if (this.nostrGateway != null) {
                try {
                    if (this.ledgerAdapter == null) {
                        this.ledgerAdapter = new VoucherLedgerAdapter(this.nostrGateway, this.identityKeyService);
                    }
                    this.ledgerAdapter.publishVoucher(voucher, VoucherStatus.ISSUED);
                    log.info("voucher_service voucher_published_to_ledger voucher_id={}", (Object)voucherId);
                }
                catch (Exception e) {
                    log.warn("voucher_service ledger_publish_failed voucher_id={} error={}", (Object)voucherId, (Object)e.getMessage());
                }
            } else {
                log.debug("voucher_service ledger_unavailable voucher_id={} skipping_publish", (Object)voucherId);
            }
            boolean backedUp = false;
            String backupMessage = null;
            try {
                this.backupToNostr();
                backedUp = true;
                log.info("voucher_service voucher_backed_up voucher_id={}", (Object)voucherId);
            }
            catch (Exception e) {
                log.warn("voucher_service backup_failed voucher_id={} error={}", (Object)voucherId, (Object)e.getMessage());
                backupMessage = "Voucher created but backup failed: " + e.getMessage();
            }
            return new VoucherService.IssueVoucherResult(voucher, token, backedUp, (String)(backupMessage != null ? backupMessage : "Voucher issued successfully"));
        }
        catch (Exception e) {
            if (preparedSend != null) {
                try {
                    preparedSend.cancel();
                    log.debug("voucher_service proof_reservation_cancelled");
                }
                catch (Exception cancelEx) {
                    log.warn("voucher_service cancel_reservation_failed error={}", (Object)cancelEx.getMessage());
                }
            }
            log.error("voucher_service issuance_failed_unexpected error={}", (Object)e.getMessage(), (Object)e);
            throw new IllegalStateException("Failed to issue voucher: " + e.getMessage(), e);
        }
    }

    @Override
    public VoucherService.RedeemVoucherResult redeemVoucher(VoucherService.RedeemVoucherRequest request) throws WalletOperationException {
        this.ensureInitialized();
        Objects.requireNonNull(request, "request");
        WalletState state = this.loadWalletState();
        StoredVoucher current = state.vouchers().stream().filter(v -> v.voucherId().equals(request.voucherId())).findFirst().orElseThrow(() -> VoucherRedemptionException.notFound(request.voucherId()));
        if ("redeemed".equalsIgnoreCase(current.status())) {
            return new VoucherService.RedeemVoucherResult(current.voucherId(), "already_redeemed", "Voucher was already redeemed.", false, false);
        }
        if ("revoked".equalsIgnoreCase(current.status())) {
            throw VoucherRedemptionException.revoked(current.voucherId());
        }
        if (current.isExpired()) {
            throw VoucherRedemptionException.expired(current.voucherId());
        }
        boolean ledgerVerified = false;
        boolean ledgerUpdated = false;
        Object message = "Voucher redeemed successfully.";
        Optional<VoucherLedgerAdapter> ledgerOptional = this.ledgerAdapterOptional();
        if (request.verifyLedger() && ledgerOptional.isPresent()) {
            try {
                Optional ledgerStatus = ledgerOptional.get().queryStatus(current.voucherId());
                ledgerVerified = true;
                if (ledgerStatus.isPresent()) {
                    VoucherStatus status = (VoucherStatus)ledgerStatus.get();
                    if (status == VoucherStatus.REDEEMED) {
                        return new VoucherService.RedeemVoucherResult(current.voucherId(), "already_redeemed", "Ledger already marked this voucher as redeemed.", true, true);
                    }
                    if (status == VoucherStatus.REVOKED) {
                        throw VoucherRedemptionException.revoked(current.voucherId());
                    }
                    if (status == VoucherStatus.EXPIRED) {
                        throw VoucherRedemptionException.expired(current.voucherId());
                    }
                }
            }
            catch (VoucherRedemptionException e) {
                throw e;
            }
            catch (Exception e) {
                ledgerVerified = false;
                log.warn("voucher_service ledger_verify_failed voucher_id={} reason={}", (Object)current.voucherId(), (Object)e.getMessage());
            }
        }
        StoredVoucher updated = new StoredVoucher(current.voucherId(), current.issuerId(), current.unit(), current.faceValue(), current.expiresAt(), current.memo(), current.issuerSignature(), current.issuerPublicKey(), current.issuedAt(), "redeemed");
        ArrayList<StoredVoucher> updatedVouchers = new ArrayList<StoredVoucher>(state.vouchers());
        updatedVouchers.removeIf(v -> v.voucherId().equals(current.voucherId()));
        updatedVouchers.add(updated);
        this.walletStorage.save(DEFAULT_WALLET_ID, state.withVouchers(updatedVouchers));
        if (ledgerOptional.isPresent()) {
            try {
                ledgerOptional.get().updateStatus(updated.voucherId(), VoucherStatus.REDEEMED);
                ledgerUpdated = true;
            }
            catch (Exception e) {
                log.warn("voucher_service ledger_update_failed voucher_id={} error={}", (Object)updated.voucherId(), (Object)e.getMessage());
                message = "Voucher redeemed locally but ledger update failed: " + e.getMessage();
            }
        }
        return new VoucherService.RedeemVoucherResult(updated.voucherId(), updated.status(), (String)message, ledgerVerified, ledgerUpdated);
    }

    @Override
    public VoucherService.RevokeVoucherResult revokeVoucher(VoucherService.RevokeVoucherRequest request) throws WalletOperationException {
        this.ensureInitialized();
        Objects.requireNonNull(request, "request");
        log.info("voucher_service revoking_voucher voucher_id={} reason={}", (Object)request.voucherId(), (Object)request.reason());
        WalletState state = this.loadWalletState();
        StoredVoucher current = state.vouchers().stream().filter(v -> v.voucherId().equals(request.voucherId())).findFirst().orElseThrow(() -> new IllegalArgumentException("Voucher not found: " + request.voucherId()));
        String previousStatus = current.status();
        if ("revoked".equalsIgnoreCase(previousStatus)) {
            log.debug("voucher_service already_revoked voucher_id={}", (Object)current.voucherId());
            return new VoucherService.RevokeVoucherResult(current.voucherId(), previousStatus, "Voucher was already revoked.", false);
        }
        if (!"issued".equalsIgnoreCase(previousStatus) && !"redeemed".equalsIgnoreCase(previousStatus)) {
            throw new IllegalStateException("Cannot revoke voucher in status: " + previousStatus + ". Only issued or redeemed vouchers can be revoked.");
        }
        StoredVoucher updated = new StoredVoucher(current.voucherId(), current.issuerId(), current.unit(), current.faceValue(), current.expiresAt(), request.reason(), current.issuerSignature(), current.issuerPublicKey(), current.issuedAt(), "revoked");
        ArrayList<StoredVoucher> updatedVouchers = new ArrayList<StoredVoucher>(state.vouchers());
        updatedVouchers.removeIf(v -> v.voucherId().equals(current.voucherId()));
        updatedVouchers.add(updated);
        this.walletStorage.save(DEFAULT_WALLET_ID, state.withVouchers(updatedVouchers));
        log.info("voucher_service voucher_revoked_locally voucher_id={} previous_status={}", (Object)updated.voucherId(), (Object)previousStatus);
        boolean ledgerPublished = false;
        Object message = "Voucher revoked successfully.";
        if (request.publishToLedger()) {
            Optional<VoucherLedgerAdapter> ledgerOptional = this.ledgerAdapterOptional();
            if (ledgerOptional.isPresent()) {
                try {
                    ledgerOptional.get().updateStatus(updated.voucherId(), VoucherStatus.REVOKED);
                    ledgerPublished = true;
                    log.info("voucher_service revocation_published_to_ledger voucher_id={}", (Object)updated.voucherId());
                }
                catch (Exception e) {
                    log.warn("voucher_service ledger_publish_failed voucher_id={} error={}", (Object)updated.voucherId(), (Object)e.getMessage());
                    message = "Voucher revoked locally but ledger publication failed: " + e.getMessage();
                }
            } else {
                log.warn("voucher_service ledger_unavailable voucher_id={} skipping_publish", (Object)updated.voucherId());
                message = "Voucher revoked locally (ledger not available for publication).";
            }
        }
        return new VoucherService.RevokeVoucherResult(updated.voucherId(), previousStatus, (String)message, ledgerPublished);
    }

    @Override
    public VoucherService.VerifyVoucherResult verifyVoucher(VoucherService.VerifyVoucherRequest request) throws WalletOperationException {
        this.ensureInitialized();
        Objects.requireNonNull(request, "request");
        log.info("voucher_service verifying_voucher voucher_id={}", (Object)request.voucherId());
        WalletState state = this.loadWalletState();
        StoredVoucher voucher = state.vouchers().stream().filter(v -> v.voucherId().equals(request.voucherId())).findFirst().orElseThrow(() -> new IllegalArgumentException("Voucher not found: " + request.voucherId()));
        String metadataPayload = String.format("%s:%s:%d:%s:%d", voucher.voucherId(), voucher.unit(), voucher.faceValue(), voucher.issuerId(), voucher.issuedAt().getEpochSecond());
        boolean signatureValid = this.verifyVoucherSignature(metadataPayload, voucher.issuerSignature(), voucher.issuerPublicKey());
        boolean expired = voucher.isExpired();
        String status = voucher.status();
        Object message = !signatureValid ? "Signature verification failed. Voucher may have been tampered with." : (expired ? "Voucher has expired." : (!"issued".equals(status) ? "Voucher status: " + status : "Voucher is valid and can be redeemed."));
        log.info("voucher_service verification_complete voucher_id={} signature_valid={} expired={} status={}", new Object[]{voucher.voucherId(), signatureValid, expired, status});
        return new VoucherService.VerifyVoucherResult(voucher.voucherId(), signatureValid, expired, status, (String)message);
    }

    private String generateVoucherId() {
        byte[] randomBytes = new byte[16];
        new SecureRandom().nextBytes(randomBytes);
        return HexFormat.of().formatHex(randomBytes);
    }

    private String signVoucherMetadata(String payload, String privkeyHex) {
        try {
            byte[] privkeyBytes = HexFormat.of().parseHex(privkeyHex);
            byte[] message = payload.getBytes(StandardCharsets.UTF_8);
            Ed25519PrivateKeyParameters privateKey = new Ed25519PrivateKeyParameters(privkeyBytes, 0);
            Ed25519Signer signer = new Ed25519Signer();
            signer.init(true, (CipherParameters)privateKey);
            signer.update(message, 0, message.length);
            byte[] signature = signer.generateSignature();
            return HexFormat.of().formatHex(signature);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to sign voucher metadata: " + e.getMessage(), e);
        }
    }

    private boolean verifyVoucherSignature(String payload, String signatureHex, String pubkeyHex) {
        try {
            byte[] pubkeyBytes = HexFormat.of().parseHex(pubkeyHex);
            byte[] signatureBytes = HexFormat.of().parseHex(signatureHex);
            byte[] message = payload.getBytes(StandardCharsets.UTF_8);
            Ed25519PublicKeyParameters publicKey = new Ed25519PublicKeyParameters(pubkeyBytes, 0);
            Ed25519Signer verifier = new Ed25519Signer();
            verifier.init(false, (CipherParameters)publicKey);
            verifier.update(message, 0, message.length);
            return verifier.verifySignature(signatureBytes);
        }
        catch (Exception e) {
            log.warn("voucher_service signature_verification_failed error={}", (Object)e.getMessage());
            return false;
        }
    }

    private String createVoucherTokenWithProofs(StoredVoucher voucher, List<ProofRecord> proofs, String mintUrl, String unit) {
        log.debug("voucher_service encoding_token voucher_id={} proof_count={}", (Object)voucher.voucherId(), (Object)proofs.size());
        try {
            List tokenProofs = proofs.stream().map(proof -> new TokenEnvelope.TokenProof(proof.amount(), proof.cHex(), proof.secret(), proof.keysetId())).collect(Collectors.toList());
            TokenEnvelope envelope = new TokenEnvelope(mintUrl, unit, tokenProofs);
            String token = this.tokenCodec.encode(envelope);
            log.debug("voucher_service token_encoded voucher_id={} token_length={}", (Object)voucher.voucherId(), (Object)token.length());
            return token;
        }
        catch (TokenDecodingException e) {
            log.error("voucher_service token_encoding_failed voucher_id={} error={}", new Object[]{voucher.voucherId(), e.getMessage(), e});
            throw new IllegalStateException("Failed to encode voucher token: " + e.getMessage(), e);
        }
    }

    @Override
    public List<StoredVoucher> listVouchers() {
        this.ensureInitialized();
        WalletState state = this.loadWalletState();
        return new ArrayList<StoredVoucher>(state.vouchers());
    }

    @Override
    public Optional<StoredVoucher> getVoucher(String voucherId) {
        this.ensureInitialized();
        Objects.requireNonNull(voucherId, "voucherId");
        WalletState state = this.loadWalletState();
        return state.vouchers().stream().filter(v -> v.voucherId().equals(voucherId)).findFirst();
    }

    @Override
    public void backupToNostr() throws WalletOperationException {
        this.ensureInitialized();
        log.info("voucher_service starting_backup");
        WalletState state = this.loadWalletState();
        List vouchers = state.vouchers();
        if (vouchers.isEmpty()) {
            log.info("voucher_service no_vouchers_to_backup");
            return;
        }
        log.info("voucher_service backing_up voucher_count={}", (Object)vouchers.size());
        try {
            IdentityKey identity = this.identityKeyService.loadOrCreate();
            String ownerPubkeyHex = identity.publicKeyHex();
            String ownerPrivkeyHex = HexFormat.of().formatHex(identity.privateKeyEncoded());
            String voucherJson = this.backupService.serializeVouchers(vouchers);
            log.debug("voucher_service serialized json_length={}", (Object)voucherJson.length());
            String encryptedPayload = this.backupService.encryptBackup(voucherJson, ownerPrivkeyHex, ownerPubkeyHex);
            log.debug("voucher_service encrypted payload_length={}", (Object)encryptedPayload.length());
            String eventId = this.backupService.publishBackup(encryptedPayload, ownerPubkeyHex);
            log.info("voucher_service backup_published event_id={} voucher_count={}", (Object)eventId, (Object)vouchers.size());
            VoucherBackupState updatedBackupState = state.voucherBackupState().withBackup(vouchers.size());
            WalletState updatedState = state.withVoucherBackupState(updatedBackupState);
            this.walletStorage.save(DEFAULT_WALLET_ID, updatedState);
            log.info("voucher_service backup_complete event_id={} voucher_count={}", (Object)eventId, (Object)vouchers.size());
        }
        catch (WalletOperationException e) {
            log.error("voucher_service backup_failed error_code={} retryable={} error={}", new Object[]{e.getErrorCode(), e.isRetryable(), e.getMessage(), e});
            throw e;
        }
        catch (Exception e) {
            log.error("voucher_service backup_failed_unexpected error={}", (Object)e.getMessage(), (Object)e);
            throw new IllegalStateException("Unexpected error during voucher backup: " + e.getMessage(), e);
        }
    }

    @Override
    public int restoreFromNostr() throws WalletOperationException {
        this.ensureInitialized();
        log.info("voucher_service starting_restore");
        try {
            IdentityKey identity = this.identityKeyService.loadOrCreate();
            String ownerPubkeyHex = identity.publicKeyHex();
            String ownerPrivkeyHex = HexFormat.of().formatHex(identity.privateKeyEncoded());
            List<String> encryptedBackups = this.backupService.queryBackups(ownerPubkeyHex, 10);
            if (encryptedBackups.isEmpty()) {
                log.info("voucher_service no_backups_found");
                return 0;
            }
            log.info("voucher_service found_backups backup_count={}", (Object)encryptedBackups.size());
            WalletState state = this.loadWalletState();
            ArrayList<StoredVoucher> currentVouchers = new ArrayList<StoredVoucher>(state.vouchers());
            int restoredCount = 0;
            for (String encryptedPayload : encryptedBackups) {
                try {
                    String voucherJson = this.backupService.decryptBackup(encryptedPayload, ownerPrivkeyHex, ownerPubkeyHex);
                    List<StoredVoucher> backedUpVouchers = this.backupService.deserializeVouchers(voucherJson);
                    for (StoredVoucher voucher : backedUpVouchers) {
                        if (this.isDuplicate(voucher, currentVouchers)) {
                            log.debug("voucher_service skipping_duplicate voucher_id={}", (Object)voucher.voucherId());
                            continue;
                        }
                        currentVouchers.add(voucher);
                        ++restoredCount;
                        log.info("voucher_service restored_voucher voucher_id={} amount={}", (Object)voucher.voucherId(), (Object)voucher.faceValue());
                    }
                }
                catch (WalletOperationException e) {
                    log.warn("voucher_service backup_decrypt_failed error={}", (Object)e.getMessage());
                }
            }
            if (restoredCount > 0) {
                WalletState updatedState = state.withVouchers(currentVouchers);
                this.walletStorage.save(DEFAULT_WALLET_ID, updatedState);
                log.info("voucher_service restore_complete restored_count={}", (Object)restoredCount);
            } else {
                log.info("voucher_service no_new_vouchers_restored");
            }
            return restoredCount;
        }
        catch (WalletOperationException e) {
            log.error("voucher_service restore_failed error_code={} error={}", new Object[]{e.getErrorCode(), e.getMessage(), e});
            throw e;
        }
        catch (Exception e) {
            log.error("voucher_service restore_failed_unexpected error={}", (Object)e.getMessage(), (Object)e);
            throw new IllegalStateException("Failed to restore vouchers from Nostr: " + e.getMessage(), e);
        }
    }

    private boolean isDuplicate(StoredVoucher voucher, List<StoredVoucher> existingVouchers) {
        return existingVouchers.stream().anyMatch(existing -> existing.voucherId().equals(voucher.voucherId()));
    }

    @Override
    public Optional<StoredVoucher> refreshStatus(String voucherId) {
        this.ensureInitialized();
        Objects.requireNonNull(voucherId, "voucherId");
        log.info("voucher_service refreshing_status voucher_id={}", (Object)voucherId);
        try {
            Optional ledgerStatus;
            WalletState state = this.loadWalletState();
            Optional<StoredVoucher> localVoucher = state.vouchers().stream().filter(v -> v.voucherId().equals(voucherId)).findFirst();
            if (localVoucher.isEmpty()) {
                log.warn("voucher_service voucher_not_found_locally voucher_id={}", (Object)voucherId);
                return Optional.empty();
            }
            if (this.nostrGateway == null || this.ledgerAdapter == null) {
                if (this.nostrGateway != null && this.ledgerAdapter == null) {
                    this.ledgerAdapter = new VoucherLedgerAdapter(this.nostrGateway, this.identityKeyService);
                } else {
                    log.warn("voucher_service ledger_not_available voucher_id={} returning_local_status={}", (Object)voucherId, (Object)localVoucher.get().status());
                    return localVoucher;
                }
            }
            if ((ledgerStatus = this.ledgerAdapter.queryStatus(voucherId)).isEmpty()) {
                log.debug("voucher_service no_ledger_status voucher_id={} using_local={}", (Object)voucherId, (Object)localVoucher.get().status());
                return localVoucher;
            }
            String newStatus = ((VoucherStatus)ledgerStatus.get()).name().toLowerCase();
            StoredVoucher current = localVoucher.get();
            if (!current.status().equals(newStatus)) {
                StatusTransitionResult transitionResult = this.validateStatusTransition(current.status(), newStatus, voucherId);
                log.info("voucher_service status_transition voucher_id={} old_status={} new_status={} valid={} suspicious={}", new Object[]{voucherId, current.status(), newStatus, transitionResult.isValid(), transitionResult.isSuspicious()});
                if (transitionResult.isSuspicious()) {
                    DoubleSpendDetectionResult doubleSpend = this.detectDoubleSpendAttempt(voucherId, current, newStatus, transitionResult);
                    if (doubleSpend.isDoubleSpendDetected()) {
                        log.error("voucher_service DOUBLE_SPEND_DETECTED voucher_id={} local_status={} ledger_status={} detection_reason={} risk_level={} recommendation={}", new Object[]{voucherId, current.status(), newStatus, doubleSpend.getDetectionReason(), doubleSpend.getRiskLevel(), doubleSpend.getRecommendation()});
                    } else {
                        log.warn("voucher_service suspicious_transition voucher_id={} old_status={} new_status={} reason={}", new Object[]{voucherId, current.status(), newStatus, transitionResult.getReason()});
                    }
                }
                StoredVoucher updated = new StoredVoucher(current.voucherId(), current.issuerId(), current.unit(), current.faceValue(), current.expiresAt(), current.memo(), current.issuerSignature(), current.issuerPublicKey(), current.issuedAt(), newStatus);
                ArrayList<StoredVoucher> updatedVouchers = new ArrayList<StoredVoucher>(state.vouchers());
                updatedVouchers.removeIf(v -> v.voucherId().equals(voucherId));
                updatedVouchers.add(updated);
                WalletState updatedState = state.withVouchers(updatedVouchers);
                this.walletStorage.save(DEFAULT_WALLET_ID, updatedState);
                log.info("voucher_service status_synchronized voucher_id={} old_status={} new_status={} source=ledger", new Object[]{voucherId, current.status(), newStatus});
                return Optional.of(updated);
            }
            log.debug("voucher_service status_unchanged voucher_id={} status={} source=ledger", (Object)voucherId, (Object)current.status());
            return Optional.of(current);
        }
        catch (Exception e) {
            log.error("voucher_service refresh_failed voucher_id={} error={}", new Object[]{voucherId, e.getMessage(), e});
            throw new RuntimeException("Failed to refresh voucher status: " + e.getMessage(), e);
        }
    }

    @Override
    public void importVoucher(StoredVoucher voucher) throws WalletOperationException {
        this.ensureInitialized();
        Objects.requireNonNull(voucher, "voucher");
        log.debug("voucher_service import_voucher_start voucher_id={}", (Object)voucher.voucherId());
        try {
            WalletState state = this.loadWalletState();
            ArrayList<StoredVoucher> currentVouchers = new ArrayList<StoredVoucher>(state.vouchers());
            boolean exists = currentVouchers.stream().anyMatch(v -> v.voucherId().equals(voucher.voucherId()));
            if (exists) {
                log.warn("voucher_service voucher_already_exists voucher_id={} outcome=skipped", (Object)voucher.voucherId());
                return;
            }
            currentVouchers.add(voucher);
            WalletState updatedState = state.withVouchers(currentVouchers);
            this.walletStorage.save(DEFAULT_WALLET_ID, updatedState);
            log.info("voucher_service voucher_imported voucher_id={} amount={} unit={}", new Object[]{voucher.voucherId(), voucher.faceValue(), voucher.unit()});
        }
        catch (Exception e) {
            log.error("voucher_service import_failed voucher_id={} error={}", new Object[]{voucher.voucherId(), e.getMessage(), e});
            throw VoucherImportException.forStorageFailure(voucher.voucherId(), e);
        }
    }

    private void ensureInitialized() {
        if (this.config == null) {
            throw new IllegalStateException("VoucherService not initialized. Call init() first.");
        }
    }

    private boolean isValidUnit(String unit) {
        if (unit == null || unit.trim().isEmpty()) {
            return false;
        }
        String normalized = unit.trim().toLowerCase();
        if (Set.of("sat", "msat", "btc").contains(normalized)) {
            return true;
        }
        if (unit.length() == 3) {
            try {
                Currency.getInstance(unit.toUpperCase());
                return true;
            }
            catch (IllegalArgumentException e) {
                return false;
            }
        }
        return false;
    }

    private WalletState loadWalletState() {
        return this.walletStorage.load(DEFAULT_WALLET_ID).orElseGet(this::createDefaultState);
    }

    private WalletState createDefaultState() {
        WalletSchemaMetadata schema = this.walletStorage.load(DEFAULT_WALLET_ID).map(WalletState::schema).orElse(null);
        if (schema == null) {
            throw new IllegalStateException("Cannot load wallet state. Wallet may not be initialized.");
        }
        WalletState defaultState = new WalletState(schema, Instant.now(), List.of(), List.of(), List.of());
        this.walletStorage.save(DEFAULT_WALLET_ID, defaultState);
        return defaultState;
    }

    private Optional<VoucherLedgerAdapter> ledgerAdapterOptional() {
        if (this.nostrGateway == null) {
            return Optional.empty();
        }
        if (this.ledgerAdapter == null) {
            this.ledgerAdapter = new VoucherLedgerAdapter(this.nostrGateway, this.identityKeyService);
        }
        return Optional.of(this.ledgerAdapter);
    }

    private StatusTransitionResult validateStatusTransition(String oldStatus, String newStatus, String voucherId) {
        String to;
        String from = oldStatus.toLowerCase();
        if (from.equals(to = newStatus.toLowerCase())) {
            return new StatusTransitionResult(true, false, "no_change");
        }
        if (from.equals("issued") && (to.equals("redeemed") || to.equals("revoked") || to.equals("expired"))) {
            return new StatusTransitionResult(true, false, "valid_forward_transition");
        }
        if (from.equals("redeemed") && to.equals("revoked")) {
            return new StatusTransitionResult(true, false, "valid_refund_transition");
        }
        if (to.equals("issued")) {
            if (from.equals("redeemed")) {
                return new StatusTransitionResult(false, true, "reverse_redemption_detected_possible_double_spend");
            }
            if (from.equals("revoked")) {
                return new StatusTransitionResult(false, true, "reverse_revocation_detected_possible_tampering");
            }
            if (from.equals("expired")) {
                return new StatusTransitionResult(false, true, "reverse_expiry_detected_possible_tampering");
            }
        }
        if (from.equals("redeemed") && to.equals("expired")) {
            return new StatusTransitionResult(false, true, "redeemed_to_expired_invalid_transition");
        }
        return new StatusTransitionResult(true, true, "unknown_transition_" + from + "_to_" + to);
    }

    private DoubleSpendDetectionResult detectDoubleSpendAttempt(String voucherId, StoredVoucher currentVoucher, String newStatus, StatusTransitionResult transitionResult) {
        String oldStatus = currentVoucher.status();
        String reason = transitionResult.getReason();
        if (oldStatus.equals("redeemed") && newStatus.equals("issued")) {
            return new DoubleSpendDetectionResult(true, "DOUBLE_SPEND_ATTEMPT", "HIGH", "Voucher was previously redeemed but ledger now shows issued status. This indicates a possible double-spend attempt where someone is trying to redeem the voucher again after it was already spent.", "DO NOT accept this voucher. Contact the issuer to verify. The voucher was already redeemed and should remain in redeemed status.");
        }
        if (oldStatus.equals("revoked") && newStatus.equals("issued")) {
            return new DoubleSpendDetectionResult(true, "LEDGER_TAMPERING", "HIGH", "Voucher was revoked by issuer but ledger now shows issued status. This indicates possible ledger manipulation or conflicting events.", "DO NOT accept this voucher. The issuer revoked it and it should remain revoked. Contact the issuer to verify the voucher's legitimacy.");
        }
        if (oldStatus.equals("expired") && newStatus.equals("issued")) {
            return new DoubleSpendDetectionResult(true, "EXPIRY_REVERSAL", "MEDIUM", "Voucher was expired but ledger now shows issued status. This may indicate ledger manipulation or clock skew issues.", "Verify the voucher's expiry date. If it genuinely expired, do not accept it. This could be an attempt to revive an expired voucher.");
        }
        if (oldStatus.equals("redeemed") && newStatus.equals("expired")) {
            return new DoubleSpendDetectionResult(false, "INVALID_TRANSITION", "LOW", "Ledger shows transition from redeemed to expired, which is invalid. A voucher cannot expire after being redeemed.", "Keep local status as 'redeemed'. This transition is logically invalid and should be ignored.");
        }
        return new DoubleSpendDetectionResult(false, reason.toUpperCase(), "MEDIUM", "Detected suspicious status transition that doesn't follow normal voucher lifecycle.", "Review the voucher history and verify with the issuer if unsure.");
    }

    private static class StatusTransitionResult {
        private final boolean valid;
        private final boolean suspicious;
        private final String reason;

        public StatusTransitionResult(boolean valid, boolean suspicious, String reason) {
            this.valid = valid;
            this.suspicious = suspicious;
            this.reason = reason;
        }

        public boolean isValid() {
            return this.valid;
        }

        public boolean isSuspicious() {
            return this.suspicious;
        }

        public String getReason() {
            return this.reason;
        }
    }

    private static class DoubleSpendDetectionResult {
        private final boolean doubleSpendDetected;
        private final String detectionReason;
        private final String riskLevel;
        private final String forensicDetails;
        private final String recommendation;

        public DoubleSpendDetectionResult(boolean doubleSpendDetected, String detectionReason, String riskLevel, String forensicDetails, String recommendation) {
            this.doubleSpendDetected = doubleSpendDetected;
            this.detectionReason = detectionReason;
            this.riskLevel = riskLevel;
            this.forensicDetails = forensicDetails;
            this.recommendation = recommendation;
        }

        public boolean isDoubleSpendDetected() {
            return this.doubleSpendDetected;
        }

        public String getDetectionReason() {
            return this.detectionReason;
        }

        public String getRiskLevel() {
            return this.riskLevel;
        }

        public String getForensicDetails() {
            return this.forensicDetails;
        }

        public String getRecommendation() {
            return this.recommendation;
        }
    }
}

