/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.domain;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import xyz.tcheeric.wallet.core.domain.events.DomainEvent;
import xyz.tcheeric.wallet.core.domain.events.ProofCommittedEvent;
import xyz.tcheeric.wallet.core.domain.events.ProofImportedEvent;
import xyz.tcheeric.wallet.core.domain.events.ProofReservedEvent;
import xyz.tcheeric.wallet.core.proof.NewProof;
import xyz.tcheeric.wallet.core.proof.ProofRecord;

public class WalletAggregate {
    private final String walletId;
    private final String mintUrl;
    private final String unit;
    private long version;
    private final Map<Long, ProofValueObject> availableProofs;
    private final Map<String, ProofReservation> activeReservations;
    private final Set<String> importedSignatures;
    private final List<DomainEvent> uncommittedEvents;

    public WalletAggregate(String walletId, String mintUrl, String unit, long version) {
        this.walletId = Objects.requireNonNull(walletId, "walletId");
        this.mintUrl = Objects.requireNonNull(mintUrl, "mintUrl");
        this.unit = Objects.requireNonNull(unit, "unit");
        this.version = version;
        this.availableProofs = new HashMap<Long, ProofValueObject>();
        this.activeReservations = new HashMap<String, ProofReservation>();
        this.importedSignatures = new HashSet<String>();
        this.uncommittedEvents = new ArrayList<DomainEvent>();
    }

    public void loadProofs(List<ProofRecord> proofs) {
        for (ProofRecord proof : proofs) {
            ProofValueObject vo = new ProofValueObject(proof.id(), proof.amount(), proof.cHex(), proof.secret(), proof.keysetId());
            this.availableProofs.put(proof.id(), vo);
            this.importedSignatures.add(proof.cHex());
        }
    }

    public ProofReservation reserveProofs(long amount, String recipient) {
        if (amount <= 0L) {
            throw new IllegalArgumentException("Amount must be positive");
        }
        ArrayList<ProofValueObject> selected = new ArrayList<ProofValueObject>();
        long total = 0L;
        ArrayList<ProofValueObject> sorted = new ArrayList<ProofValueObject>(this.availableProofs.values());
        sorted.sort((a, b) -> Integer.compare(b.amount(), a.amount()));
        for (ProofValueObject proof : sorted) {
            if (this.isReserved(proof.id())) continue;
            selected.add(proof);
            if ((total += (long)proof.amount()) < amount) continue;
            break;
        }
        if (total < amount) {
            throw new InsufficientProofsException(String.format("Requested %d %s but only %d %s available", amount, this.unit, total, this.unit));
        }
        String reservationId = UUID.randomUUID().toString();
        ProofReservation reservation = new ProofReservation(reservationId, selected, total, recipient, Instant.now());
        this.activeReservations.put(reservationId, reservation);
        this.uncommittedEvents.add((DomainEvent)new ProofReservedEvent(this.mintUrl, this.unit, selected.size(), total, recipient));
        return reservation;
    }

    public void commitReservation(String reservationId) {
        ProofReservation reservation = this.activeReservations.remove(reservationId);
        if (reservation == null) {
            throw new ReservationNotFoundException("Reservation not found: " + reservationId);
        }
        for (ProofValueObject proof : reservation.proofs()) {
            this.availableProofs.remove(proof.id());
        }
        this.uncommittedEvents.add((DomainEvent)new ProofCommittedEvent(this.mintUrl, this.unit, reservation.proofs().size(), reservation.totalAmount(), reservationId));
    }

    public void cancelReservation(String reservationId) {
        this.activeReservations.remove(reservationId);
    }

    public ImportResult importProofs(List<NewProof> newProofs) {
        int imported = 0;
        int duplicates = 0;
        ArrayList<Long> importedIds = new ArrayList<Long>();
        for (NewProof newProof : newProofs) {
            String signature = newProof.cHex();
            if (this.importedSignatures.contains(signature)) {
                ++duplicates;
                continue;
            }
            long tempId = -(imported + 1);
            ProofValueObject vo = new ProofValueObject(tempId, newProof.amount(), newProof.cHex(), newProof.secret(), newProof.keysetId());
            this.availableProofs.put(tempId, vo);
            this.importedSignatures.add(signature);
            importedIds.add(tempId);
            ++imported;
        }
        if (imported > 0) {
            this.uncommittedEvents.add((DomainEvent)new ProofImportedEvent(this.mintUrl, this.unit, imported, duplicates));
        }
        return new ImportResult(imported, duplicates, importedIds);
    }

    public List<DomainEvent> collectUncommittedEvents() {
        List<DomainEvent> events = List.copyOf(this.uncommittedEvents);
        this.uncommittedEvents.clear();
        return events;
    }

    public List<DomainEvent> getUncommittedEvents() {
        return Collections.unmodifiableList(this.uncommittedEvents);
    }

    public void incrementVersion() {
        ++this.version;
    }

    public String getWalletId() {
        return this.walletId;
    }

    public String getMintUrl() {
        return this.mintUrl;
    }

    public String getUnit() {
        return this.unit;
    }

    public long getVersion() {
        return this.version;
    }

    public int getAvailableProofCount() {
        return this.availableProofs.size() - this.countReservedProofs();
    }

    public long getTotalAvailableAmount() {
        return this.availableProofs.values().stream().filter(proof -> !this.isReserved(proof.id())).mapToLong(ProofValueObject::amount).sum();
    }

    public List<ProofValueObject> getAvailableProofs() {
        return this.availableProofs.values().stream().filter(proof -> !this.isReserved(proof.id())).toList();
    }

    public Optional<ProofReservation> getReservation(String reservationId) {
        return Optional.ofNullable(this.activeReservations.get(reservationId));
    }

    private boolean isReserved(long proofId) {
        return this.activeReservations.values().stream().anyMatch(res -> res.containsProof(proofId));
    }

    private int countReservedProofs() {
        HashSet<Long> reservedIds = new HashSet<Long>();
        for (ProofReservation reservation : this.activeReservations.values()) {
            for (ProofValueObject proof : reservation.proofs()) {
                reservedIds.add(proof.id());
            }
        }
        return reservedIds.size();
    }

    public record ProofValueObject(long id, int amount, String cHex, byte[] secret, String keysetId) {
        private final byte[] secret;

        public ProofValueObject {
            Objects.requireNonNull(cHex, "cHex");
            Objects.requireNonNull(secret, "secret");
            Objects.requireNonNull(keysetId, "keysetId");
            secret = (byte[])secret.clone();
        }

        public byte[] secret() {
            return (byte[])this.secret.clone();
        }

        public ProofRecord toRecord() {
            return new ProofRecord(this.id, null, null, this.amount, this.cHex, this.secret, this.keysetId);
        }
    }

    public static class InsufficientProofsException
    extends RuntimeException {
        public InsufficientProofsException(String message) {
            super(message);
        }
    }

    public record ProofReservation(String reservationId, List<ProofValueObject> proofs, long totalAmount, String recipient, Instant reservedAt) {
        public ProofReservation {
            Objects.requireNonNull(reservationId, "reservationId");
            Objects.requireNonNull(proofs, "proofs");
            proofs = List.copyOf(proofs);
            Objects.requireNonNull(recipient, "recipient");
            Objects.requireNonNull(reservedAt, "reservedAt");
        }

        public boolean containsProof(long proofId) {
            return this.proofs.stream().anyMatch(p -> p.id() == proofId);
        }

        public List<ProofRecord> toProofRecords(String mintUrl, String unit) {
            return this.proofs.stream().map(vo -> new ProofRecord(vo.id(), mintUrl, unit, vo.amount(), vo.cHex(), vo.secret(), vo.keysetId())).toList();
        }
    }

    public static class ReservationNotFoundException
    extends RuntimeException {
        public ReservationNotFoundException(String message) {
            super(message);
        }
    }

    public record ImportResult(int importedCount, int duplicateCount, List<Long> importedIds) {
        public ImportResult {
            importedIds = List.copyOf(importedIds);
        }

        public boolean hasNewProofs() {
            return this.importedCount > 0;
        }
    }
}

