/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.security;

import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.SecureRandom;
import javax.crypto.Cipher;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import xyz.tcheeric.wallet.core.security.EncryptionException;
import xyz.tcheeric.wallet.core.security.EncryptionMetadataRepository;
import xyz.tcheeric.wallet.core.security.KeyDerivationException;
import xyz.tcheeric.wallet.core.security.KeyDerivationService;

public class EncryptionService {
    private static final String ENC_VERSION = "1";
    private static final String KDF_NAME = "pbkdf2";
    private static final int KDF_ITER = 200000;
    private static final int SALT_LEN = 16;
    private static final int GCM_IV_LEN = 12;
    private static final int GCM_TAG_BITS = 128;
    private final SecureRandom rng = new SecureRandom();
    private final EncryptionMetadataRepository repo = new EncryptionMetadataRepository();
    private final KeyDerivationService kdfService = new KeyDerivationService();
    private EncryptionMetadataRepository.EncryptionMetadata metadata = new EncryptionMetadataRepository.EncryptionMetadata(false, null, null, 0, null, null, null);
    private byte[] dek;
    private boolean enabled;
    private String kdfName = "pbkdf2";
    private int iter = 200000;

    public void load() {
        this.metadata = this.repo.load();
        this.enabled = this.metadata.enabled();
        this.kdfName = this.metadata.kdf() != null ? this.metadata.kdf() : KDF_NAME;
        this.iter = this.metadata.kdfIter() > 0 ? this.metadata.kdfIter() : 200000;
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public void initializeAndStore(String passphrase) throws EncryptionException {
        if (this.enabled) {
            return;
        }
        try {
            byte[] salt = new byte[16];
            this.rng.nextBytes(salt);
            byte[] kek = this.kdfService.derive(this.kdfName, passphrase.toCharArray(), salt, this.iter);
            byte[] newDek = new byte[32];
            this.rng.nextBytes(newDek);
            byte[] dekIv = new byte[12];
            this.rng.nextBytes(dekIv);
            byte[] wrapped = EncryptionService.aesGcmEncrypt(kek, dekIv, newDek);
            this.metadata = new EncryptionMetadataRepository.EncryptionMetadata(true, ENC_VERSION, this.kdfName, this.iter, salt, dekIv, wrapped);
            this.repo.save(this.metadata);
            this.enabled = true;
            this.dek = newDek;
        }
        catch (RuntimeException | EncryptionException | KeyDerivationException e) {
            throw new EncryptionException(e.getMessage(), e);
        }
    }

    public void changePassphrase(String current, String next) throws EncryptionException {
        if (!this.enabled) {
            throw new IllegalStateException("Encryption not enabled");
        }
        try {
            String currentKdf = this.metadata.kdf() != null ? this.metadata.kdf() : KDF_NAME;
            int currentIter = this.metadata.kdfIter() > 0 ? this.metadata.kdfIter() : this.iter;
            byte[] kek = this.kdfService.derive(currentKdf, current.toCharArray(), this.metadata.salt(), currentIter);
            byte[] dek = EncryptionService.aesGcmDecrypt(kek, this.metadata.dekIv(), this.metadata.dekCt());
            byte[] salt2 = new byte[16];
            this.rng.nextBytes(salt2);
            byte[] dekIv2 = new byte[12];
            this.rng.nextBytes(dekIv2);
            byte[] kek2 = this.kdfService.derive(this.kdfName, next.toCharArray(), salt2, this.iter);
            byte[] wrapped2 = EncryptionService.aesGcmEncrypt(kek2, dekIv2, dek);
            this.metadata = new EncryptionMetadataRepository.EncryptionMetadata(true, ENC_VERSION, this.kdfName, this.iter, salt2, dekIv2, wrapped2);
            this.repo.save(this.metadata);
            this.dek = dek;
        }
        catch (RuntimeException | EncryptionException | KeyDerivationException e) {
            throw new EncryptionException(e.getMessage(), e);
        }
    }

    public void changePassphraseWithKdf(String current, String next, String newKdf) throws EncryptionException {
        if (newKdf != null && !newKdf.isBlank()) {
            String selectedKdf = newKdf.trim().toLowerCase();
            if (!selectedKdf.equals(KDF_NAME) && !selectedKdf.equals("argon2id")) {
                throw new IllegalArgumentException("Unsupported KDF: " + newKdf);
            }
            this.kdfName = selectedKdf;
        }
        this.changePassphrase(current, next);
    }

    public void unlock(String passphrase) throws EncryptionException {
        if (!this.enabled) {
            return;
        }
        try {
            String currentKdf = this.metadata.kdf() != null ? this.metadata.kdf() : KDF_NAME;
            int currentIter = this.metadata.kdfIter() > 0 ? this.metadata.kdfIter() : this.iter;
            byte[] kek = this.kdfService.derive(currentKdf, passphrase.toCharArray(), this.metadata.salt(), currentIter);
            this.dek = EncryptionService.aesGcmDecrypt(kek, this.metadata.dekIv(), this.metadata.dekCt());
        }
        catch (RuntimeException | EncryptionException | KeyDerivationException e) {
            throw new EncryptionException(e.getMessage(), e);
        }
    }

    public byte[] encrypt(byte[] plaintext) throws EncryptionException {
        byte[] ct;
        if (!this.enabled) {
            return plaintext;
        }
        if (this.dek == null) {
            throw new IllegalStateException("Wallet locked: no DEK");
        }
        byte[] iv = new byte[12];
        this.rng.nextBytes(iv);
        try {
            ct = EncryptionService.aesGcmEncrypt(this.dek, iv, plaintext);
        }
        catch (EncryptionException e) {
            throw new EncryptionException("Failed to encrypt data", e);
        }
        byte[] out = new byte[1 + iv.length + ct.length];
        out[0] = 49;
        System.arraycopy(iv, 0, out, 1, iv.length);
        System.arraycopy(ct, 0, out, 1 + iv.length, ct.length);
        return out;
    }

    public byte[] decrypt(byte[] blob) throws EncryptionException {
        byte ver;
        if (!this.enabled) {
            return blob;
        }
        if (this.dek == null) {
            throw new IllegalStateException("Wallet locked: no DEK");
        }
        if (blob == null || blob.length < 29) {
            throw new IllegalArgumentException("Invalid ciphertext");
        }
        int pos = 0;
        if ((ver = blob[pos++]) != 49) {
            throw new IllegalArgumentException("Unsupported ciphertext version: " + (char)ver);
        }
        byte[] iv = new byte[12];
        System.arraycopy(blob, pos, iv, 0, 12);
        byte[] ct = new byte[blob.length - (pos += 12)];
        System.arraycopy(blob, pos, ct, 0, ct.length);
        try {
            return EncryptionService.aesGcmDecrypt(this.dek, iv, ct);
        }
        catch (EncryptionException e) {
            throw new EncryptionException("Failed to decrypt data", e);
        }
    }

    private static byte[] aesGcmEncrypt(byte[] key, byte[] iv, byte[] plaintext) throws EncryptionException {
        try {
            Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
            cipher.init(1, (Key)new SecretKeySpec(key, "AES"), new GCMParameterSpec(128, iv));
            return cipher.doFinal(plaintext);
        }
        catch (GeneralSecurityException e) {
            throw new EncryptionException("AES-GCM encryption failed", e);
        }
    }

    private static byte[] aesGcmDecrypt(byte[] key, byte[] iv, byte[] ciphertext) throws EncryptionException {
        try {
            Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
            cipher.init(2, (Key)new SecretKeySpec(key, "AES"), new GCMParameterSpec(128, iv));
            return cipher.doFinal(ciphertext);
        }
        catch (GeneralSecurityException e) {
            throw new EncryptionException("AES-GCM decryption failed", e);
        }
    }
}

