/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.nostr;

import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.util.HexFormat;
import java.util.List;
import java.util.Locale;
import org.bouncycastle.asn1.sec.SECNamedCurves;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.math.ec.ECPoint;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import xyz.tcheeric.wallet.core.nostr.NostrEvent;
import xyz.tcheeric.wallet.core.nostr.NostrEventVerifier;

class NostrEventVerifierTest {
    private static final HexFormat HEX = HexFormat.of();
    private static final X9ECParameters CURVE_PARAMS = SECNamedCurves.getByName((String)"secp256k1");
    private static final ECDomainParameters DOMAIN = new ECDomainParameters(CURVE_PARAMS.getCurve(), CURVE_PARAMS.getG(), CURVE_PARAMS.getN(), CURVE_PARAMS.getH());
    private static final BigInteger GROUP_ORDER = DOMAIN.getN();
    private static final byte[] TAG_AUX = NostrEventVerifierTest.hashTag("BIP0340/aux");
    private static final byte[] TAG_NONCE = NostrEventVerifierTest.hashTag("BIP0340/nonce");
    private static final byte[] TAG_CHALLENGE = NostrEventVerifierTest.hashTag("BIP0340/challenge");
    private static final String TEST_SECRET_KEY = "0000000000000000000000000000000000000000000000000000000000000003";
    private static final String TEST_AUX = "0000000000000000000000000000000000000000000000000000000000000000";
    private static final Instant TEST_CREATED_AT = Instant.parse("2024-01-01T00:00:00Z");
    private static final List<List<String>> TEST_TAGS = List.of(List.of("p", "npub1example"));
    private static final int TEST_KIND = 1;
    private static final String TEST_CONTENT = "hello nostr";
    private static final String TEST_PUBKEY = NostrEventVerifierTest.derivePublicKeyHex("0000000000000000000000000000000000000000000000000000000000000003");
    private final NostrEventVerifier verifier = new NostrEventVerifier();

    NostrEventVerifierTest() {
    }

    @Test
    void verifiesValidSignatureFromBip340Vector() {
        NostrEvent signedEvent = NostrEventVerifierTest.createSignedEvent();
        NostrEventVerifier.VerificationResult result = this.verifier.verify(signedEvent, signedEvent.sig());
        Assertions.assertTrue((boolean)result.isValid());
        Assertions.assertNull((Object)result.getFailureReason());
    }

    @Test
    void rejectsInvalidSignature() {
        NostrEvent signedEvent = NostrEventVerifierTest.createSignedEvent();
        String invalidSignature = NostrEventVerifierTest.mutateSignature(signedEvent.sig());
        NostrEvent tamperedSignatureEvent = new NostrEvent(signedEvent.id(), signedEvent.pubkey(), signedEvent.kind(), signedEvent.content(), signedEvent.createdAt(), signedEvent.tags(), invalidSignature);
        NostrEventVerifier.VerificationResult result = this.verifier.verify(tamperedSignatureEvent, invalidSignature);
        Assertions.assertFalse((boolean)result.isValid());
        Assertions.assertEquals((Object)"INVALID_SIGNATURE", (Object)result.getFailureReason());
    }

    @Test
    void rejectsWhenSignatureMissing() {
        NostrEvent event = new NostrEvent("0000000000000000000000000000000000000000000000000000000000000001", TEST_PUBKEY, 1, TEST_CONTENT, TEST_CREATED_AT, TEST_TAGS, null);
        NostrEventVerifier.VerificationResult result = this.verifier.verify(event, null);
        Assertions.assertFalse((boolean)result.isValid());
        Assertions.assertEquals((Object)"MISSING_SIGNATURE", (Object)result.getFailureReason());
    }

    @Test
    void rejectsWhenEventIdDoesNotMatchCanonicalSerialization() {
        String canonicalId = NostrEvent.generateDeterministicId((String)TEST_PUBKEY, (int)1, (String)TEST_CONTENT, TEST_TAGS, (Instant)TEST_CREATED_AT);
        NostrEvent tampered = new NostrEvent(canonicalId, TEST_PUBKEY, 1, "tampered", TEST_CREATED_AT, TEST_TAGS, NostrEventVerifierTest.createSignedEvent().sig());
        NostrEventVerifier.VerificationResult result = this.verifier.verify(tampered, tampered.sig());
        Assertions.assertFalse((boolean)result.isValid());
        Assertions.assertEquals((Object)"EVENT_ID_MISMATCH", (Object)result.getFailureReason());
    }

    private static NostrEvent createSignedEvent() {
        NostrEvent unsignedEvent = NostrEvent.unsigned((String)TEST_PUBKEY, (int)1, (String)TEST_CONTENT, (Instant)TEST_CREATED_AT, TEST_TAGS);
        String signature = NostrEventVerifierTest.signEvent(TEST_SECRET_KEY, TEST_AUX, unsignedEvent);
        return new NostrEvent(unsignedEvent.id(), unsignedEvent.pubkey(), unsignedEvent.kind(), unsignedEvent.content(), unsignedEvent.createdAt(), unsignedEvent.tags(), signature);
    }

    private static String mutateSignature(String signature) {
        char first = signature.charAt(0);
        char replacement = first == 'A' ? (char)'B' : 'A';
        return replacement + signature.substring(1);
    }

    private static String derivePublicKeyHex(String secretKeyHex) {
        BigInteger secret = new BigInteger(1, HEX.parseHex(secretKeyHex));
        AdjustedKey adjusted = NostrEventVerifierTest.adjustSecret(secret);
        byte[] xCoord = NostrEventVerifierTest.toFixedLength(adjusted.point().getAffineXCoord().toBigInteger(), 32);
        return HEX.formatHex(xCoord).toUpperCase(Locale.ROOT);
    }

    private static String signEvent(String secretKeyHex, String auxHex, NostrEvent event) {
        BigInteger secret = new BigInteger(1, HEX.parseHex(secretKeyHex));
        if (secret.signum() == 0 || secret.compareTo(GROUP_ORDER) >= 0) {
            throw new IllegalArgumentException("Secret key must be in range 1..n-1");
        }
        byte[] aux = HEX.parseHex(auxHex);
        if (aux.length != 32) {
            throw new IllegalArgumentException("Auxiliary randomness must be 32 bytes");
        }
        AdjustedKey adjusted = NostrEventVerifierTest.adjustSecret(secret);
        ECPoint publicPoint = adjusted.point();
        BigInteger adjustedSecret = adjusted.secret();
        String canonicalId = NostrEvent.generateDeterministicId((String)event.pubkey(), (int)event.kind(), (String)event.content(), (List)event.tags(), (Instant)event.createdAt());
        byte[] message = HEX.parseHex(canonicalId);
        byte[] t = NostrEventVerifierTest.toFixedLength(adjustedSecret, 32);
        byte[] rand = NostrEventVerifierTest.taggedHash(TAG_AUX, new byte[][]{aux});
        for (int i = 0; i < rand.length; ++i) {
            int n = i;
            rand[n] = (byte)(rand[n] ^ t[i]);
        }
        byte[] px = NostrEventVerifierTest.toFixedLength(publicPoint.getAffineXCoord().toBigInteger(), 32);
        BigInteger k0 = new BigInteger(1, NostrEventVerifierTest.taggedHash(TAG_NONCE, rand, px, message)).mod(GROUP_ORDER);
        if (k0.signum() == 0) {
            throw new IllegalStateException("Nonce generation failed");
        }
        ECPoint R = DOMAIN.getG().multiply(k0).normalize();
        if (R.isInfinity()) {
            throw new IllegalStateException("Nonce point is at infinity");
        }
        BigInteger k = R.getAffineYCoord().toBigInteger().testBit(0) ? GROUP_ORDER.subtract(k0) : k0;
        byte[] rx = NostrEventVerifierTest.toFixedLength(R.getAffineXCoord().toBigInteger(), 32);
        BigInteger e = new BigInteger(1, NostrEventVerifierTest.taggedHash(TAG_CHALLENGE, rx, px, message)).mod(GROUP_ORDER);
        BigInteger s = k.add(e.multiply(adjustedSecret)).mod(GROUP_ORDER);
        byte[] signature = new byte[64];
        System.arraycopy(rx, 0, signature, 0, 32);
        System.arraycopy(NostrEventVerifierTest.toFixedLength(s, 32), 0, signature, 32, 32);
        return HEX.formatHex(signature).toUpperCase(Locale.ROOT);
    }

    private static AdjustedKey adjustSecret(BigInteger secret) {
        ECPoint point = DOMAIN.getG().multiply(secret).normalize();
        if (point.getAffineYCoord().toBigInteger().testBit(0)) {
            BigInteger adjustedSecret = GROUP_ORDER.subtract(secret);
            ECPoint adjustedPoint = DOMAIN.getG().multiply(adjustedSecret).normalize();
            return new AdjustedKey(adjustedPoint, adjustedSecret);
        }
        return new AdjustedKey(point, secret);
    }

    private static byte[] taggedHash(byte[] tagHash, byte[] ... data) {
        MessageDigest digest = NostrEventVerifierTest.newDigest();
        digest.update(tagHash);
        digest.update(tagHash);
        for (byte[] part : data) {
            digest.update(part);
        }
        return digest.digest();
    }

    private static byte[] hashTag(String tag) {
        return NostrEventVerifierTest.sha256(tag.getBytes(StandardCharsets.US_ASCII));
    }

    private static byte[] sha256(byte[] input) {
        return NostrEventVerifierTest.newDigest().digest(input);
    }

    private static MessageDigest newDigest() {
        try {
            return MessageDigest.getInstance("SHA-256");
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("SHA-256 digest not available", e);
        }
    }

    private static byte[] toFixedLength(BigInteger value, int length) {
        byte[] bytes = value.toByteArray();
        if (bytes.length == length) {
            return bytes;
        }
        byte[] result = new byte[length];
        int srcPos = Math.max(0, bytes.length - length);
        int destPos = Math.max(0, length - bytes.length);
        int copyLength = Math.min(length, bytes.length);
        System.arraycopy(bytes, srcPos, result, destPos, copyLength);
        return result;
    }

    private record AdjustedKey(ECPoint point, BigInteger secret) {
    }
}

