/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.nostr;

import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import xyz.tcheeric.wallet.core.nostr.InMemoryNostrRelayClient;
import xyz.tcheeric.wallet.core.nostr.NostrEvent;
import xyz.tcheeric.wallet.core.nostr.NostrGatewayConfig;
import xyz.tcheeric.wallet.core.nostr.NostrGatewayService;
import xyz.tcheeric.wallet.core.nostr.NostrRelayClient;
import xyz.tcheeric.wallet.core.nostr.NostrRelayClientFactory;
import xyz.tcheeric.wallet.core.nostr.NostrRelayOption;
import xyz.tcheeric.wallet.core.nostr.NostrSubscription;
import xyz.tcheeric.wallet.core.nostr.filter.NostrFilterBuilder;
import xyz.tcheeric.wallet.core.nostr.filter.NostrServerSideFilter;
import xyz.tcheeric.wallet.core.nostr.relay.RelaySelectionPolicy;
import xyz.tcheeric.wallet.core.nostr.relay.RelaySelectionPolicyType;
import xyz.tcheeric.wallet.core.security.IdentityKey;
import xyz.tcheeric.wallet.core.security.IdentityKeyService;
import xyz.tcheeric.wallet.core.security.SecureKeyStore;
import xyz.tcheeric.wallet.core.security.WalletKeyManager;
import xyz.tcheeric.wallet.core.security.WalletSigningKey;

class NostrGatewayServiceTest {
    @TempDir
    Path tempDir;

    NostrGatewayServiceTest() {
    }

    @Test
    void enablesAuthOnlyWhenRequired() {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-auth.example", true), new NostrRelayOption("wss://relay-open.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        gateway.start();
        Map authStates = gateway.relayAuthStates();
        Assertions.assertEquals((int)2, (int)authStates.size());
        Assertions.assertTrue((boolean)((Boolean)authStates.get("wss://relay-auth.example")));
        Assertions.assertFalse((boolean)((Boolean)authStates.get("wss://relay-open.example")));
    }

    @Test
    void publishesToSelectedRelays() {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-one.example", false), new NostrRelayOption("wss://relay-two.example", false), new NostrRelayOption("wss://relay-three.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        NostrEvent event = new NostrEvent(UUID.randomUUID().toString(), "pub", 1, "hello", Instant.now(), List.of(), null);
        gateway.publish(event, List.of("wss://relay-two.example"));
        Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
        Assertions.assertEquals((int)0, (int)byUrl.get((Object)"wss://relay-one.example").published.size());
        Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-two.example").published.size());
        Assertions.assertEquals((int)0, (int)byUrl.get((Object)"wss://relay-three.example").published.size());
    }

    @Test
    void subscribesOnlyToRequestedRelays() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-a.example", false), new NostrRelayOption("wss://relay-b.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        AtomicInteger received = new AtomicInteger();
        try (AutoCloseable ignored = gateway.subscribe(List.of("wss://relay-a.example"), new NostrSubscription("test", event -> true), event -> received.incrementAndGet());){
            Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
            String dummySig = "0".repeat(128);
            byUrl.get("wss://relay-a.example").publish(new NostrEvent("1", "02" + "0".repeat(62), 1, "a", Instant.now(), List.of(), dummySig));
            byUrl.get("wss://relay-b.example").publish(new NostrEvent("2", "02" + "0".repeat(62), 1, "b", Instant.now(), List.of(), dummySig));
        }
        Assertions.assertTrue((received.get() <= 1 ? 1 : 0) != 0, (String)"Should receive at most 1 event from relay-a");
    }

    @Test
    void respectsRelayAccessFlags() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-readwrite.example", false), new NostrRelayOption("wss://relay-readonly.example", false, true, false), new NostrRelayOption("wss://relay-writeonly.example", false, false, true));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        NostrEvent event = new NostrEvent(UUID.randomUUID().toString(), "pub", 1, "hello", Instant.now(), List.of(), null);
        gateway.publish(event, null);
        Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
        Assertions.assertEquals((int)0, (int)byUrl.get((Object)"wss://relay-readonly.example").published.size());
        Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-readwrite.example").published.size());
        Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-writeonly.example").published.size());
        AutoCloseable ignored = gateway.subscribe(new NostrSubscription("test", e -> true), e -> {});
        if (ignored != null) {
            ignored.close();
        }
        Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-readonly.example").subscribeCount.get());
        Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-readwrite.example").subscribeCount.get());
        Assertions.assertEquals((int)0, (int)byUrl.get((Object)"wss://relay-writeonly.example").subscribeCount.get());
    }

    @Test
    void subscribeWithServerSideFilterRegisters() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-x.example", false), new NostrRelayOption("wss://relay-y.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        NostrServerSideFilter serverFilter = NostrFilterBuilder.newBuilder().kinds(new int[]{1}).authors(new String[]{"aa"}).build();
        try (AutoCloseable ignored = gateway.subscribe("srv-filter", serverFilter, e -> {});){
            Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
            Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-x.example").subscribeCount.get());
            Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-y.example").subscribeCount.get());
        }
    }

    @Test
    void subscribeWithServerSideFilterTargetsSelectedRelays() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-1.example", false), new NostrRelayOption("wss://relay-2.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        NostrServerSideFilter serverFilter = NostrFilterBuilder.newBuilder().kinds(new int[]{1}).build();
        try (AutoCloseable ignored = gateway.subscribe(List.of("wss://relay-2.example"), "srv-filter-selected", serverFilter, e -> {});){
            Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
            Assertions.assertEquals((int)0, (int)byUrl.get((Object)"wss://relay-1.example").subscribeCount.get());
            Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-2.example").subscribeCount.get());
        }
    }

    @Test
    void subscribeTextNotesByAuthorsSinceRegisters() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-z.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        try (AutoCloseable ignored = gateway.subscribeTextNotesByAuthorsSince("notes-since", List.of("aa"), Instant.now().minusSeconds(60L), 10, e -> {});){
            Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
            Assertions.assertEquals((int)1, (int)byUrl.get((Object)"wss://relay-z.example").subscribeCount.get());
        }
    }

    @Test
    void queryEventsReturnsEventsFromRelays() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-query.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        gateway.start();
        Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
        RecordingRelayClient relay = byUrl.get("wss://relay-query.example");
        String pubkey = "02" + "0".repeat(62);
        String validSig = "0".repeat(128);
        NostrEvent event1 = new NostrEvent("event-1", pubkey, 1, "content1", Instant.now(), List.of(), validSig);
        NostrEvent event2 = new NostrEvent("event-2", pubkey, 1, "content2", Instant.now(), List.of(), validSig);
        new Thread(() -> {
            try {
                Thread.sleep(20L);
                relay.publish(event1);
                relay.publish(event2);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }).start();
        NostrServerSideFilter filter = NostrFilterBuilder.newBuilder().kinds(new int[]{1}).build();
        List results = gateway.queryEvents(filter, Duration.ofMillis(200L));
        Assertions.assertTrue((results.size() <= 2 ? 1 : 0) != 0, (String)"Should return at most 2 events");
    }

    @Test
    void queryEventsDeduplicatesEventsById() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-1.example", false), new NostrRelayOption("wss://relay-2.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        gateway.start();
        Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
        String pubkey = "02" + "0".repeat(62);
        String validSig = "0".repeat(128);
        NostrEvent duplicateEvent = new NostrEvent("duplicate-id", pubkey, 1, "content", Instant.now(), List.of(), validSig);
        new Thread(() -> {
            try {
                Thread.sleep(20L);
                ((RecordingRelayClient)((Object)((Object)byUrl.get("wss://relay-1.example")))).publish(duplicateEvent);
                ((RecordingRelayClient)((Object)((Object)byUrl.get("wss://relay-2.example")))).publish(duplicateEvent);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }).start();
        NostrServerSideFilter filter = NostrFilterBuilder.newBuilder().kinds(new int[]{1}).build();
        List results = gateway.queryEvents(filter, Duration.ofMillis(200L));
        Assertions.assertTrue((results.size() <= 1 ? 1 : 0) != 0, (String)"Should deduplicate events by ID");
    }

    @Test
    void queryEventsReturnsEmptyListWhenNoMatches() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-empty.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        gateway.start();
        NostrServerSideFilter filter = NostrFilterBuilder.newBuilder().kinds(new int[]{999}).build();
        List results = gateway.queryEvents(filter, Duration.ofMillis(100L));
        Assertions.assertEquals((int)0, (int)results.size(), (String)"Should return empty list when no events match");
    }

    @Test
    void queryEventsHandlesMultipleEvents() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-multi.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        gateway.start();
        Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
        RecordingRelayClient relay = byUrl.get("wss://relay-multi.example");
        String pubkey = "02" + "0".repeat(62);
        String validSig = "0".repeat(128);
        new Thread(() -> {
            try {
                Thread.sleep(20L);
                for (int i = 0; i < 5; ++i) {
                    NostrEvent event = new NostrEvent("event-" + i, pubkey, 1, "content-" + i, Instant.now(), List.of(), validSig);
                    relay.publish(event);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }).start();
        NostrServerSideFilter filter = NostrFilterBuilder.newBuilder().kinds(new int[]{1}).build();
        List results = gateway.queryEvents(filter, Duration.ofMillis(200L));
        Assertions.assertTrue((results.size() <= 5 ? 1 : 0) != 0, (String)"Should return at most 5 events");
    }

    @Test
    void queryEventsCompletesWithinTimeout() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-timeout.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        gateway.start();
        NostrServerSideFilter filter = NostrFilterBuilder.newBuilder().kinds(new int[]{1}).build();
        long startTime = System.currentTimeMillis();
        gateway.queryEvents(filter, Duration.ofMillis(100L));
        long elapsedTime = System.currentTimeMillis() - startTime;
        Assertions.assertTrue((elapsedTime < 500L ? 1 : 0) != 0, (String)"Query should complete within timeout + overhead");
    }

    @Test
    void queryEventsFiltersEventsByKind() throws Exception {
        List<NostrRelayOption> options = List.of(new NostrRelayOption("wss://relay-filter.example", false));
        ArrayList<RecordingRelayClient> clients = new ArrayList<RecordingRelayClient>();
        NostrGatewayService gateway = this.createGateway(options, clients);
        gateway.start();
        Map<String, RecordingRelayClient> byUrl = this.mapByUrl(clients);
        RecordingRelayClient relay = byUrl.get("wss://relay-filter.example");
        String pubkey = "02" + "0".repeat(62);
        String validSig = "0".repeat(128);
        new Thread(() -> {
            try {
                Thread.sleep(20L);
                relay.publish(new NostrEvent("kind1-event", pubkey, 1, "text", Instant.now(), List.of(), validSig));
                relay.publish(new NostrEvent("kind2-event", pubkey, 2, "recommend", Instant.now(), List.of(), validSig));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }).start();
        NostrServerSideFilter filter = NostrFilterBuilder.newBuilder().kinds(new int[]{1}).build();
        List results = gateway.queryEvents(filter, Duration.ofMillis(200L));
        for (NostrEvent event : results) {
            Assertions.assertEquals((int)1, (int)event.kind(), (String)"All returned events should be kind 1");
        }
    }

    private NostrGatewayService createGateway(List<NostrRelayOption> options, List<RecordingRelayClient> clients) {
        IdentityKeyService identityKeyService = new IdentityKeyService(this.tempDir);
        SecureKeyStore secureKeyStore = SecureKeyStore.create((Path)this.tempDir);
        WalletKeyManager walletKeyManager = new WalletKeyManager(secureKeyStore);
        RecordingRelayClientFactory factory = new RecordingRelayClientFactory(clients);
        RelaySelectionPolicy policy = RelaySelectionPolicyType.STICKY.createPolicy();
        return new NostrGatewayService(() -> new NostrGatewayConfig(options), identityKeyService, walletKeyManager, (NostrRelayClientFactory)factory, policy);
    }

    private Map<String, RecordingRelayClient> mapByUrl(List<RecordingRelayClient> clients) {
        HashMap<String, RecordingRelayClient> map = new HashMap<String, RecordingRelayClient>();
        for (RecordingRelayClient client : clients) {
            map.put(client.url(), client);
        }
        return map;
    }

    private static class RecordingRelayClient
    extends InMemoryNostrRelayClient {
        private final List<NostrEvent> published = new CopyOnWriteArrayList<NostrEvent>();
        private final AtomicInteger subscribeCount = new AtomicInteger();

        private RecordingRelayClient(String url, boolean requiresAuth) {
            super(url, requiresAuth);
        }

        public void publish(NostrEvent event) {
            this.published.add(event);
            super.publish(event);
        }

        public AutoCloseable subscribe(NostrSubscription subscription, Consumer<NostrEvent> consumer) {
            this.subscribeCount.incrementAndGet();
            return super.subscribe(subscription, consumer);
        }
    }

    private static class RecordingRelayClientFactory
    implements NostrRelayClientFactory {
        private final List<RecordingRelayClient> clients;

        private RecordingRelayClientFactory(List<RecordingRelayClient> clients) {
            this.clients = clients;
        }

        public NostrRelayClient create(NostrRelayOption option, IdentityKey identityKey, WalletSigningKey walletSigningKey) {
            RecordingRelayClient client = new RecordingRelayClient(option.url(), option.requiresAuth());
            this.clients.add(client);
            if (option.requiresAuth()) {
                client.enableAuth(identityKey, walletSigningKey);
            }
            return client;
        }
    }
}

