/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.nostr.adapter;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import nostr.base.Kind;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import xyz.tcheeric.messaging.contracts.MessageClient;
import xyz.tcheeric.wallet.core.nostr.NostrEvent;
import xyz.tcheeric.wallet.core.nostr.NostrSubscription;
import xyz.tcheeric.wallet.core.nostr.adapter.BaseRelayIntegrationTest;
import xyz.tcheeric.wallet.core.nostr.adapter.NostrJavaRelayClient;
import xyz.tcheeric.wallet.core.nostr.adapter.TestMessageClient;
import xyz.tcheeric.wallet.core.security.IdentityKey;
import xyz.tcheeric.wallet.core.security.WalletSigningKey;

class NostrJavaRelayClientIntegrationTest
extends BaseRelayIntegrationTest {
    private NostrJavaRelayClient client;
    private IdentityKey testIdentityKey;
    private WalletSigningKey testWalletSigningKey;
    private TestMessageClient testMessageClient;

    NostrJavaRelayClientIntegrationTest() {
    }

    @BeforeEach
    void setUp() throws Exception {
        int i;
        KeyPairGenerator keyGen = KeyPairGenerator.getInstance("EC");
        keyGen.initialize(256);
        KeyPair keyPair = keyGen.generateKeyPair();
        this.testIdentityKey = new IdentityKey(keyPair.getPrivate(), keyPair.getPublic());
        byte[] privateKey = new byte[32];
        byte[] publicKey = new byte[32];
        byte[] salt = new byte[16];
        for (i = 0; i < 32; ++i) {
            privateKey[i] = (byte)i;
            publicKey[i] = (byte)(i + 100);
        }
        for (i = 0; i < 16; ++i) {
            salt[i] = (byte)(i + 200);
        }
        this.testWalletSigningKey = new WalletSigningKey(privateKey, publicKey, salt, 1000);
        this.testMessageClient = new TestMessageClient(this.testIdentityKey);
        this.client = new NostrJavaRelayClient(NostrJavaRelayClientIntegrationTest.getRelayUri(), (MessageClient)this.testMessageClient);
    }

    @AfterEach
    void tearDown() {
        if (this.client != null) {
            try {
                this.client.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    @Test
    void testConnect_successfullyConnects() {
        this.client.connect();
        Assertions.assertTrue((boolean)this.client.isConnected(), (String)"Client should be connected after connect()");
    }

    @Test
    void testConnect_isIdempotent() {
        this.client.connect();
        this.client.connect();
        this.client.connect();
        Assertions.assertTrue((boolean)this.client.isConnected(), (String)"Client should be connected");
    }

    @Test
    void testDisconnect_successfullyDisconnects() {
        this.client.connect();
        Assertions.assertTrue((boolean)this.client.isConnected());
        this.client.disconnect();
        Assertions.assertFalse((boolean)this.client.isConnected(), (String)"Client should be disconnected after disconnect()");
    }

    @Test
    void testPublish_successfullyPublishesEvent() {
        this.client.connect();
        String validDummyPubkey = "0".repeat(64);
        NostrEvent event = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.TEXT_NOTE.getValue(), (String)"Integration test event from NostrJavaRelayClient", (Instant)Instant.now(), List.of());
        Assertions.assertDoesNotThrow(() -> this.client.publish(event), (String)"Publishing event should not throw exception");
    }

    @Test
    void testPublish_multipleEventsSequentially() {
        this.client.connect();
        String validDummyPubkey = "0".repeat(64);
        for (int i = 0; i < 5; ++i) {
            NostrEvent event = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.TEXT_NOTE.getValue(), (String)("Test event " + i), (Instant)Instant.now(), List.of());
            Assertions.assertDoesNotThrow(() -> this.client.publish(event), (String)("Publishing event " + i + " should not throw exception"));
        }
        Assertions.assertTrue((boolean)this.client.isConnected(), (String)"Client should remain connected");
    }

    @Test
    void testPublish_throwsExceptionWhenNotConnected() {
        Assertions.assertFalse((boolean)this.client.isConnected());
        String validDummyPubkey = "0".repeat(64);
        NostrEvent event = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.TEXT_NOTE.getValue(), (String)"Test event", (Instant)Instant.now(), List.of());
        Assertions.assertThrows(IllegalStateException.class, () -> this.client.publish(event), (String)"Should throw IllegalStateException when publishing without connecting");
    }

    @Test
    void testSubscribe_receivesPublishedEvents() throws Exception {
        this.client.enableAuth("test-auth-token");
        this.client.connect();
        CountDownLatch eventReceived = new CountDownLatch(1);
        AtomicReference receivedEvent = new AtomicReference();
        NostrSubscription subscription = new NostrSubscription("test-subscription", event -> true);
        AutoCloseable subscriptionHandle = this.client.subscribe(subscription, event -> {
            receivedEvent.set(event);
            eventReceived.countDown();
        });
        String validDummyPubkey = "0".repeat(64);
        NostrEvent publishedEvent = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.TEXT_NOTE.getValue(), (String)"Event for subscription test", (Instant)Instant.now(), List.of());
        this.client.publish(publishedEvent);
        boolean received = eventReceived.await(10L, TimeUnit.SECONDS);
        Assertions.assertTrue((boolean)received, (String)"Event should be received within 10 seconds");
        Assertions.assertNotNull(receivedEvent.get(), (String)"Received event should not be null");
        Assertions.assertEquals((Object)publishedEvent.content(), (Object)((NostrEvent)receivedEvent.get()).content(), (String)"Event content should match");
        subscriptionHandle.close();
    }

    @Test
    void testSubscribe_clientSideFilteringWorks() throws Exception {
        this.client.enableAuth("test-auth-token");
        this.client.connect();
        String validDummyPubkey = "0".repeat(64);
        CountDownLatch kind1Received = new CountDownLatch(1);
        CountDownLatch kind3Received = new CountDownLatch(1);
        AtomicInteger eventsReceived = new AtomicInteger(0);
        NostrSubscription subscription = new NostrSubscription("kind-1-only", event -> event.kind() == Kind.TEXT_NOTE.getValue());
        AutoCloseable subscriptionHandle = this.client.subscribe(subscription, event -> {
            eventsReceived.incrementAndGet();
            if (event.kind() == Kind.TEXT_NOTE.getValue()) {
                kind1Received.countDown();
            } else if (event.kind() == Kind.CONTACT_LIST.getValue()) {
                kind3Received.countDown();
            }
        });
        NostrEvent kind1Event = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.TEXT_NOTE.getValue(), (String)"This is a text note", (Instant)Instant.now(), List.of());
        this.client.publish(kind1Event);
        NostrEvent kind3Event = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.CONTACT_LIST.getValue(), (String)"{}", (Instant)Instant.now(), List.of());
        this.client.publish(kind3Event);
        boolean kind1ReceivedSuccess = kind1Received.await(10L, TimeUnit.SECONDS);
        Assertions.assertTrue((boolean)kind1ReceivedSuccess, (String)"Kind 1 event should be received");
        subscriptionHandle.close();
    }

    @Test
    void testSubscribe_multipleSubscriptionsWork() throws Exception {
        this.client.enableAuth("test-auth-token");
        this.client.connect();
        CountDownLatch sub1Received = new CountDownLatch(1);
        CountDownLatch sub2Received = new CountDownLatch(1);
        NostrSubscription subscription1 = new NostrSubscription("sub1", event -> true);
        NostrSubscription subscription2 = new NostrSubscription("sub2", event -> true);
        AutoCloseable handle1 = this.client.subscribe(subscription1, event -> sub1Received.countDown());
        AutoCloseable handle2 = this.client.subscribe(subscription2, event -> sub2Received.countDown());
        String validDummyPubkey = "0".repeat(64);
        NostrEvent event2 = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.TEXT_NOTE.getValue(), (String)"Event for multiple subscriptions", (Instant)Instant.now(), List.of());
        this.client.publish(event2);
        boolean sub1Success = sub1Received.await(10L, TimeUnit.SECONDS);
        boolean sub2Success = sub2Received.await(10L, TimeUnit.SECONDS);
        Assertions.assertTrue((boolean)sub1Success, (String)"Subscription 1 should receive event");
        Assertions.assertTrue((boolean)sub2Success, (String)"Subscription 2 should receive event");
        handle1.close();
        handle2.close();
    }

    @Test
    void testSubscribe_canBeClosedCleanly() throws Exception {
        this.client.connect();
        NostrSubscription subscription = new NostrSubscription("closable", event -> true);
        AutoCloseable handle = this.client.subscribe(subscription, event -> {});
        Assertions.assertDoesNotThrow(() -> handle.close(), (String)"Closing subscription should not throw");
        Assertions.assertDoesNotThrow(() -> handle.close(), (String)"Closing again should not throw");
    }

    @Test
    void testSubscribe_throwsExceptionWhenNotConnected() {
        Assertions.assertFalse((boolean)this.client.isConnected());
        NostrSubscription subscription = new NostrSubscription("test", event -> true);
        Assertions.assertThrows(IllegalStateException.class, () -> this.client.subscribe(subscription, event -> {}), (String)"Should throw IllegalStateException when subscribing without connecting");
    }

    @Test
    void testEnableAuth_canBeEnabled() {
        Assertions.assertFalse((boolean)this.client.isAuthEnabled());
        Assertions.assertDoesNotThrow(() -> this.client.enableAuth(this.testIdentityKey, this.testWalletSigningKey), (String)"Enabling auth should not throw exception");
        Assertions.assertTrue((boolean)this.client.isAuthEnabled(), (String)"Auth should be enabled after enableAuth()");
    }

    @Test
    void testEnableAuth_clientStillWorksWithAuthEnabled() {
        this.client.enableAuth(this.testIdentityKey, this.testWalletSigningKey);
        Assertions.assertTrue((boolean)this.client.isAuthEnabled());
        Assertions.assertDoesNotThrow(() -> this.client.connect(), (String)"Connecting with auth enabled should not throw");
        String validDummyPubkey = "0".repeat(64);
        NostrEvent event = NostrEvent.unsigned((String)validDummyPubkey, (int)Kind.TEXT_NOTE.getValue(), (String)"Event with auth enabled", (Instant)Instant.now(), List.of());
        Assertions.assertDoesNotThrow(() -> this.client.publish(event), (String)"Publishing with auth enabled should not throw");
        Assertions.assertTrue((boolean)this.client.isConnected());
    }

    @Test
    void testClose_cleansUpResources() {
        this.client.connect();
        Assertions.assertTrue((boolean)this.client.isConnected());
        NostrSubscription subscription = new NostrSubscription("cleanup-test", event -> true);
        AutoCloseable handle = (AutoCloseable)Assertions.assertDoesNotThrow(() -> this.client.subscribe(subscription, event -> {}), (String)"Subscribe should not throw");
        Assertions.assertDoesNotThrow(() -> this.client.close(), (String)"Close should not throw exception");
        Assertions.assertFalse((boolean)this.client.isConnected(), (String)"Client should not be connected after close");
        Assertions.assertTrue((boolean)this.client.getSpringContext().isClosed(), (String)"Spring context should be closed");
        Assertions.assertThrows(IllegalStateException.class, () -> this.client.connect(), (String)"Operations after close should throw IllegalStateException");
    }

    @Test
    void testClose_isIdempotent() {
        this.client.connect();
        Assertions.assertDoesNotThrow(() -> {
            this.client.close();
            this.client.close();
            this.client.close();
        }, (String)"Multiple close calls should not throw");
        Assertions.assertFalse((boolean)this.client.isConnected());
    }

    @Test
    void testCircuitBreaker_isInitialized() {
        Assertions.assertNotNull((Object)this.client.getCircuitBreaker(), (String)"Circuit breaker should be initialized");
        Assertions.assertEquals((Object)("relay-" + NostrJavaRelayClientIntegrationTest.getRelayUri()), (Object)this.client.getCircuitBreaker().getName(), (String)"Circuit breaker name should include relay URL");
    }

    @Test
    void testPublish_nullEventThrowsException() {
        this.client.connect();
        Assertions.assertThrows(NullPointerException.class, () -> this.client.publish(null), (String)"Publishing null event should throw NullPointerException");
    }

    @Test
    void testSubscribe_nullSubscriptionThrowsException() {
        this.client.connect();
        Assertions.assertThrows(NullPointerException.class, () -> this.client.subscribe(null, event -> {}), (String)"Null subscription should throw NullPointerException");
    }

    @Test
    void testSubscribe_nullConsumerThrowsException() {
        this.client.connect();
        NostrSubscription subscription = new NostrSubscription("test", event -> true);
        Assertions.assertThrows(NullPointerException.class, () -> this.client.subscribe(subscription, null), (String)"Null consumer should throw NullPointerException");
    }
}

