/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.nostr.relay;

import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import xyz.tcheeric.wallet.core.nostr.NostrEvent;
import xyz.tcheeric.wallet.core.nostr.NostrGatewayConfig;
import xyz.tcheeric.wallet.core.nostr.NostrGatewayService;
import xyz.tcheeric.wallet.core.nostr.NostrRelayClientFactory;
import xyz.tcheeric.wallet.core.nostr.NostrRelayOption;
import xyz.tcheeric.wallet.core.nostr.relay.ConfigurableRelayClientFactory;
import xyz.tcheeric.wallet.core.nostr.relay.FailingNostrRelayClient;
import xyz.tcheeric.wallet.core.nostr.relay.RelayHealthMonitor;
import xyz.tcheeric.wallet.core.nostr.relay.RelayHealthRepository;
import xyz.tcheeric.wallet.core.nostr.relay.RelayMetrics;
import xyz.tcheeric.wallet.core.nostr.relay.RelayReEvaluator;
import xyz.tcheeric.wallet.core.nostr.relay.RelaySelectionPolicyType;
import xyz.tcheeric.wallet.core.nostr.relay.RelayTelemetry;
import xyz.tcheeric.wallet.core.security.IdentityKeyService;
import xyz.tcheeric.wallet.core.security.SecureKeyStore;
import xyz.tcheeric.wallet.core.security.WalletKeyManager;

class RelayFailureIntegrationTest {
    @TempDir
    Path tempDir;
    private ConfigurableRelayClientFactory clientFactory;
    private RelayHealthMonitor healthMonitor;
    private RelayReEvaluator reEvaluator;
    private RelayHealthRepository healthRepository;
    private NostrGatewayService gateway;

    RelayFailureIntegrationTest() {
    }

    @BeforeEach
    void setUp() {
        this.clientFactory = new ConfigurableRelayClientFactory();
        this.healthMonitor = new RelayHealthMonitor();
        this.reEvaluator = new RelayReEvaluator(Duration.ofMillis(100L), 5);
        this.healthRepository = new RelayHealthRepository(this.tempDir.resolve("relay-health-test.properties"));
    }

    @AfterEach
    void tearDown() {
        if (this.gateway != null) {
            this.gateway.close();
        }
    }

    @Test
    void failedRelayDoesNotBlockHealthyRelays() {
        List<NostrRelayOption> relays = List.of(new NostrRelayOption("wss://healthy.example", false), new NostrRelayOption("wss://failing.example", false));
        this.clientFactory.setFailureMode("wss://failing.example", FailingNostrRelayClient.FailureMode.ALWAYS);
        this.gateway = this.createGateway(relays);
        this.gateway.start();
        NostrEvent event = this.createTestEvent();
        Assertions.assertDoesNotThrow(() -> {
            for (int i = 0; i < 5; ++i) {
                try {
                    this.gateway.publish(event);
                    break;
                }
                catch (Exception e) {
                    if (i != 4) continue;
                    throw e;
                }
            }
        });
    }

    @Test
    void relayHealthScoresUpdateOnFailures() {
        List<NostrRelayOption> relays = List.of(new NostrRelayOption("wss://degrading.example", false));
        this.clientFactory.setFailureMode("wss://degrading.example", FailingNostrRelayClient.FailureMode.INTERMITTENT);
        this.gateway = this.createGateway(relays);
        this.gateway.start();
        NostrEvent event = this.createTestEvent();
        for (int i = 0; i < 20; ++i) {
            try {
                this.gateway.publish(event);
                continue;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        Map healthSummary = this.gateway.getRelayHealthSummary();
        Double score = (Double)healthSummary.get("wss://degrading.example");
        Assertions.assertNotNull((Object)score);
        Assertions.assertTrue((score < 100.0 ? 1 : 0) != 0, (String)"Health score should degrade with failures");
    }

    @Test
    void slowRelayShowsDegradedHealth() {
        List<NostrRelayOption> relays = List.of(new NostrRelayOption("wss://slow.example", false));
        this.clientFactory.setFailureMode("wss://slow.example", FailingNostrRelayClient.FailureMode.SLOW);
        this.gateway = this.createGateway(relays);
        this.gateway.start();
        NostrEvent event = this.createTestEvent();
        for (int i = 0; i < 5; ++i) {
            this.gateway.publish(event);
        }
        RelayMetrics metrics = this.healthMonitor.getMetrics("wss://slow.example");
        Assertions.assertNotNull((Object)metrics);
        if (metrics.p95Latency() != null) {
            Assertions.assertTrue((metrics.p95Latency().toMillis() > 100L ? 1 : 0) != 0, (String)"Slow relay should show high latency");
        }
    }

    @Test
    void relayEvictionAndRecovery() throws InterruptedException {
        List<NostrRelayOption> relays = List.of(new NostrRelayOption("wss://recovering.example", false), new NostrRelayOption("wss://stable.example", false));
        this.clientFactory.setFailureMode("wss://recovering.example", FailingNostrRelayClient.FailureMode.ALWAYS);
        this.clientFactory.setFailureMode("wss://stable.example", FailingNostrRelayClient.FailureMode.NEVER);
        this.gateway = this.createGateway(relays);
        this.gateway.start();
        NostrEvent event = this.createTestEvent();
        for (int i = 0; i < 10; ++i) {
            try {
                this.gateway.publish(event);
                continue;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        Map evictionCandidates = this.healthMonitor.getRelaysForEviction();
        if (!evictionCandidates.isEmpty()) {
            for (String url : evictionCandidates.keySet()) {
                this.reEvaluator.evictRelay(url, ((Double)evictionCandidates.get(url)).doubleValue());
            }
            Assertions.assertTrue((boolean)this.reEvaluator.isEvicted("wss://recovering.example"), (String)"Failing relay should be evicted");
            this.clientFactory.setFailureMode("wss://recovering.example", FailingNostrRelayClient.FailureMode.NEVER);
            Thread.sleep(150L);
            int n = this.gateway.performRelayReEvaluation();
        }
    }

    @Test
    void telemetryReflectsRelayHealth() {
        List<NostrRelayOption> relays = List.of(new NostrRelayOption("wss://healthy1.example", false), new NostrRelayOption("wss://healthy2.example", false), new NostrRelayOption("wss://failing.example", false));
        this.clientFactory.setFailureMode("wss://failing.example", FailingNostrRelayClient.FailureMode.ALWAYS);
        this.gateway = new NostrGatewayService(() -> new NostrGatewayConfig(relays), new IdentityKeyService(this.tempDir), new WalletKeyManager(SecureKeyStore.create((Path)this.tempDir)), (NostrRelayClientFactory)this.clientFactory, RelaySelectionPolicyType.RANDOM.createPolicy(), this.healthMonitor, this.reEvaluator, this.healthRepository, true);
        this.gateway.start();
        NostrEvent event = this.createTestEvent();
        for (int i = 0; i < 10; ++i) {
            try {
                this.gateway.publish(event);
                continue;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        RelayTelemetry telemetry = this.gateway.getRelayTelemetry();
        Assertions.assertEquals((int)3, (int)telemetry.totalRelays());
        Assertions.assertTrue((telemetry.totalOperations() > 0L ? 1 : 0) != 0);
        Assertions.assertNotNull((Object)telemetry.toLogString());
    }

    @Test
    void healthStatePersistsAcrossRestarts() {
        List<NostrRelayOption> relays = List.of(new NostrRelayOption("wss://persistent.example", false));
        this.gateway = this.createGateway(relays);
        this.gateway.start();
        NostrEvent event = this.createTestEvent();
        for (int i = 0; i < 10; ++i) {
            this.gateway.publish(event);
        }
        RelayMetrics beforeMetrics = this.healthMonitor.getMetrics("wss://persistent.example");
        Assertions.assertNotNull((Object)beforeMetrics);
        long operationsBefore = beforeMetrics.totalOperations();
        this.gateway.close();
        RelayHealthMonitor newMonitor = new RelayHealthMonitor();
        RelayReEvaluator newReEvaluator = new RelayReEvaluator();
        this.gateway = new NostrGatewayService(() -> new NostrGatewayConfig(relays), new IdentityKeyService(this.tempDir), new WalletKeyManager(SecureKeyStore.create((Path)this.tempDir)), (NostrRelayClientFactory)this.clientFactory, RelaySelectionPolicyType.STICKY.createPolicy(), newMonitor, newReEvaluator, this.healthRepository, false);
        this.gateway.start();
        RelayMetrics afterMetrics = newMonitor.getMetrics("wss://persistent.example");
        Assertions.assertNotNull((Object)afterMetrics);
        Assertions.assertEquals((long)operationsBefore, (long)afterMetrics.totalOperations(), (String)"Metrics should persist across restarts");
    }

    @Test
    void stickyPolicyMaintainsAffinity() {
        List<NostrRelayOption> relays = List.of(new NostrRelayOption("wss://relay1.example", false), new NostrRelayOption("wss://relay2.example", false), new NostrRelayOption("wss://relay3.example", false));
        this.gateway = new NostrGatewayService(() -> new NostrGatewayConfig(relays), new IdentityKeyService(this.tempDir), new WalletKeyManager(SecureKeyStore.create((Path)this.tempDir)), (NostrRelayClientFactory)this.clientFactory, RelaySelectionPolicyType.STICKY.createPolicy(), this.healthMonitor, this.reEvaluator, this.healthRepository, false);
        this.gateway.start();
        NostrEvent event = this.createTestEvent();
        for (int i = 0; i < 10; ++i) {
            this.gateway.publish(event);
        }
        boolean hasAffinity = false;
        for (String url : List.of("wss://relay1.example", "wss://relay2.example", "wss://relay3.example")) {
            FailingNostrRelayClient client = this.clientFactory.getClient(url);
            if (client == null || client.getOperationCount() < 5) continue;
            hasAffinity = true;
            break;
        }
        Assertions.assertTrue((boolean)hasAffinity, (String)"STICKY policy should show relay affinity");
    }

    private NostrGatewayService createGateway(List<NostrRelayOption> relays) {
        return new NostrGatewayService(() -> new NostrGatewayConfig(relays), new IdentityKeyService(this.tempDir), new WalletKeyManager(SecureKeyStore.create((Path)this.tempDir)), (NostrRelayClientFactory)this.clientFactory, RelaySelectionPolicyType.RANDOM.createPolicy(), this.healthMonitor, this.reEvaluator, this.healthRepository, false);
    }

    private NostrEvent createTestEvent() {
        return new NostrEvent("test-id", "test-pubkey", 1, "test content", Instant.now(), List.of(), null);
    }
}

