/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.nostr.relay;

import java.time.Duration;
import java.util.Map;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import xyz.tcheeric.wallet.core.nostr.relay.RelayHealthMonitor;
import xyz.tcheeric.wallet.core.nostr.relay.RelayMetrics;
import xyz.tcheeric.wallet.core.nostr.relay.RelayStatus;

class RelayHealthMonitorTest {
    private RelayHealthMonitor monitor;

    RelayHealthMonitorTest() {
    }

    @BeforeEach
    void setUp() {
        this.monitor = new RelayHealthMonitor();
    }

    @Test
    void constructorWithDefaultThreshold() {
        RelayHealthMonitor defaultMonitor = new RelayHealthMonitor();
        Assertions.assertEquals((double)30.0, (double)defaultMonitor.getEvictionThreshold());
    }

    @Test
    void constructorWithCustomThreshold() {
        RelayHealthMonitor customMonitor = new RelayHealthMonitor(50.0);
        Assertions.assertEquals((double)50.0, (double)customMonitor.getEvictionThreshold());
    }

    @Test
    void constructorRejectsInvalidThreshold() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> new RelayHealthMonitor(-1.0));
        Assertions.assertThrows(IllegalArgumentException.class, () -> new RelayHealthMonitor(101.0));
    }

    @Test
    void registerRelayCreatesInitialMetrics() {
        this.monitor.registerRelay("wss://relay.example");
        RelayMetrics metrics = this.monitor.getMetrics("wss://relay.example");
        Assertions.assertNotNull((Object)metrics);
        Assertions.assertEquals((long)0L, (long)metrics.totalOperations());
        Assertions.assertEquals((long)0L, (long)metrics.connectionAttempts());
    }

    @Test
    void registerRelayIsIdempotent() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.registerRelay("wss://relay.example");
        Assertions.assertEquals((int)1, (int)this.monitor.getRelayCount());
    }

    @Test
    void recordConnectionAttemptUpdatesMetrics() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.recordConnectionAttempt("wss://relay.example", true);
        RelayMetrics metrics = this.monitor.getMetrics("wss://relay.example");
        Assertions.assertEquals((long)1L, (long)metrics.connectionAttempts());
        Assertions.assertEquals((long)1L, (long)metrics.successfulConnections());
    }

    @Test
    void recordConnectionFailureUpdatesMetrics() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.recordConnectionAttempt("wss://relay.example", false);
        RelayMetrics metrics = this.monitor.getMetrics("wss://relay.example");
        Assertions.assertEquals((long)1L, (long)metrics.connectionAttempts());
        Assertions.assertEquals((long)0L, (long)metrics.successfulConnections());
    }

    @Test
    void recordOperationUpdatesMetrics() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.recordOperation("wss://relay.example", true, Duration.ofMillis(100L));
        RelayMetrics metrics = this.monitor.getMetrics("wss://relay.example");
        Assertions.assertEquals((long)1L, (long)metrics.totalOperations());
        Assertions.assertEquals((long)1L, (long)metrics.successfulOperations());
        Assertions.assertEquals((Object)Duration.ofMillis(100L), (Object)metrics.p95Latency());
    }

    @Test
    void recordOperationFailureUpdatesMetrics() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.recordOperation("wss://relay.example", false, Duration.ofMillis(500L));
        RelayMetrics metrics = this.monitor.getMetrics("wss://relay.example");
        Assertions.assertEquals((long)1L, (long)metrics.totalOperations());
        Assertions.assertEquals((long)0L, (long)metrics.successfulOperations());
        Assertions.assertEquals((Object)Duration.ofMillis(500L), (Object)metrics.p95Latency());
    }

    @Test
    void calculateScoreForRegisteredRelay() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.recordConnectionAttempt("wss://relay.example", true);
        this.monitor.recordOperation("wss://relay.example", true, Duration.ofMillis(100L));
        Double score = this.monitor.calculateScore("wss://relay.example");
        Assertions.assertNotNull((Object)score);
        Assertions.assertTrue((score >= 0.0 && score <= 100.0 ? 1 : 0) != 0);
    }

    @Test
    void calculateScoreReturnsNullForUnregisteredRelay() {
        Double score = this.monitor.calculateScore("wss://unknown.example");
        Assertions.assertNull((Object)score);
    }

    @Test
    void determineStatusForRegisteredRelay() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.recordConnectionAttempt("wss://relay.example", true);
        this.monitor.recordOperation("wss://relay.example", true, Duration.ofMillis(100L));
        RelayStatus status = this.monitor.determineStatus("wss://relay.example");
        Assertions.assertNotNull((Object)status);
    }

    @Test
    void determineStatusReturnsNullForUnregisteredRelay() {
        RelayStatus status = this.monitor.determineStatus("wss://unknown.example");
        Assertions.assertNull((Object)status);
    }

    @Test
    void getRelaysForEvictionIncludesFailingRelays() {
        this.monitor.registerRelay("wss://failing.example");
        this.monitor.recordConnectionAttempt("wss://failing.example", true);
        for (int i = 0; i < 100; ++i) {
            this.monitor.recordOperation("wss://failing.example", i < 10, Duration.ofSeconds(6L));
        }
        Map evictionCandidates = this.monitor.getRelaysForEviction();
        Assertions.assertTrue((boolean)evictionCandidates.containsKey("wss://failing.example"));
        Assertions.assertTrue(((Double)evictionCandidates.get("wss://failing.example") < this.monitor.getEvictionThreshold() ? 1 : 0) != 0);
    }

    @Test
    void getRelaysForEvictionExcludesHealthyRelays() {
        this.monitor.registerRelay("wss://healthy.example");
        this.monitor.recordConnectionAttempt("wss://healthy.example", true);
        for (int i = 0; i < 100; ++i) {
            this.monitor.recordOperation("wss://healthy.example", true, Duration.ofMillis(100L));
        }
        Map evictionCandidates = this.monitor.getRelaysForEviction();
        Assertions.assertFalse((boolean)evictionCandidates.containsKey("wss://healthy.example"));
    }

    @Test
    void getRelaysForEvictionExcludesRelaysWithInsufficientData() {
        this.monitor.registerRelay("wss://new.example");
        Map evictionCandidates = this.monitor.getRelaysForEviction();
        Assertions.assertFalse((boolean)evictionCandidates.containsKey("wss://new.example"));
    }

    @Test
    void getHealthSummaryIncludesAllRelays() {
        this.monitor.registerRelay("wss://relay1.example");
        this.monitor.registerRelay("wss://relay2.example");
        Map summary = this.monitor.getHealthSummary();
        Assertions.assertEquals((int)2, (int)summary.size());
        Assertions.assertTrue((boolean)summary.containsKey("wss://relay1.example"));
        Assertions.assertTrue((boolean)summary.containsKey("wss://relay2.example"));
    }

    @Test
    void removeRelayRemovesFromMonitoring() {
        this.monitor.registerRelay("wss://relay.example");
        boolean removed = this.monitor.removeRelay("wss://relay.example");
        Assertions.assertTrue((boolean)removed);
        Assertions.assertEquals((int)0, (int)this.monitor.getRelayCount());
        Assertions.assertNull((Object)this.monitor.getMetrics("wss://relay.example"));
    }

    @Test
    void removeRelayReturnsFalseForUnregistered() {
        boolean removed = this.monitor.removeRelay("wss://unknown.example");
        Assertions.assertFalse((boolean)removed);
    }

    @Test
    void resetMetricsCreatesCleanState() {
        this.monitor.registerRelay("wss://relay.example");
        this.monitor.recordOperation("wss://relay.example", true, Duration.ofMillis(100L));
        this.monitor.resetMetrics("wss://relay.example");
        RelayMetrics metrics = this.monitor.getMetrics("wss://relay.example");
        Assertions.assertEquals((long)0L, (long)metrics.totalOperations());
        Assertions.assertEquals((long)0L, (long)metrics.connectionAttempts());
        Assertions.assertNull((Object)metrics.p95Latency());
    }

    @Test
    void getRelayCountReturnsCorrectCount() {
        Assertions.assertEquals((int)0, (int)this.monitor.getRelayCount());
        this.monitor.registerRelay("wss://relay1.example");
        Assertions.assertEquals((int)1, (int)this.monitor.getRelayCount());
        this.monitor.registerRelay("wss://relay2.example");
        Assertions.assertEquals((int)2, (int)this.monitor.getRelayCount());
        this.monitor.removeRelay("wss://relay1.example");
        Assertions.assertEquals((int)1, (int)this.monitor.getRelayCount());
    }

    @Test
    void recordOperationAutoRegistersRelay() {
        this.monitor.recordOperation("wss://new.example", true, Duration.ofMillis(100L));
        RelayMetrics metrics = this.monitor.getMetrics("wss://new.example");
        Assertions.assertNotNull((Object)metrics);
        Assertions.assertEquals((long)1L, (long)metrics.totalOperations());
    }

    @Test
    void recordConnectionAttemptAutoRegistersRelay() {
        this.monitor.recordConnectionAttempt("wss://new.example", true);
        RelayMetrics metrics = this.monitor.getMetrics("wss://new.example");
        Assertions.assertNotNull((Object)metrics);
        Assertions.assertEquals((long)1L, (long)metrics.connectionAttempts());
    }

    @Test
    void methodsRejectNullRelayUrl() {
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.registerRelay(null));
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.recordConnectionAttempt(null, true));
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.recordOperation(null, true, null));
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.getMetrics(null));
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.calculateScore(null));
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.determineStatus(null));
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.removeRelay(null));
        Assertions.assertThrows(NullPointerException.class, () -> this.monitor.resetMetrics(null));
    }
}

