/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.nostr;

import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HexFormat;
import java.util.Locale;
import java.util.Objects;
import org.bouncycastle.asn1.sec.SECNamedCurves;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECFieldElement;
import org.bouncycastle.math.ec.ECPoint;
import xyz.tcheeric.wallet.core.nostr.NostrEvent;

public final class NostrEventVerifier {
    private static final HexFormat HEX = HexFormat.of();
    private static final X9ECParameters CURVE_PARAMS = SECNamedCurves.getByName((String)"secp256k1");
    private static final ECCurve CURVE = CURVE_PARAMS.getCurve();
    private static final ECDomainParameters DOMAIN = new ECDomainParameters(CURVE_PARAMS.getCurve(), CURVE_PARAMS.getG(), CURVE_PARAMS.getN(), CURVE_PARAMS.getH());
    private static final BigInteger FIELD_MODULUS = ((ECCurve.AbstractFp)CURVE).getQ();
    private static final BigInteger GROUP_ORDER = CURVE_PARAMS.getN();
    private static final byte[] TAG_CHALLENGE = NostrEventVerifier.hashTag("BIP0340/challenge");

    public VerificationResult verify(NostrEvent event, String signatureHex) {
        Objects.requireNonNull(event, "event");
        if (signatureHex == null || signatureHex.isBlank()) {
            return VerificationResult.invalid("MISSING_SIGNATURE", "Signature is required for verification");
        }
        if (event.id() == null || event.id().isBlank()) {
            return VerificationResult.invalid("MISSING_EVENT_ID", "Event id is required for verification");
        }
        if (event.pubkey() == null || event.pubkey().isBlank()) {
            return VerificationResult.invalid("MISSING_PUBKEY", "Public key is required for verification");
        }
        try {
            byte[] signature = HEX.parseHex(signatureHex);
            byte[] pubkey = HEX.parseHex(event.pubkey());
            byte[] providedId = HEX.parseHex(event.id());
            if (signature.length != 64) {
                return VerificationResult.invalid("INVALID_SIGNATURE_LENGTH", "Signature must be 64 bytes");
            }
            if (pubkey.length != 32) {
                return VerificationResult.invalid("INVALID_PUBKEY_LENGTH", "Public key must be 32 bytes");
            }
            if (providedId.length != 32) {
                return VerificationResult.invalid("INVALID_EVENT_ID_LENGTH", "Event id must be 32 bytes");
            }
            String canonicalId = NostrEvent.generateDeterministicId(event.pubkey(), event.kind(), event.content(), event.tags(), event.createdAt());
            if (!canonicalId.equalsIgnoreCase(event.id())) {
                return VerificationResult.invalid("EVENT_ID_MISMATCH", "Event id does not match canonical serialization");
            }
            byte[] message = HEX.parseHex(canonicalId.toLowerCase(Locale.ROOT));
            if (this.verifySignature(pubkey, message, signature)) {
                return VerificationResult.valid();
            }
            return VerificationResult.invalid("INVALID_SIGNATURE", "Signature does not match the provided event");
        }
        catch (IllegalArgumentException e) {
            return VerificationResult.invalid("INVALID_HEX", "Failed to parse hexadecimal data: " + e.getMessage());
        }
    }

    private boolean verifySignature(byte[] publicKey, byte[] message, byte[] signature) {
        ECPoint eP;
        BigInteger r = new BigInteger(1, NostrEventVerifier.slice(signature, 0, 32));
        BigInteger s = new BigInteger(1, NostrEventVerifier.slice(signature, 32, 64));
        if (r.signum() == 0 || r.compareTo(FIELD_MODULUS) >= 0) {
            return false;
        }
        if (s.signum() == 0 || s.compareTo(GROUP_ORDER) >= 0) {
            return false;
        }
        ECPoint P = NostrEventVerifier.liftX(publicKey);
        if (P == null) {
            return false;
        }
        byte[] eBytes = NostrEventVerifier.taggedHash(TAG_CHALLENGE, NostrEventVerifier.toFixedLength(r, 32), publicKey, message);
        BigInteger e = new BigInteger(1, eBytes).mod(GROUP_ORDER);
        ECPoint sG = DOMAIN.getG().multiply(s);
        ECPoint R = sG.subtract(eP = P.multiply(e)).normalize();
        if (R.isInfinity()) {
            return false;
        }
        BigInteger xCoord = R.getAffineXCoord().toBigInteger();
        if (R.getAffineYCoord().toBigInteger().testBit(0)) {
            return false;
        }
        return xCoord.equals(r);
    }

    private static ECPoint liftX(byte[] publicKey) {
        BigInteger x = new BigInteger(1, publicKey);
        if (x.signum() < 0 || x.compareTo(FIELD_MODULUS) >= 0) {
            return null;
        }
        ECFieldElement xField = CURVE.fromBigInteger(x);
        ECFieldElement alpha = xField.multiply(xField.square()).add(CURVE.getB());
        ECFieldElement beta = alpha.sqrt();
        if (beta == null) {
            return null;
        }
        BigInteger y = beta.toBigInteger();
        if (y.testBit(0)) {
            y = FIELD_MODULUS.subtract(y);
        }
        return CURVE.validatePoint(x, y);
    }

    private static byte[] taggedHash(byte[] tagHash, byte[] ... data) {
        MessageDigest digest = NostrEventVerifier.newDigest();
        digest.update(tagHash);
        digest.update(tagHash);
        for (byte[] part : data) {
            digest.update(part);
        }
        return digest.digest();
    }

    private static byte[] toFixedLength(BigInteger value, int length) {
        byte[] bytes = value.toByteArray();
        if (bytes.length == length) {
            return bytes;
        }
        byte[] result = new byte[length];
        int srcPos = Math.max(0, bytes.length - length);
        int destPos = Math.max(0, length - bytes.length);
        int copyLength = Math.min(length, bytes.length);
        System.arraycopy(bytes, srcPos, result, destPos, copyLength);
        return result;
    }

    private static byte[] slice(byte[] input, int from, int to) {
        byte[] slice = new byte[to - from];
        System.arraycopy(input, from, slice, 0, slice.length);
        return slice;
    }

    private static byte[] hashTag(String tag) {
        return NostrEventVerifier.sha256(tag.getBytes(StandardCharsets.US_ASCII));
    }

    private static byte[] sha256(byte[] input) {
        return NostrEventVerifier.newDigest().digest(input);
    }

    private static MessageDigest newDigest() {
        try {
            return MessageDigest.getInstance("SHA-256");
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("SHA-256 digest not available", e);
        }
    }

    public static final class VerificationResult {
        private final boolean valid;
        private final String failureReason;
        private final String failureMessage;

        private VerificationResult(boolean valid, String failureReason, String failureMessage) {
            this.valid = valid;
            this.failureReason = failureReason;
            this.failureMessage = failureMessage;
        }

        public static VerificationResult valid() {
            return new VerificationResult(true, null, null);
        }

        public static VerificationResult invalid(String failureReason, String failureMessage) {
            return new VerificationResult(false, failureReason, failureMessage);
        }

        public boolean isValid() {
            return this.valid;
        }

        public String getFailureReason() {
            return this.failureReason;
        }

        public String getFailureMessage() {
            return this.failureMessage;
        }
    }
}

