/*
 * Decompiled with CFR 0.152.
 */
package xyz.tcheeric.wallet.core.nostr.relay;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.tcheeric.wallet.core.StoragePaths;
import xyz.tcheeric.wallet.core.nostr.relay.EvictedRelay;
import xyz.tcheeric.wallet.core.nostr.relay.RelayHealthSnapshot;
import xyz.tcheeric.wallet.core.nostr.relay.RelayMetrics;

public class RelayHealthRepository {
    private static final Logger LOGGER = LoggerFactory.getLogger(RelayHealthRepository.class);
    private static final Base64.Encoder URL_ENCODER = Base64.getUrlEncoder().withoutPadding();
    private static final Base64.Decoder URL_DECODER = Base64.getUrlDecoder();
    private static final String LEGACY_WSS_PREFIX = "wss___";
    private static final String LEGACY_WS_PREFIX = "ws___";
    private final Path storagePath;

    public RelayHealthRepository() {
        this(StoragePaths.relayHealthFile());
    }

    public RelayHealthRepository(Path storagePath) {
        this.storagePath = Objects.requireNonNull(storagePath, "storagePath");
    }

    public synchronized void save(Map<String, RelayMetrics> metricsMap, Map<String, EvictedRelay> evictedMap) {
        String prefix;
        String sanitized;
        String url;
        Objects.requireNonNull(metricsMap, "metricsMap");
        Objects.requireNonNull(evictedMap, "evictedMap");
        Properties props = new Properties();
        for (Map.Entry<String, RelayMetrics> entry : metricsMap.entrySet()) {
            url = entry.getKey();
            RelayMetrics metrics = entry.getValue();
            sanitized = this.sanitizeUrl(url);
            prefix = "metrics." + sanitized;
            props.setProperty(prefix + ".url", url);
            props.setProperty(prefix + ".successfulOperations", String.valueOf(metrics.successfulOperations()));
            props.setProperty(prefix + ".totalOperations", String.valueOf(metrics.totalOperations()));
            props.setProperty(prefix + ".successfulConnections", String.valueOf(metrics.successfulConnections()));
            props.setProperty(prefix + ".connectionAttempts", String.valueOf(metrics.connectionAttempts()));
            if (metrics.p95Latency() != null) {
                props.setProperty(prefix + ".p95LatencyMs", String.valueOf(metrics.p95Latency().toMillis()));
            }
            props.setProperty(prefix + ".capturedAt", metrics.capturedAt().toString());
        }
        for (Map.Entry<String, Record> entry : evictedMap.entrySet()) {
            url = entry.getKey();
            EvictedRelay evicted = (EvictedRelay)entry.getValue();
            sanitized = this.sanitizeUrl(url);
            prefix = "evicted." + sanitized;
            props.setProperty(prefix + ".url", url);
            props.setProperty(prefix + ".evictionScore", String.valueOf(evicted.evictionScore()));
            props.setProperty(prefix + ".retryCount", String.valueOf(evicted.retryCount()));
            props.setProperty(prefix + ".evictedAt", evicted.evictedAt().toString());
        }
        try (FileOutputStream out = new FileOutputStream(this.storagePath.toFile());){
            props.store(out, "Relay Health State");
            LOGGER.info("relay_health_repository saved metrics_count={} evicted_count={} path={}", new Object[]{metricsMap.size(), evictedMap.size(), this.storagePath});
        }
        catch (IOException e) {
            LOGGER.error("relay_health_repository save_failed path={} reason={}", new Object[]{this.storagePath, e.getMessage(), e});
            throw new RuntimeException("Failed to save relay health state", e);
        }
    }

    public synchronized RelayHealthSnapshot load() {
        if (!Files.exists(this.storagePath, new LinkOption[0])) {
            LOGGER.debug("relay_health_repository load_skipped reason=file_not_found path={}", (Object)this.storagePath);
            return new RelayHealthSnapshot(Map.of(), Map.of(), Instant.now());
        }
        Properties props = new Properties();
        try (FileInputStream in = new FileInputStream(this.storagePath.toFile());){
            props.load(in);
        }
        catch (IOException e) {
            LOGGER.error("relay_health_repository load_failed path={} reason={}", new Object[]{this.storagePath, e.getMessage(), e});
            throw new RuntimeException("Failed to load relay health state", e);
        }
        ConcurrentHashMap<String, RelayMetrics> metricsMap = new ConcurrentHashMap<String, RelayMetrics>();
        ConcurrentHashMap<String, EvictedRelay> evictedMap = new ConcurrentHashMap<String, EvictedRelay>();
        Set<String> metricPrefixes = this.extractPrefixes(props, "metrics.");
        for (String prefix : metricPrefixes) {
            try {
                String sanitized = prefix.substring("metrics.".length());
                String url = props.getProperty(prefix + ".url");
                if (url == null) {
                    url = this.desanitizeUrl(sanitized);
                }
                long successfulOps = Long.parseLong(props.getProperty(prefix + ".successfulOperations", "0"));
                long totalOps = Long.parseLong(props.getProperty(prefix + ".totalOperations", "0"));
                long successfulConns = Long.parseLong(props.getProperty(prefix + ".successfulConnections", "0"));
                long connAttempts = Long.parseLong(props.getProperty(prefix + ".connectionAttempts", "0"));
                Duration p95Latency = null;
                String latencyStr = props.getProperty(prefix + ".p95LatencyMs");
                if (latencyStr != null) {
                    p95Latency = Duration.ofMillis(Long.parseLong(latencyStr));
                }
                Instant capturedAt = Instant.parse(props.getProperty(prefix + ".capturedAt", Instant.now().toString()));
                RelayMetrics metrics = new RelayMetrics(url, p95Latency, successfulOps, totalOps, successfulConns, connAttempts, capturedAt);
                metricsMap.put(url, metrics);
            }
            catch (Exception e) {
                LOGGER.warn("relay_health_repository metric_parse_failed prefix={} reason={}", (Object)prefix, (Object)e.getMessage());
            }
        }
        Set<String> evictedPrefixes = this.extractPrefixes(props, "evicted.");
        for (String prefix : evictedPrefixes) {
            try {
                String sanitized = prefix.substring("evicted.".length());
                String url = props.getProperty(prefix + ".url");
                if (url == null) {
                    url = this.desanitizeUrl(sanitized);
                }
                double evictionScore = Double.parseDouble(props.getProperty(prefix + ".evictionScore", "0"));
                int retryCount = Integer.parseInt(props.getProperty(prefix + ".retryCount", "0"));
                Instant evictedAt = Instant.parse(props.getProperty(prefix + ".evictedAt", Instant.now().toString()));
                EvictedRelay evicted = new EvictedRelay(url, evictedAt, evictionScore, retryCount);
                evictedMap.put(url, evicted);
            }
            catch (Exception e) {
                LOGGER.warn("relay_health_repository evicted_parse_failed prefix={} reason={}", (Object)prefix, (Object)e.getMessage());
            }
        }
        LOGGER.info("relay_health_repository loaded metrics_count={} evicted_count={} path={}", new Object[]{metricsMap.size(), evictedMap.size(), this.storagePath});
        return new RelayHealthSnapshot(metricsMap, evictedMap, Instant.now());
    }

    public synchronized void clear() {
        try {
            Files.deleteIfExists(this.storagePath);
            LOGGER.info("relay_health_repository cleared path={}", (Object)this.storagePath);
        }
        catch (IOException e) {
            LOGGER.error("relay_health_repository clear_failed path={} reason={}", new Object[]{this.storagePath, e.getMessage(), e});
        }
    }

    private Set<String> extractPrefixes(Properties props, String basePrefix) {
        HashSet<String> prefixes = new HashSet<String>();
        for (String key : props.stringPropertyNames()) {
            int lastDot;
            if (!key.startsWith(basePrefix) || (lastDot = key.lastIndexOf(46)) <= 0) continue;
            prefixes.add(key.substring(0, lastDot));
        }
        return prefixes;
    }

    private String sanitizeUrl(String url) {
        return URL_ENCODER.encodeToString(url.getBytes(StandardCharsets.UTF_8));
    }

    private String desanitizeUrl(String sanitized) {
        try {
            byte[] decoded = URL_DECODER.decode(sanitized);
            return new String(decoded, StandardCharsets.UTF_8);
        }
        catch (IllegalArgumentException e) {
            return this.legacyDesanitizeUrl(sanitized);
        }
    }

    private String legacyDesanitizeUrl(String legacyEncoded) {
        if (legacyEncoded.startsWith(LEGACY_WSS_PREFIX)) {
            return this.rebuildLegacyUrl("wss://", legacyEncoded, LEGACY_WSS_PREFIX.length());
        }
        if (legacyEncoded.startsWith(LEGACY_WS_PREFIX)) {
            return this.rebuildLegacyUrl("ws://", legacyEncoded, LEGACY_WS_PREFIX.length());
        }
        return legacyEncoded.replace("_", ".");
    }

    private String rebuildLegacyUrl(String protocol, String encoded, int prefixLength) {
        return protocol + encoded.substring(prefixLength).replace("_", ".");
    }
}

