const vouchers = [
    {
        id: 'alpha',
        name: 'Voucher A',
        amount: 1250,
        mint: 'cashu://greenlight.mint',
        health: 'green',
        description: 'Primary Lightning voucher for day-to-day spending.',
        issuer: {
            name: 'Greenlight Mint',
            description: 'Low-fee mint tuned for Lightning routing reliability.',
            location: 'Zurich, CH',
            contact: 'support@greenlight.mint',
            profileImage: 'originals/original-b9af2a510b25938c6f9635243e56834a.webp',
            bannerImage: 'originals/original-fc6b92e9f83ccd78e5eab377f75c8003.png',
            username: '@greenlight',
            npub: 'npub1greenlightpublickey',
            mintName: 'Greenlight Mint'
        }
    },
    {
        id: 'beta',
        name: 'Voucher B',
        amount: 820,
        mint: 'cashu://purple.mint',
        health: 'orange',
        description: 'Backup balance reserved for service payments.',
        issuer: {
            name: 'Lavender Mint',
            description: 'Regional mint with predictable settlement windows.',
            location: 'Lisbon, PT',
            contact: 'hello@lavender.mint',
            profileImage: 'originals/original-c6ceee01307ac05094326904874de902.webp',
            bannerImage: 'originals/original-fc6b92e9f83ccd78e5eab377f75c8003.png',
            username: '@lavender',
            npub: 'npub1lavenderpublickey',
            mintName: 'Lavender Mint'
        }
    },
    {
        id: 'gamma',
        name: 'Voucher C',
        amount: 540,
        mint: 'cashu://satlight.mint',
        health: 'red',
        description: 'Smaller voucher ideal for testing new relays.',
        issuer: {
            name: 'Satlight Mint',
            description: 'Experimental mint with fast key rotation.',
            location: 'Austin, US',
            contact: 'ops@satlight.mint',
            profileImage: 'originals/original-f1dae34cea98e5aa010b1632253c1e77.png',
            bannerImage: 'originals/original-f1dae34cea98e5aa010b1632253c1e77.png',
            username: '@satlight',
            npub: 'npub1satlightpublickey',
            mintName: 'Satlight Mint'
        }
    }
];

const services = [
    { name: 'Node monitoring', desc: 'Lightning uptime with webhooks', amount: '120 sats/mo' },
    { name: 'Relay priority', desc: 'Pinned slots for ACKs', amount: '70 sats/mo' },
    { name: 'Support plan', desc: 'Response in under 2h', amount: '250 sats/mo' }
];

const activities = [
    { type: 'Receive', detail: 'From @heyden', amount: '+410 sats', time: '2m ago' },
    { type: 'Send', detail: 'To @sandy', amount: '-250 sats', time: '8m ago' },
    { type: 'Pay', detail: 'Invoice lnbc1xy9…', amount: '-90 sats', time: '28m ago' }
];

const notifications = [
    { id: 'n1', title: 'Mint health degraded', time: '2m ago', detail: 'Lavender Mint is experiencing delays.', severity: 'orange' },
    { id: 'n2', title: 'New voucher received', time: '1h ago', detail: 'You received 210 sats from @node.', severity: 'green' },
    { id: 'n3', title: 'Relay reconnect', time: '3h ago', detail: 'Relay wss://relay.cashu.org reconnected.', severity: 'green' }
];

const themePreferenceKey = 'theme-preference';
const themeCycle = ['system', 'light', 'dark'];

const transactions = [
    { id: 't1', voucher: 'alpha', type: 'Send', to: '@sandy', amount: -250, date: '2024-06-10' },
    { id: 't2', voucher: 'beta', type: 'Receive', to: '@heyden', amount: 410, date: '2024-06-09' },
    { id: 't3', voucher: 'gamma', type: 'Pay', to: 'Service', amount: -90, date: '2024-06-05' },
    { id: 't4', voucher: 'alpha', type: 'Receive', to: '@relay', amount: 320, date: '2024-05-29' },
    { id: 't5', voucher: 'beta', type: 'Send', to: '@stacker', amount: -120, date: '2024-05-22' },
    { id: 't6', voucher: 'gamma', type: 'Receive', to: '@node', amount: 210, date: '2024-05-15' }
];

const relays = [
    { url: 'wss://relay.cashu.org', health: 'green' },
    { url: 'wss://nostr.green', health: 'orange' },
    { url: 'wss://civkit.relay', health: 'red' }
];
const mints = [
    { name: 'Greenlight Mint', health: 'green' },
    { name: 'Lavender Mint', health: 'orange' }
];
const recipients = [
    { id: 'u1', name: 'Sandy', handle: '@sandy' },
    { id: 'u2', name: 'Heyden', handle: '@heyden' },
    { id: 'u3', name: 'Relay Ops', handle: '@relay' }
];
const contacts = [
    {
        id: 'u1',
        name: 'Sandy Carter',
        handle: '@sandy',
        description: 'Lightning dev and relay runner.',
        location: 'Berlin',
        npub: 'npub1sandy',
        mint: 'Greenlight Mint',
        avatar: 'originals/original-b9af2a510b25938c6f9635243e56834a.webp'
    },
    {
        id: 'u2',
        name: 'Heyden Brooks',
        handle: '@heyden',
        description: 'Runs Lavender Mint desk.',
        location: 'Lisbon',
        npub: 'npub1heyden',
        mint: 'Lavender Mint',
        avatar: 'originals/original-c6ceee01307ac05094326904874de902.webp'
    },
    {
        id: 'u3',
        name: 'Relay Ops',
        handle: '@relay',
        description: 'Monitoring partner for Nostr relays.',
        location: 'Austin',
        npub: 'npub1relayops',
        mint: 'Satlight Mint',
        avatar: 'originals/original-f1dae34cea98e5aa010b1632253c1e77.png'
    }
];

let selectedVoucher = vouchers[0];
let voucherPage = 0;
const voucherPageSize = 3;
let voucherSearch = '';
let walletVoucherSearch = '';
let walletVoucherPage = 0;
let selectedContact = contacts[0];

function showPage(targetId) {
    document.querySelectorAll('.page').forEach(page => page.classList.toggle('active', page.id === targetId));
}

function formatSats(amount) {
    return `${amount.toLocaleString()} sats`;
}

function buildTransactionRow(tx) {
    const row = document.createElement('div');
    row.className = 'transaction-row';
    const amountLabel = `${tx.amount > 0 ? '+' : ''}${tx.amount} sats`;
    row.innerHTML = `
        <div>
            <p class="transaction-title">${tx.type}</p>
            <p class="muted">${tx.to} • ${tx.date}</p>
        </div>
        <span class="chip ${tx.amount < 0 ? 'danger' : 'success'}">${amountLabel}</span>
    `;
    return row;
}

function setStatusLight(element, health) {
    if (!element) return;
    element.className = `status-light ${health}`;
    element.setAttribute('aria-label', `${health} health`);
}

function renderTransactionList(container, txList, emptyText) {
    if (!container) {
        return;
    }

    container.innerHTML = '';

    if (!txList.length) {
        container.innerHTML = `<p class="muted">${emptyText}</p>`;
        return;
    }

    txList.forEach(tx => container.appendChild(buildTransactionRow(tx)));
}

function renderReceiveState() {
    const name = document.getElementById('receive-voucher-name');
    const mintLabel = document.getElementById('receive-mint');
    const statusLabel = document.getElementById('receive-status');

    if (name) {
        name.textContent = `${selectedVoucher.name} QR`;
    }
    if (mintLabel) {
        mintLabel.textContent = selectedVoucher.issuer.name;
    }
    if (statusLabel) {
        statusLabel.textContent = `${selectedVoucher.health} health`;
    }
}

function renderVoucherDetail() {
    const pageTitle = document.getElementById('voucher-page-title');
    const amount = document.getElementById('voucher-detail-amount');
    const healthLight = document.getElementById('voucher-detail-health');
    const issuerLabel = document.getElementById('voucher-detail-issuer');
    const issuerMint = document.getElementById('voucher-issuer-mint');
    const issuerUsername = document.getElementById('voucher-issuer-username');
    const issuerNpub = document.getElementById('voucher-issuer-npub');
    const desc = document.getElementById('voucher-detail-description');
    const issuerName = document.getElementById('voucher-issuer-name');
    const issuerDesc = document.getElementById('voucher-issuer-desc');
    const issuerLocation = document.getElementById('voucher-issuer-location');
    const issuerContact = document.getElementById('voucher-issuer-contact');
    const banner = document.getElementById('issuer-banner');
    const avatar = document.getElementById('issuer-avatar');

    if (pageTitle) pageTitle.textContent = selectedVoucher.name;
    if (amount) amount.textContent = formatSats(selectedVoucher.amount);
    setStatusLight(healthLight, selectedVoucher.health);
    if (issuerLabel) issuerLabel.textContent = selectedVoucher.issuer.name;
    if (issuerMint) issuerMint.textContent = selectedVoucher.issuer.mintName;
    if (issuerUsername) issuerUsername.textContent = selectedVoucher.issuer.username;
    if (issuerNpub) issuerNpub.textContent = selectedVoucher.issuer.npub;
    if (desc) desc.textContent = selectedVoucher.description;
    if (issuerName) issuerName.textContent = selectedVoucher.issuer.name;
    if (issuerDesc) issuerDesc.textContent = selectedVoucher.issuer.description;
    if (issuerLocation) issuerLocation.textContent = selectedVoucher.issuer.location;
    if (issuerContact) issuerContact.textContent = selectedVoucher.issuer.contact;
    if (banner) banner.style.backgroundImage = `url('${selectedVoucher.issuer.bannerImage}')`;
    if (avatar) avatar.style.backgroundImage = `url('${selectedVoucher.issuer.profileImage}')`;

    renderVoucherTransactions();
    renderReceiveState();
}

function setSelectedVoucher(voucherId, options = {}) {
    selectedVoucher = vouchers.find(v => v.id === voucherId) || vouchers[0];
    const balance = document.getElementById('selected-balance');
    const issuer = document.getElementById('selected-issuer');
    const health = document.getElementById('selected-health');
    const heroName = document.getElementById('hero-voucher-name');
    if (balance) balance.textContent = formatSats(selectedVoucher.amount);
    if (issuer) issuer.textContent = selectedVoucher.issuer.name;
    setStatusLight(health, selectedVoucher.health);
    if (heroName) heroName.textContent = selectedVoucher.name;

    renderWalletHero();

    document.querySelectorAll('.voucher-card').forEach(card => {
        card.classList.toggle('active', card.dataset.voucher === selectedVoucher.id);
    });

    renderVoucherDetail();

    if (options.navigateToDetail) {
        showPage('voucher-page');
    }
}

function walletFilteredVouchers() {
    if (!walletVoucherSearch.trim()) {
        return vouchers;
    }
    const q = walletVoucherSearch.toLowerCase();
    return vouchers.filter(v =>
        v.name.toLowerCase().includes(q) ||
        v.issuer.name.toLowerCase().includes(q) ||
        v.issuer.username.toLowerCase().includes(q)
    );
}

function renderWalletPagination(totalItems) {
    const totalPages = Math.max(1, Math.ceil(totalItems / voucherPageSize));
    if (walletVoucherPage >= totalPages) {
        walletVoucherPage = totalPages - 1;
    }
    const label = document.getElementById('wallet-voucher-page-label');
    if (label) {
        label.textContent = `Page ${walletVoucherPage + 1} of ${totalPages}`;
    }
}

function renderWalletCarousel() {
    const carousel = document.getElementById('wallet-voucher-carousel');
    if (!carousel) return;

    carousel.innerHTML = '';
    const visible = walletFilteredVouchers();
    renderWalletPagination(visible.length);
    const totalPages = Math.max(1, Math.ceil(visible.length / voucherPageSize));
    if (walletVoucherPage >= totalPages) {
        walletVoucherPage = totalPages - 1;
    }
    const start = walletVoucherPage * voucherPageSize;
    const pageItems = visible.slice(start, start + voucherPageSize);

    if (!pageItems.length) {
        carousel.innerHTML = '<p class="muted">No vouchers in your wallet match this search.</p>';
        return;
    }

    pageItems.forEach(voucher => {
        const card = document.createElement('button');
        card.type = 'button';
        card.className = 'voucher-card wallet-card';
        card.dataset.voucher = voucher.id;
        if (voucher.id === selectedVoucher.id) {
            card.classList.add('active');
        }
        card.style.setProperty('--card-bg', `url('${voucher.issuer.profileImage}')`);
        card.innerHTML = `
            <div class="voucher-head">
                <span class="status-light ${voucher.health}" aria-label="${voucher.health} health"></span>
                <span class="muted">${voucher.issuer.name}</span>
            </div>
            <p class="voucher-name">${voucher.name}</p>
            <p class="voucher-amount">${formatSats(voucher.amount)}</p>
        `;
        card.addEventListener('click', () => setSelectedVoucher(voucher.id));
        carousel.appendChild(card);
    });
}

function renderCarousel() {
    const carousel = document.getElementById('voucher-carousel');
    carousel.innerHTML = '';
    const visible = filteredVouchers();
    renderVoucherPagination(visible.length);
    const totalPages = Math.max(1, Math.ceil(visible.length / voucherPageSize));
    if (voucherPage >= totalPages) {
        voucherPage = totalPages - 1;
    }
    const start = voucherPage * voucherPageSize;
    const pageItems = visible.slice(start, start + voucherPageSize);

    if (!pageItems.length) {
        carousel.innerHTML = '<p class="muted">No vouchers match your search.</p>';
        return;
    }

    pageItems.forEach(voucher => {
        const card = document.createElement('button');
        card.type = 'button';
        card.className = 'voucher-card';
        card.dataset.voucher = voucher.id;
        if (voucher.id === selectedVoucher.id) {
            card.classList.add('active');
        }
        card.style.setProperty('--card-bg', `url('${voucher.issuer.profileImage}')`);
        card.innerHTML = `
            <div class="voucher-head">
                <span class="status-light ${voucher.health}" aria-label="${voucher.health} health"></span>
                <span class="muted">${voucher.issuer.name}</span>
            </div>
            <p class="voucher-name">${voucher.name}</p>
            <p class="voucher-amount">${voucher.amount.toLocaleString()} sats</p>
        `;
        card.addEventListener('click', () => setSelectedVoucher(voucher.id));
        carousel.appendChild(card);
    });
}

function renderServices() {
    // Services section removed per latest UX; keep stub for future use.
}

function renderActivity() {
    const list = document.getElementById('activity-list');
    list.innerHTML = '';
    activities.forEach(item => {
        const row = document.createElement('div');
        row.className = 'activity-row';
        row.innerHTML = `
            <div>
                <p class="activity-title">${item.type}</p>
                <p class="muted">${item.detail} • ${item.time}</p>
            </div>
            <span class="chip ${item.amount.startsWith('-') ? 'danger' : 'success'}">${item.amount}</span>
        `;
        list.appendChild(row);
    });
}

function renderNotifications() {
    const list = document.getElementById('notification-list');
    if (!list) return;
    list.innerHTML = '';

    notifications.forEach(note => {
        const row = document.createElement('div');
        row.className = 'notification-row';
        row.innerHTML = `
            <div class="notification-meta">
                <span class="status-light ${note.severity}"></span>
                <div>
                    <p class="notification-title">${note.title}</p>
                    <p class="notification-time">${note.time}</p>
                    <p class="muted">${note.detail}</p>
                </div>
            </div>
        `;
        list.appendChild(row);
    });
}

function renderNotificationBadge() {
    const badge = document.getElementById('notification-badge');
    if (!badge) return;
    const count = notifications.length;
    if (count > 0) {
        badge.textContent = count > 9 ? '9+' : String(count);
        badge.style.display = 'inline-flex';
    } else {
        badge.style.display = 'none';
    }
}

function filteredVouchers() {
    if (!voucherSearch.trim()) {
        return vouchers;
    }
    const q = voucherSearch.toLowerCase();
    return vouchers.filter(v =>
        v.name.toLowerCase().includes(q) ||
        v.issuer.name.toLowerCase().includes(q) ||
        v.issuer.username.toLowerCase().includes(q)
    );
}

function renderVoucherPagination(totalItems) {
    const totalPages = Math.max(1, Math.ceil(totalItems / voucherPageSize));
    if (voucherPage >= totalPages) {
        voucherPage = totalPages - 1;
    }
    const label = document.getElementById('voucher-page-label');
    if (label) {
        label.textContent = `Page ${voucherPage + 1} of ${totalPages}`;
    }
}

function renderTransactions() {
    const voucherFilter = document.getElementById('voucher-filter');
    const periodFilter = document.getElementById('period-filter');

    const previousVoucher = voucherFilter.value || 'all';
    voucherFilter.innerHTML = '<option value="all">All vouchers</option>';
    vouchers.forEach(v => {
        const option = document.createElement('option');
        option.value = v.id;
        option.textContent = v.name;
        voucherFilter.appendChild(option);
    });
    voucherFilter.value = previousVoucher;
    if (!voucherFilter.value) {
        voucherFilter.value = 'all';
    }

    const selectedVoucherFilter = voucherFilter.value;
    const selectedPeriod = Number(periodFilter.value);
    const now = new Date();

    const filtered = transactions.filter(tx => {
        const txDate = new Date(tx.date);
        const diffDays = (now - txDate) / (1000 * 60 * 60 * 24);
        const voucherMatch = selectedVoucherFilter === 'all' || tx.voucher === selectedVoucherFilter;
        return voucherMatch && diffDays <= selectedPeriod;
    });

    const list = document.getElementById('transaction-list');
    renderTransactionList(list, filtered, 'No transactions for this filter.');
}

function renderVoucherTransactions() {
    const voucherTransactions = transactions.filter(tx => tx.voucher === selectedVoucher.id);
    const container = document.getElementById('voucher-detail-transactions');
    renderTransactionList(container, voucherTransactions, 'No activity for this voucher yet.');
}

function renderSettings() {
    const relayList = document.getElementById('relay-list');
    const mintList = document.getElementById('mint-list');

    relayList.innerHTML = '';
    relays.forEach(relay => {
        const li = document.createElement('li');
        const indicator = document.createElement('span');
        indicator.className = `status-light ${relay.health}`;
        indicator.setAttribute('aria-label', `${relay.health} health`);
        const label = document.createElement('span');
        label.textContent = 'Relay';
        label.className = 'muted';
        const value = document.createElement('span');
        value.textContent = relay.url;
        const wrapper = document.createElement('div');
        wrapper.className = 'meta-inline';
        wrapper.append(label, indicator);
        li.append(wrapper, value);
        relayList.appendChild(li);
    });

    mintList.innerHTML = '';
    mints.forEach(mint => {
        const li = document.createElement('li');
        const indicator = document.createElement('span');
        indicator.className = `status-light ${mint.health}`;
        indicator.setAttribute('aria-label', `${mint.health} health`);
        const label = document.createElement('span');
        label.textContent = 'Mint';
        label.className = 'muted';
        const value = document.createElement('span');
        value.textContent = mint.name;
        const wrapper = document.createElement('div');
        wrapper.className = 'meta-inline';
        wrapper.append(label, indicator);
        li.append(wrapper, value);
        mintList.appendChild(li);
    });
}

function renderRecipients() {
    const select = document.getElementById('send-recipient');
    if (!select) return;
    select.innerHTML = '';
    recipients.forEach(recipient => {
        const option = document.createElement('option');
        option.value = recipient.id;
        option.textContent = `${recipient.name} (${recipient.handle})`;
        select.appendChild(option);
    });
}

function renderContactsList() {
    const list = document.getElementById('contact-list');
    if (!list) return;
    list.innerHTML = '';

    contacts.forEach(contact => {
        const row = document.createElement('button');
        row.type = 'button';
        row.className = 'contact-row';
        row.innerHTML = `
            <div class="contact-avatar-thumb" style="background-image:url('${contact.avatar}')"></div>
            <div class="contact-meta">
                <p class="contact-name">${contact.name}</p>
                <p class="muted">${contact.handle} • ${contact.location}</p>
            </div>
            <span class="chip">${contact.mint}</span>
        `;
        row.addEventListener('click', () => {
            selectedContact = contact;
            renderContactProfile();
            showPage('contact-profile-page');
        });
        list.appendChild(row);
    });
}

function renderContactProfile() {
    const name = document.getElementById('contact-name');
    const fullname = document.getElementById('contact-fullname');
    const desc = document.getElementById('contact-description');
    const handle = document.getElementById('contact-handle');
    const location = document.getElementById('contact-location');
    const npub = document.getElementById('contact-npub');
    const mint = document.getElementById('contact-mint');
    const avatar = document.getElementById('contact-avatar');
    const sendBtn = document.getElementById('contact-send-btn');
    const activityList = document.getElementById('contact-activity-list');

    if (!selectedContact) return;
    if (name) name.textContent = selectedContact.name;
    if (fullname) fullname.textContent = selectedContact.name;
    if (desc) desc.textContent = selectedContact.description;
    if (handle) handle.textContent = selectedContact.handle;
    if (location) location.textContent = selectedContact.location;
    if (npub) npub.textContent = selectedContact.npub;
    if (mint) mint.textContent = selectedContact.mint;
    if (avatar) avatar.style.backgroundImage = `url('${selectedContact.avatar}')`;
    if (sendBtn) {
        sendBtn.onclick = () => {
            prefillSendForm(selectedContact.id);
            showPage('send-page');
        };
    }
    if (activityList) {
        const related = transactions.filter(tx => tx.to === selectedContact.handle || tx.to === selectedContact.name);
        renderTransactionList(activityList, related, 'No activity with this contact yet.');
    }
}

function renderWalletHero() {
    const total = vouchers.reduce((sum, v) => sum + v.amount, 0);
    const walletBalance = document.getElementById('wallet-balance');
    const walletMint = document.getElementById('wallet-mint-name');
    const healthLight = document.getElementById('wallet-health');
    const healthLabel = document.getElementById('wallet-health-label');

    if (walletBalance) walletBalance.textContent = `${total.toLocaleString()} sats`;
    if (walletMint) walletMint.textContent = selectedVoucher.issuer.mintName;
    setStatusLight(healthLight, selectedVoucher.health);
    if (healthLabel) {
        const label = selectedVoucher.health === 'green'
            ? 'Healthy'
            : selectedVoucher.health === 'orange'
                ? 'Degraded'
                : 'Unavailable';
        healthLabel.textContent = label;
    }
}

function wireNavigation() {
    const navButtons = document.querySelectorAll('[data-nav]');

    navButtons.forEach(button => {
        button.addEventListener('click', () => {
            const target = button.getAttribute('data-nav');
            if (target) {
                showPage(target);
            }
        });
    });
}

function wireActions() {
    document.querySelectorAll('[data-action]').forEach(actionBtn => {
        actionBtn.addEventListener('click', () => {
            const action = actionBtn.getAttribute('data-action');
            if (action === 'buy') {
                prefillBuyForm();
                showPage('buy-page');
                return;
            }
            if (action === 'send') {
                prefillSendForm();
                showPage('send-page');
                return;
            }
            if (action === 'copy-link') {
                const voucherLink = `${selectedVoucher.mint}/voucher/${selectedVoucher.id}`;
                if (navigator.clipboard?.writeText) {
                    navigator.clipboard.writeText(voucherLink).catch(() => {});
                }
                alert(`Copied voucher link for ${selectedVoucher.name}`);
                return;
            }

            const messages = {
                scan: `Opening scanner to import ${selectedVoucher.name}`
            };
            alert(messages[action] || `Triggered ${action} for ${selectedVoucher.name}`);
        });
    });
}

function prefillBuyForm() {
    const voucherField = document.getElementById('buy-voucher');
    const amountField = document.getElementById('buy-amount');
    if (voucherField) voucherField.value = selectedVoucher.name;
    if (amountField) amountField.value = selectedVoucher.amount;
}

function prefillSendForm(recipientId) {
    const voucherField = document.getElementById('send-voucher');
    const amountField = document.getElementById('send-amount');
    const recipientSelect = document.getElementById('send-recipient');
    if (voucherField) voucherField.value = selectedVoucher.name;
    if (amountField) amountField.value = Math.min(Math.max(1, selectedVoucher.amount), selectedVoucher.amount);
    if (recipientSelect && recipientId) {
        recipientSelect.value = recipientId;
    }
}

function wireBuyFlow() {
    const reviewBtn = document.getElementById('buy-review');
    if (!reviewBtn) return;
    reviewBtn.addEventListener('click', () => {
        const amount = document.getElementById('buy-amount')?.value || '';
        const method = document.getElementById('buy-method')?.value || 'bitcoin';
        document.getElementById('buy-summary-voucher').textContent = selectedVoucher.name;
        document.getElementById('buy-summary-amount').textContent = `${amount} sats`;
        document.getElementById('buy-summary-method').textContent = method === 'bitcoin' ? 'Bitcoin (Lightning)' : method;
        showPage('buy-summary-page');
    });
}

function wireSendFlow() {
    const reviewBtn = document.getElementById('send-review');
    if (!reviewBtn) return;
    reviewBtn.addEventListener('click', () => {
        const amount = document.getElementById('send-amount')?.value || '';
        const recipientId = document.getElementById('send-recipient')?.value;
        const recipient = recipients.find(r => r.id === recipientId) || recipients[0];
        document.getElementById('send-summary-voucher').textContent = selectedVoucher.name;
        document.getElementById('send-summary-amount').textContent = `${amount} sats`;
        document.getElementById('send-summary-recipient').textContent = `${recipient.name} (${recipient.handle})`;
        showPage('send-summary-page');
    });
}

function applyTheme(preference) {
    const resolved = preference === 'system'
        ? (window.matchMedia && window.matchMedia('(prefers-color-scheme: light)').matches ? 'light' : 'dark')
        : preference;
    document.documentElement.setAttribute('data-theme', resolved);
    const select = document.getElementById('theme-select');
    if (select) select.value = preference;
    localStorage.setItem(themePreferenceKey, preference);
    const toggleLabel = document.getElementById('theme-toggle-label');
    if (toggleLabel) {
        const icon = preference === 'system' ? 'A' : preference === 'light' ? '☀' : '☾';
        toggleLabel.textContent = `${icon} ${preference.charAt(0).toUpperCase() + preference.slice(1)}`;
    }
}

function wireTheme() {
    const select = document.getElementById('theme-select');
    const saved = localStorage.getItem(themePreferenceKey) || 'system';
    applyTheme(saved);

    if (select) {
        select.addEventListener('change', e => applyTheme(e.target.value));
    }

    const toggle = document.getElementById('theme-toggle');
    if (toggle) {
        toggle.addEventListener('click', () => {
            const current = localStorage.getItem(themePreferenceKey) || 'system';
            const idx = themeCycle.indexOf(current);
            const next = themeCycle[(idx + 1) % themeCycle.length];
            applyTheme(next);
        });
    }

    if (window.matchMedia) {
        const mediaQuery = window.matchMedia('(prefers-color-scheme: light)');
        mediaQuery.addEventListener('change', () => {
            const current = localStorage.getItem(themePreferenceKey) || 'system';
            if (current === 'system') {
                applyTheme('system');
            }
        });
    }
}

function init() {
    renderCarousel();
    renderWalletCarousel();
    renderServices();
    renderActivity();
    renderTransactions();
    renderNotifications();
    renderNotificationBadge();
    renderSettings();
    renderRecipients();
    renderWalletHero();
    renderContactsList();
    renderContactProfile();
    wireNavigation();
    wireActions();
    wireTheme();
    wireBuyFlow();
    wireSendFlow();
    const search = document.getElementById('voucher-search');
    if (search) {
        search.addEventListener('input', e => {
            voucherSearch = e.target.value;
            voucherPage = 0;
            renderCarousel();
            setSelectedVoucher(selectedVoucher.id);
        });
    }
    const walletSearch = document.getElementById('wallet-voucher-search');
    if (walletSearch) {
        walletSearch.addEventListener('input', e => {
            walletVoucherSearch = e.target.value;
            walletVoucherPage = 0;
            renderWalletCarousel();
            setSelectedVoucher(selectedVoucher.id);
        });
    }
    const prev = document.getElementById('voucher-prev');
    const next = document.getElementById('voucher-next');
    if (prev) {
        prev.addEventListener('click', () => {
            if (voucherPage > 0) {
                voucherPage -= 1;
                renderCarousel();
                setSelectedVoucher(selectedVoucher.id);
            }
        });
    }
    if (next) {
        next.addEventListener('click', () => {
            const totalPages = Math.max(1, Math.ceil(filteredVouchers().length / voucherPageSize));
            if (voucherPage < totalPages - 1) {
                voucherPage += 1;
                renderCarousel();
                setSelectedVoucher(selectedVoucher.id);
            }
        });
    }
    const walletPrev = document.getElementById('wallet-voucher-prev');
    const walletNext = document.getElementById('wallet-voucher-next');
    if (walletPrev) {
        walletPrev.addEventListener('click', () => {
            if (walletVoucherPage > 0) {
                walletVoucherPage -= 1;
                renderWalletCarousel();
                setSelectedVoucher(selectedVoucher.id);
            }
        });
    }
    if (walletNext) {
        walletNext.addEventListener('click', () => {
            const totalPages = Math.max(1, Math.ceil(walletFilteredVouchers().length / voucherPageSize));
            if (walletVoucherPage < totalPages - 1) {
                walletVoucherPage += 1;
                renderWalletCarousel();
                setSelectedVoucher(selectedVoucher.id);
            }
        });
    }
    const openVoucherPageBtn = document.getElementById('open-voucher-page');
    if (openVoucherPageBtn) {
        openVoucherPageBtn.addEventListener('click', () => showPage('voucher-page'));
    }

    document.getElementById('voucher-filter').addEventListener('change', renderTransactions);
    document.getElementById('period-filter').addEventListener('change', renderTransactions);
    setSelectedVoucher(selectedVoucher.id);
}

document.addEventListener('DOMContentLoaded', init);
